# Admin Certificate Upload & Student Read-Only Profile Feature

## Overview
Added functionality for administrators to upload earned certificates to student profiles. Students can view their certificates in read-only mode but cannot edit or delete them. Admin notes remain hidden from students, maintaining privacy of administrative information.

## Implementation Date
October 13, 2025

---

## Features Implemented

### 1. Earned Certificates Section
**File:** `views/student_profile.ejs`

Added a new "Earned Certificates" section that displays differently based on user role (admin vs student).

**Section Structure:**
```html
<!-- Earned Certificates Section -->
<section class="cardish">
  <div class="section-title">🏆 Earned Certificates</div>
  <% const earnedCertificates = (student.profile && student.profile.certificates) || []; %>
  
  <% if (role === 'admin') { %>
    <!-- Admin Upload Form -->
  <% } %>

  <!-- Display Certificates (visible to both admin and student) -->
</section>
```

### 2. Admin Upload Form
**File:** `views/student_profile.ejs`

Administrators can upload multiple certificates at once.

**Upload Form:**
```html
<form class="row g-3 mb-4" method="post" action="/admin/students/<%= student.id %>/certificates" enctype="multipart/form-data">
  <div class="col-md-8">
    <label class="form-label fw-bold">Upload Certificate(s)</label>
    <input type="file" name="certificates" class="form-control" multiple accept=".pdf,.jpg,.jpeg,.png">
    <small class="text-muted">Upload certificates earned by this student (PDF, JPG, PNG - Max 10MB each)</small>
  </div>
  <div class="col-md-4 d-flex align-items-end">
    <button type="submit" class="btn btn-primary">
      <i class="bi bi-upload"></i> Upload Certificates
    </button>
  </div>
</form>
```

**Features:**
- Multiple file upload support
- Accepts PDF, JPG, PNG formats
- 10MB per file limit
- Only visible to admins
- Upload icon for visual clarity

### 3. Certificate Display Table
**File:** `views/student_profile.ejs`

Certificates are displayed in a table with different actions based on role.

**Display Table:**
```html
<div class="table-responsive">
  <table class="table table-bordered align-middle mb-0">
    <thead>
      <tr>
        <th>Certificate Name</th>
        <th>Size</th>
        <th class="text-nowrap">Uploaded Date</th>
        <th style="width:<%= role === 'admin' ? '180px' : '120px' %>;">Action</th>
      </tr>
    </thead>
    <tbody>
      <% earnedCertificates.forEach((cert, index) => { %>
        <tr>
          <td>
            <i class="bi bi-award text-warning"></i>
            <%= cert.originalName %>
          </td>
          <td><%= Math.round((cert.size||0)/1024) %> KB</td>
          <td class="text-nowrap"><%= cert.uploadedAt ? new Date(cert.uploadedAt).toLocaleDateString() : '-' %></td>
          <td>
            <a class="btn btn-sm btn-outline-primary" href="<%= cert.url %>" target="_blank" rel="noopener" download>
              <i class="bi bi-download"></i> Download
            </a>
            <% if (role === 'admin') { %>
              <button class="btn btn-sm btn-outline-danger" onclick="deleteEarnedCertificate(<%= student.id %>, <%= index %>, '<%= cert.originalName %>')">
                <i class="bi bi-trash"></i>
              </button>
            <% } %>
          </td>
        </tr>
      <% }) %>
    </tbody>
  </table>
</div>
```

**Display Features:**
- Award icon next to certificate name
- File size in KB
- Upload date
- Download button (visible to both admin and student)
- Delete button (admin only)
- Responsive table layout

### 4. Empty State Message
**File:** `views/student_profile.ejs`

Different messages for admin and student when no certificates exist.

**Empty State:**
```html
<% if (!earnedCertificates.length) { %>
  <div class="alert alert-info mb-0">
    <i class="bi bi-info-circle"></i> No certificates have been uploaded yet.
    <% if (role === 'student') { %>
      Your certificates will appear here once they are uploaded by an administrator.
    <% } %>
  </div>
<% } %>
```

### 5. Delete Functionality (Admin Only)
**File:** `views/student_profile.ejs`

JavaScript function for deleting certificates with SweetAlert2 confirmation.

**Delete Function:**
```javascript
async function deleteEarnedCertificate(studentId, certIndex, certName) {
  const confirmed = await Swal.fire({
    title: 'Delete Certificate?',
    html: `Are you sure you want to delete <strong>${certName}</strong>?<br><span class="text-danger">This action cannot be undone.</span>`,
    icon: 'warning',
    showCancelButton: true,
    confirmButtonColor: '#dc3545',
    cancelButtonColor: '#6c757d',
    confirmButtonText: 'Yes, Delete',
    cancelButtonText: 'Cancel'
  });

  if (confirmed.isConfirmed) {
    try {
      const response = await fetch(`/admin/students/${studentId}/delete-certificate/${certIndex}`, {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        await Swal.fire({
          title: 'Deleted!',
          text: 'Certificate has been deleted successfully.',
          icon: 'success',
          timer: 2000,
          showConfirmButton: false
        });
        window.location.reload();
      } else {
        const error = await response.text();
        throw new Error(error || 'Failed to delete certificate');
      }
    } catch (error) {
      console.error('Delete error:', error);
      Swal.fire({
        title: 'Error',
        text: error.message || 'Failed to delete certificate. Please try again.',
        icon: 'error'
      });
    }
  }
}
```

### 6. Multer Configuration
**File:** `routes/admin.js`

Dedicated multer configuration for certificate uploads.

**Multer Config:**
```javascript
// Uploader for earned certificates - admin use
const certificatesUpload = multer({
  storage: docsStorage,
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type. Only PDF, JPG, and PNG files are allowed.'));
  }
}).array('certificates', 10);
```

**Configuration:**
- **Storage:** Uses `docsStorage` (saves to `/docs/stxd/`)
- **File Size Limit:** 10MB per file
- **Allowed Types:** PDF, JPG, JPEG, PNG
- **Max Count:** 10 files per upload
- **Field Name:** `certificates`

### 7. Backend Upload Route
**File:** `routes/admin.js`

**Upload Route:**
```javascript
// Upload earned certificates - Admin only
router.post('/students/:id/certificates', (req, res) => {
  certificatesUpload(req, res, async (err) => {
    try {
      const id = Number(req.params.id);
      const student = await userModel.findById(id);
      if (!student || student.role !== 'student') return res.status(404).send('Student not found');
      if (err) {
        console.error('Certificate upload error', err);
        return res.status(400).send(err.message || 'Upload failed');
      }

      const profile = student.profile || {};
      const certificates = Array.isArray(profile.certificates) ? profile.certificates : [];
      
      // Process uploaded certificates
      if (req.files && req.files.length > 0) {
        const toMeta = (f) => ({
          originalName: f.originalname,
          mimeType: f.mimetype,
          size: f.size,
          url: `/docs/stxd/${f.filename}`,
          uploadedAt: new Date().toISOString()
        });
        
        const newCertificates = req.files.map(toMeta);
        certificates.push(...newCertificates);
        
        await userModel.updateProfile(id, { certificates });
        console.log(`Uploaded ${newCertificates.length} certificate(s) for student ${id}`);
      }

      return res.redirect(`/admin/students/${id}`);
    } catch (e) {
      console.error('Certificate upload error', e);
      return res.status(500).send('Failed to upload certificates');
    }
  });
});
```

**Process:**
1. Validates student exists and has 'student' role
2. Handles multer upload errors
3. Retrieves existing certificates array
4. Maps uploaded files to metadata objects
5. Adds upload timestamp to each certificate
6. Appends new certificates to existing array
7. Updates profile in database
8. Logs upload activity
9. Redirects back to student profile

### 8. Backend Delete Route
**File:** `routes/admin.js`

**Delete Route:**
```javascript
// Delete earned certificate - Admin only
router.delete('/students/:id/delete-certificate/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const certIndex = Number(req.params.index);
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const profile = student.profile || {};
    const certificates = Array.isArray(profile.certificates) ? profile.certificates : [];
    
    if (certIndex < 0 || certIndex >= certificates.length) {
      return res.status(400).send('Invalid certificate index');
    }
    
    // Remove the certificate from the array
    const deletedCert = certificates.splice(certIndex, 1)[0];
    
    // Update the profile with the new certificates array
    await userModel.updateProfile(studentId, { certificates });
    
    // Delete the physical file from the filesystem
    if (deletedCert && deletedCert.url) {
      const fs = require('fs');
      const path = require('path');
      const filePath = path.join(__dirname, '..', 'public', deletedCert.url);
      
      fs.unlink(filePath, (err) => {
        if (err) {
          console.warn('Failed to delete physical file:', filePath, err);
        } else {
          console.log('Deleted physical file:', filePath);
        }
      });
    }
    
    return res.status(200).json({ success: true, message: 'Certificate deleted successfully' });
  } catch (error) {
    console.error('Error deleting certificate:', error);
    return res.status(500).send('Failed to delete certificate');
  }
});
```

**Process:**
1. Validates student exists and has 'student' role
2. Validates certificate index is within bounds
3. Removes certificate from array using splice
4. Updates profile in database
5. Deletes physical file from filesystem
6. Returns JSON success response
7. Handles errors gracefully

---

## Student Read-Only Profile Access

### Current Implementation

Students can already view their own profile at `/student/profile` using the same `student_profile.ejs` template with `role: 'student'`.

**Student Profile Route:**
```javascript
// routes/student.js
router.get('/profile', async (req, res) => {
  const student = await userModel.findById(req.session.user.id);
  if (!student) return res.status(404).send('Not found');
  const updated = String(req.query.updated || '') === '1';
  
  // ... additional data fetching ...
  
  res.render('student_profile', { 
    student, 
    role: 'student',  // Key: role is 'student' not 'admin'
    signatureDocsConfig, 
    updated, 
    lead, 
    leadContacts,
    studentClasses
  });
});
```

### Role-Based Access Control

The `student_profile.ejs` template uses conditional rendering based on `role`:

**Admin-Only Sections:**
```html
<% if (role === 'admin') { %>
  <!-- Admin Notes -->
  <!-- Edit Profile Form -->
  <!-- Upload Forms -->
  <!-- Delete Buttons -->
<% } %>
```

**Student-Visible Sections:**
```html
<% if (role === 'student') { %>
  <!-- Required Documents Upload (student self-upload) -->
  <!-- Document signing forms -->
<% } %>
```

**Shared Sections (Read-Only for Students):**
```html
<!-- Personal Information Display -->
<!-- Enrolled Classes -->
<!-- Earned Certificates -->
<!-- Required Documents Status -->
```

### Hidden from Students

1. **Admin Notes Section**
   - Completely hidden with `<% if (role === 'admin') { %>`
   - Private notes not accessible to students

2. **Edit Profile Form**
   - Students cannot edit profile information
   - Only admins see edit forms

3. **Delete Buttons**
   - Students cannot delete certificates
   - Delete buttons only render for admins

4. **Upload Forms (for certificates)**
   - Students cannot upload their own certificates
   - Only admins can upload certificates

5. **Admin Actions**
   - Enrollment management
   - Document management
   - Status changes

### Visible to Students (Read-Only)

1. **Personal Information**
   - Name, email, phone
   - Address
   - Enrollment status

2. **Enrolled Classes**
   - List of classes they're enrolled in
   - Class details (read-only)

3. **Earned Certificates**
   - View and download their certificates
   - Cannot delete or modify

4. **Required Documents**
   - Upload their own ID and transcript
   - View upload status

5. **Signature Documents**
   - Download documents to sign
   - Upload signed copies

---

## User Workflows

### Administrator Uploads Certificates

1. Navigate to student profile (`/admin/students/:id`)
2. Scroll to "Earned Certificates" section
3. Click "Upload Certificate(s)" file input
4. Select one or more certificate files (PDF, JPG, PNG)
5. Click "Upload Certificates" button
6. Page redirects back to student profile
7. Certificates appear in table with download and delete buttons
8. Upload date is recorded automatically

### Administrator Deletes Certificate

1. Navigate to student profile
2. Scroll to "Earned Certificates" section
3. Find certificate to delete in table
4. Click trash icon button
5. Confirmation dialog appears with certificate name
6. Click "Yes, Delete" to confirm
7. Success message appears (2 seconds)
8. Page reloads automatically
9. Certificate removed from list
10. Physical file deleted from server

### Student Views Certificates

1. Log in as student
2. Navigate to "My Profile" (`/student/profile`)
3. Scroll to "Earned Certificates" section
4. View list of certificates uploaded by admin
5. Click "Download" button to download certificate
6. Cannot see upload form (admin only)
7. Cannot delete certificates (no delete buttons)
8. Can only view and download

### Student Views Profile (Read-Only)

1. Log in as student
2. Navigate to "My Profile" (`/student/profile`)
3. See personal information (cannot edit)
4. See enrolled classes (cannot modify)
5. See earned certificates (can download only)
6. See required documents status
7. Can upload ID/transcript if not yet uploaded
8. Cannot see Admin Notes section
9. Cannot see any admin-only forms or buttons

---

## Database Structure

### Storage Location

Earned certificates are stored in the `mdtslms_users` table in the `profile` JSON column:

```json
{
  "profile": {
    "certificates": [
      {
        "originalName": "CompTIA_A_Plus_Certificate.pdf",
        "mimeType": "application/pdf",
        "size": 234567,
        "url": "/docs/stxd/1697234567892-CompTIA_A_Plus_Certificate.pdf",
        "uploadedAt": "2025-10-13T15:30:00.000Z"
      },
      {
        "originalName": "AWS_Cloud_Practitioner.pdf",
        "mimeType": "application/pdf",
        "size": 345678,
        "url": "/docs/stxd/1697234567893-AWS_Cloud_Practitioner.pdf",
        "uploadedAt": "2025-10-13T16:45:00.000Z"
      }
    ],
    "adminNotes": [
      {
        "content": "Student completed CompTIA A+ on 10/10/2025",
        "createdAt": "2025-10-13T15:30:00.000Z"
      }
    ]
  }
}
```

### Field Structure

**Certificate Object:**
- `originalName` - Original filename of uploaded certificate
- `mimeType` - MIME type (application/pdf, image/jpeg, etc.)
- `size` - File size in bytes
- `url` - Public URL path to certificate file
- `uploadedAt` - ISO timestamp of upload

### Distinction from Certifications

**Certificates (`profile.certificates`):**
- Admin-uploaded earned certificates
- Represents completed certifications/achievements
- Students can view/download only
- Admin can upload/delete

**Certifications (`profile.certifications`):**
- Documents uploaded in "Agreements & Signatures" section
- Part of enrollment documentation
- Different purpose and workflow

---

## Security & Validation

### Access Control
- ✅ Only admin users can upload certificates
- ✅ Only admin users can delete certificates
- ✅ Students can only view and download certificates
- ✅ Upload form hidden from students with role check
- ✅ Delete buttons hidden from students with role check
- ✅ Backend routes validate admin authentication
- ✅ Admin notes completely hidden from students

### Input Validation
- ✅ Student ID must be valid number
- ✅ Student must exist in database
- ✅ Student must have 'student' role
- ✅ Certificate index validated for deletions
- ✅ Index must be within array bounds

### File Type Restrictions
- ✅ Only PDF, JPG, JPEG, PNG allowed
- ✅ Validated in multer fileFilter
- ✅ File extension checked (case-insensitive)
- ✅ Clear error message for invalid types

### File Size Limits
- ✅ Maximum 10MB per file
- ✅ Configured in multer limits
- ✅ Prevents oversized uploads

### Error Handling
- ✅ 400 error for invalid file types
- ✅ 400 error for invalid certificate index
- ✅ 404 error if student not found
- ✅ 500 error for database/filesystem failures
- ✅ User-friendly error messages via SweetAlert
- ✅ Console logging for debugging
- ✅ Graceful handling of missing physical files

### User Confirmations
- ✅ Confirmation dialog before deletion
- ✅ Shows certificate name in confirmation
- ✅ Clear warning about irreversibility
- ✅ Cancel option available
- ✅ Success confirmation after deletion

---

## Technical Details

### Route Paths

**Upload Route:**
```
POST /admin/students/:id/certificates
```

**Delete Route:**
```
DELETE /admin/students/:id/delete-certificate/:index
```

**Student Profile Route:**
```
GET /student/profile
```

**Admin Student Profile Route:**
```
GET /admin/students/:id
```

### HTTP Methods
- **POST** for upload (multipart/form-data)
- **DELETE** for deletion (AJAX with JSON response)
- **GET** for viewing profile

### Request Parameters

**Upload:**
- `:id` - Student ID (URL parameter)
- `certificates` - File upload field (array, max 10 files)

**Delete:**
- `:id` - Student ID (URL parameter)
- `:index` - Certificate index in array (URL parameter)

### Response Format

**Upload Success:**
- HTTP 302 redirect to `/admin/students/:id`

**Delete Success:**
```json
{
  "success": true,
  "message": "Certificate deleted successfully"
}
```

**Error:**
- Plain text error message with appropriate status code

### File Processing

**Upload:**
1. Multer processes multipart/form-data
2. Files saved to `/docs/stxd/` with timestamped names
3. File metadata constructed with upload timestamp
4. Metadata appended to certificates array
5. Profile updated in database

**Delete:**
1. Certificate index validated
2. Certificate removed from array using splice
3. Database updated with new array
4. Physical file path constructed
5. File deleted with `fs.unlink()`

---

## Benefits

### For Administrators
✅ **Easy Upload:** Upload multiple certificates at once
✅ **Organization:** Centralized certificate management
✅ **Control:** Full control over student certificates
✅ **Tracking:** Upload dates automatically recorded
✅ **Flexibility:** Can delete and re-upload if needed
✅ **Privacy:** Admin notes hidden from students

### For Students
✅ **Visibility:** Can see all earned certificates
✅ **Access:** Download certificates anytime
✅ **Professional:** Official record of achievements
✅ **Convenience:** Certificates available in profile
✅ **Privacy:** Cannot see admin notes
✅ **Clean Interface:** Read-only view without clutter

### For System
✅ **Consistency:** Same storage pattern as other documents
✅ **Simplicity:** Reuses existing infrastructure
✅ **Separation:** Clear distinction between certificates and certifications
✅ **Scalability:** Supports multiple certificates per student
✅ **Data Integrity:** Database and filesystem stay synchronized

---

## UI Components

### Upload Form (Admin Only)
- **Layout:** Two-column row (Bootstrap grid)
- **Style:** `form-control` for file input, `btn btn-primary` for button
- **Icon:** Upload icon (`bi-upload`) on button
- **Helper Text:** Format and size information
- **Multiple Files:** Supports multiple file selection

### Certificate Table
- **Layout:** Responsive table with Bootstrap styling
- **Icon:** Award icon (`bi-award text-warning`) next to names
- **Columns:** Name, Size, Upload Date, Actions
- **Action Width:** Dynamic based on role (180px admin, 120px student)
- **Buttons:** Download (both), Delete (admin only)

### Delete Button (Admin Only)
- **Style:** `btn btn-sm btn-outline-danger`
- **Icon:** Trash icon (`bi-trash`)
- **Action:** onclick calls `deleteEarnedCertificate()`

### Empty State
- **Style:** `alert alert-info`
- **Icon:** Info circle icon
- **Message:** Role-specific messaging

### Confirmation Dialogs (SweetAlert2)
- **Warning Dialog:** Shows certificate name, irreversibility warning
- **Success Dialog:** Auto-closes after 2 seconds
- **Error Dialog:** Shows error message with OK button

---

## Role Comparison

### Admin View of Student Profile

**Can See:**
- All personal information
- Admin notes section (with edit/delete)
- Edit profile form
- Upload certificate form
- Delete certificate buttons
- Enrollment management
- All document sections with admin controls

**Can Do:**
- Upload certificates
- Delete certificates
- Edit profile
- Add/edit/delete admin notes
- Manage enrollments
- Upload required documents
- Delete documents

### Student View of Own Profile

**Can See:**
- Personal information (read-only)
- Enrolled classes (read-only)
- Earned certificates (read-only)
- Required documents section (with upload form)
- Signature documents section (with upload forms)

**Cannot See:**
- Admin notes section
- Edit profile form
- Certificate upload form
- Delete buttons for certificates
- Admin-only controls

**Can Do:**
- View all personal information
- Download earned certificates
- Upload ID and transcript (if not uploaded)
- Upload signed documents
- View enrolled classes

**Cannot Do:**
- Edit profile information
- Upload certificates
- Delete certificates
- See admin notes
- Manage enrollments
- Delete any documents

---

## Testing Checklist

- [x] Admin upload form appears in Earned Certificates section
- [x] Upload form accepts multiple files
- [x] Upload form accepts PDF, JPG, PNG files
- [x] Upload button submits form correctly
- [x] Certificates upload successfully
- [x] Upload date recorded automatically
- [x] Page redirects after successful upload
- [x] Certificates appear in table with correct data
- [x] Download button works for certificates
- [x] Delete button appears for admin only
- [x] Delete confirmation dialog appears
- [x] Cancel button aborts deletion
- [x] Delete button removes certificate from UI
- [x] Delete button removes physical file
- [x] Delete button updates database
- [x] Page reloads after successful deletion
- [x] Student can view their profile at /student/profile
- [x] Student sees earned certificates section
- [x] Student can download certificates
- [x] Student cannot see upload form
- [x] Student cannot see delete buttons
- [x] Student cannot see admin notes section
- [x] Admin notes hidden from students
- [x] File type validation works
- [x] File size limit enforced (10MB)
- [x] Error handling for upload failures
- [x] Error handling for delete failures
- [x] Console logs upload/delete activity
- [x] Multiple sequential uploads work
- [x] Multiple sequential deletes work
- [x] Empty state message displays correctly
- [x] Empty state message varies by role

---

## Future Enhancements (Optional)

### Certificate Metadata
Add additional fields to track more information:
```javascript
{
  "originalName": "CompTIA_A_Plus_Certificate.pdf",
  "url": "/docs/stxd/cert.pdf",
  "uploadedAt": "2025-10-13T15:30:00.000Z",
  "certificateName": "CompTIA A+ Certification",
  "issuingOrganization": "CompTIA",
  "issueDate": "2025-10-10",
  "expirationDate": "2028-10-10",
  "certificateNumber": "ABC123456",
  "uploadedBy": 5
}
```

### Certificate Verification
Add verification status and external verification links:
```javascript
{
  "originalName": "cert.pdf",
  "url": "/docs/stxd/cert.pdf",
  "verified": true,
  "verificationUrl": "https://verify.comptia.org/ABC123456",
  "verifiedAt": "2025-10-13T16:00:00.000Z"
}
```

### Certificate Categories
Organize certificates by type:
```javascript
{
  "originalName": "cert.pdf",
  "url": "/docs/stxd/cert.pdf",
  "category": "IT Certification", // or "Completion", "Achievement", etc.
  "tags": ["CompTIA", "Networking", "Professional"]
}
```

### Certificate Sharing
Allow students to share certificates externally:
```javascript
{
  "originalName": "cert.pdf",
  "url": "/docs/stxd/cert.pdf",
  "publicShareUrl": "https://lms.school.edu/certificates/abc123",
  "shareEnabled": true
}
```

### Email Notifications
Notify student when certificate is uploaded:
```javascript
await transporter.sendMail({
  to: student.email,
  subject: 'New Certificate Added to Your Profile',
  html: `<p>A new certificate has been added to your profile: <strong>${cert.originalName}</strong></p>`
});
```

### Certificate Dashboard
Create a dedicated page for viewing all certificates:
```
GET /student/certificates
```
Display certificates in a grid/card layout with search and filter options.

---

## Related Files

- **View:** `views/student_profile.ejs` - Certificate section, upload form, display table
- **Routes:** `routes/admin.js` - Upload and delete routes, multer configuration
- **Routes:** `routes/student.js` - Student profile view route
- **Model:** `models/userModel.js` - updateProfile function

---

## Summary

The Admin Certificate Upload & Student Read-Only Profile feature provides a complete solution for managing student certificates. Administrators can upload and manage certificates while students can view and download them in read-only mode. The implementation maintains clear separation between admin and student capabilities, with admin notes and controls completely hidden from students. This creates a professional, organized system for tracking student achievements while giving students easy access to their earned certificates.
