# Document Editing and Template Features

## Overview
This update adds the ability to edit unsigned documents and save documents as reusable templates.

## Features Added

### 1. Edit Unsigned Documents
- **Location**: Document view page (`/document-signing/view/:documentId`)
- **Access**: Admin only
- **Condition**: Only available when NO parties have signed yet
- **Button**: "Edit Document" - redirects to the edit page

### 2. Save as Template
- **Location**: Document view page (`/document-signing/view/:documentId`)
- **Access**: Admin only  
- **Condition**: Only available when NO parties have signed yet
- **Function**: Saves the document structure (PDF + fields) as a reusable template

### 3. Template Management
- **Templates Page**: `/document-signing/templates`
- **Features**:
  - View all saved templates
  - See template details (name, PDF, field count, creator, date)
  - Delete templates
  - Use template to create new document

## Database Changes

### New Tables

#### signature_document_templates
Stores template metadata and PDF information.

```sql
CREATE TABLE signature_document_templates (
  id INT AUTO_INCREMENT PRIMARY KEY,
  template_name VARCHAR(255) NOT NULL,
  original_filename VARCHAR(255) NOT NULL,
  stored_filename VARCHAR(255) NOT NULL,
  file_path VARCHAR(500) NOT NULL,
  description TEXT,
  created_by INT NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  INDEX idx_created_by (created_by)
);
```

#### signature_template_fields
Stores field definitions for templates.

```sql
CREATE TABLE signature_template_fields (
  id INT AUTO_INCREMENT PRIMARY KEY,
  template_id INT NOT NULL,
  field_type ENUM('signature', 'initials', 'text', 'date', 'checkbox') NOT NULL,
  signer_type ENUM('student', 'staff') NOT NULL,
  field_label VARCHAR(255) NOT NULL,
  page_number INT NOT NULL,
  x_position DECIMAL(10,2) NOT NULL,
  y_position DECIMAL(10,2) NOT NULL,
  width DECIMAL(10,2) NOT NULL,
  height DECIMAL(10,2) NOT NULL,
  is_required BOOLEAN DEFAULT TRUE,
  placeholder_text VARCHAR(255),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (template_id) REFERENCES signature_document_templates(id) ON DELETE CASCADE,
  INDEX idx_template (template_id),
  INDEX idx_signer_type (signer_type)
);
```

## API Endpoints

### POST /document-signing/save-as-template/:documentId
Saves a document as a template.

**Request Body:**
```json
{
  "templateName": "Enrollment Agreement Template"
}
```

**Response:**
```json
{
  "success": true,
  "templateId": 1
}
```

**Requirements:**
- Admin access only
- Document must have NO signatures
- Template name is required

### GET /document-signing/templates
Display all available templates.

**Returns:** Rendered template list page

### GET /document-signing/template/:templateId
Get template details (for API use).

**Response:**
```json
{
  "id": 1,
  "template_name": "Enrollment Agreement Template",
  "original_filename": "enrollment.pdf",
  "stored_filename": "doc-123456.pdf",
  "file_path": "/path/to/file.pdf",
  "description": "Standard enrollment agreement",
  "created_by": 1,
  "created_at": "2025-10-12T...",
  "fields": [
    {
      "id": 1,
      "field_type": "signature",
      "signer_type": "student",
      "field_label": "Student Signature",
      "page_number": 1,
      "x_position": 10.5,
      "y_position": 80.2,
      "width": 30.0,
      "height": 10.0,
      "is_required": true
    }
  ]
}
```

### DELETE /document-signing/template/:templateId
Delete a template.

**Response:**
```json
{
  "success": true,
  "message": "Template deleted successfully"
}
```

## Model Methods

### saveAsTemplate(documentId, templateName, createdBy)
Saves a document as a template.

```javascript
const result = await signatureDocumentModel.saveAsTemplate(
  documentId,
  'My Template',
  userId
);
```

**Returns:** `{ templateId: number, success: boolean }`

### getAllTemplates()
Gets all templates with metadata.

```javascript
const templates = await signatureDocumentModel.getAllTemplates();
```

**Returns:** Array of template objects with creator info and field count

### getTemplateById(templateId)
Gets a template with all its fields.

```javascript
const template = await signatureDocumentModel.getTemplateById(templateId);
```

**Returns:** Template object with `fields` array

### deleteTemplate(templateId)
Deletes a template and its fields (cascades).

```javascript
await signatureDocumentModel.deleteTemplate(templateId);
```

## UI Components

### View Page Updates

The document view page now shows action buttons when the document hasn't been signed yet:

```html
<% if (!hasSignatures && role === 'admin') { %>
  <a href="/document-signing/edit/<%= document.id %>" class="btn btn-warning">
    <i class="bi bi-pencil-square"></i> Edit Document
  </a>
  <button type="button" class="btn btn-info" onclick="saveAsTemplate()">
    <i class="bi bi-bookmark-plus"></i> Save as Template
  </button>
<% } %>
```

### Save as Template JavaScript

```javascript
async function saveAsTemplate() {
  const templateName = prompt('Enter a name for this template:');
  if (!templateName || templateName.trim() === '') {
    return;
  }

  const response = await fetch('/document-signing/save-as-template/<%= document.id %>', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ templateName: templateName.trim() })
  });

  if (response.ok) {
    alert('Template saved successfully!');
  }
}
```

## Workflow

### Saving a Template

1. Admin creates a document with fields
2. BEFORE sending for signatures, admin views the document
3. Admin clicks "Save as Template" button
4. Enters a template name in the prompt
5. System copies:
   - PDF file reference (same file used)
   - All field definitions (position, type, labels)
   - Document metadata
6. Template is saved and appears in templates list

### Using a Template

1. Admin goes to `/document-signing/templates`
2. Finds desired template
3. Clicks "Use This Template"
4. Redirected to create document page with `?templateId=X`
5. System pre-loads:
   - PDF file
   - All field definitions
6. Admin:
   - Selects student
   - Adds staff signers
   - Sends for signatures

## Security

- Only admins can save/view/delete templates
- Templates cannot be created from signed documents
- Template files are shared references (no duplication)
- All template operations are logged in audit log

## Future Enhancements

Potential improvements:
- Template categories/tags
- Template search functionality
- Template preview before use
- Duplicate template with new PDF
- Export/import templates
- Template usage statistics
- Default templates for specific document types

## Testing

### Test Save as Template
1. Create a document with fields
2. View the document
3. Click "Save as Template"
4. Enter name "Test Template"
5. Verify template appears in templates list

### Test Template Cannot Be Saved After Signing
1. Have a student sign a document
2. View the document as admin
3. Verify "Save as Template" button is NOT visible

### Test Use Template
1. Go to templates page
2. Click "Use This Template" on any template
3. Verify redirect to create page
4. (Future: verify fields are pre-loaded)

### Test Delete Template
1. Go to templates page
2. Click delete on a template
3. Confirm deletion
4. Verify template is removed from list

## Initialization

Run the initialization script to create template tables:

```bash
node scripts/initTemplates.js
```

Or call via admin panel:
```bash
POST /document-signing/initialize-db
```

## Files Modified

- `models/signatureDocumentModel.js` - Added template methods
- `routes/documentSigning.js` - Added template routes
- `views/signature_document_view.ejs` - Added edit/template buttons
- `views/signature_document_templates.ejs` - New template list page (created)
- `scripts/initTemplates.js` - Database initialization script (created)
