# Messages System Setup Fix

## Issue
When students clicked "Contact Teacher" button, the application crashed with error:
```
Error: Table 'mdtsapps_myclass.mdtslms_messages' doesn't exist
```

## Root Cause
The messaging system was implemented in the code but the database table was never created.

## Solution
Created setup script and database table for the messaging system.

## Database Schema

### Table: `mdtslms_messages`

```sql
CREATE TABLE IF NOT EXISTS mdtslms_messages (
  id INT AUTO_INCREMENT PRIMARY KEY,
  senderId INT NOT NULL,
  recipientId INT NOT NULL,
  subject VARCHAR(255),
  body TEXT,
  sentAt TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  readAt TIMESTAMP NULL,
  INDEX idx_sender (senderId),
  INDEX idx_recipient (recipientId),
  INDEX idx_sent_at (sentAt)
)
```

### Fields:
- **id** - Auto-incrementing primary key
- **senderId** - User ID of message sender (foreign key to users table)
- **recipientId** - User ID of message recipient (foreign key to users table)
- **subject** - Message subject line (optional, 255 chars max)
- **body** - Message content (TEXT field for longer messages)
- **sentAt** - Timestamp when message was sent (auto-set on creation)
- **readAt** - Timestamp when message was read (NULL if unread)

### Indexes:
- **idx_sender** - Fast lookup of messages sent by a user
- **idx_recipient** - Fast lookup of messages received by a user
- **idx_sent_at** - Sorting messages by date

## Setup Script
**Location:** `scripts/setupMessagesTable.js`

**Usage:**
```bash
node scripts/setupMessagesTable.js
```

**What it does:**
- Connects to the database using existing db.js configuration
- Creates the `mdtslms_messages` table if it doesn't exist
- Adds all necessary indexes
- Exits with success/error code

## Messaging System Features

### For Students:
- **Send messages** to teachers
- **View mailbox** with all sent and received messages
- **Pre-selected recipient** when clicking "Contact Teacher"
- **Subject line** for organized communication
- **Message body** with unlimited text

### For Teachers:
- **Receive messages** from students
- **Reply** to student questions
- **View conversation history**
- **Professional communication channel**

### Routes:
- **GET /student/mailbox** - View messages and compose new ones
- **GET /student/mailbox?compose=123** - Pre-select recipient (teacher ID 123)
- **POST /student/mailbox** - Send a new message

### Model Functions:
- **sendMessage(senderId, recipientId, subject, body)** - Create and send message
- **getMailbox(userId)** - Get all messages for a user (sent or received)

## Files Involved

1. **scripts/setupMessagesTable.js** - Database setup script (NEW)
2. **models/messageModel.js** - Message data operations
3. **routes/student.js** - Mailbox routes
4. **views/mailbox.ejs** - Message interface
5. **views/view_class.ejs** - "Contact Teacher" button

## Testing Checklist

- [x] Database table created successfully
- [x] Server starts without errors
- [x] "Contact Teacher" button doesn't crash
- [x] Mailbox page loads
- [x] Can view messages (empty initially)
- [x] Can send message to teacher
- [x] Teacher pre-selected when coming from "Contact Teacher"
- [ ] Message appears in mailbox after sending
- [ ] Teacher receives message
- [ ] Teacher can reply

## Future Enhancements

1. **Read receipts** - Mark messages as read and show status
2. **Message notifications** - Email or dashboard alerts for new messages
3. **Threaded conversations** - Group related messages together
4. **Attachments** - Allow file uploads with messages
5. **Delete messages** - Let users remove messages from mailbox
6. **Search/filter** - Find specific messages or conversations
7. **Unread count** - Badge showing number of unread messages
8. **Rich text editor** - Format message text with bold, italic, links
9. **Archive** - Move old messages to archive folder
10. **Block/report** - Safety features for inappropriate messages

## Migration Notes

If deploying to production:
1. Run `node scripts/setupMessagesTable.js` before starting the app
2. Verify table exists: `SHOW TABLES LIKE 'mdtslms_messages';`
3. Check indexes: `SHOW INDEX FROM mdtslms_messages;`
4. Test send/receive functionality
5. Monitor for any foreign key constraint issues

## Security Considerations

- No SQL injection - using parameterized queries
- User authentication required - routes protected by session
- Students can only message teachers (enforced in UI)
- All messages stored permanently (consider data retention policy)
- No encryption - messages stored in plain text (consider encrypting sensitive data)

## Performance Notes

- Indexes on sender, recipient, and sentAt ensure fast queries
- Text field for body allows unlimited message length
- Consider pagination if mailbox grows large (100+ messages)
- Archive old messages periodically to maintain performance
