# New Features Implementation Plan - October 2025

## Overview
This document outlines the implementation of several major feature enhancements to the LMS system.

---

## 1. Terminology Change: "Pending" → "Registered Students"

### Rationale
The term "pending" has a negative connotation. "Registered Students" better reflects that these students have completed registration and are awaiting approval.

### Implementation
**Status Field Value**: Keep `status = 'pending'` in database (no breaking changes)
**Display Labels**: Change all UI references from "Pending Students" to "Registered Students"

### Files to Update
- `views/admin_dashboard.ejs` - Dashboard tile label
- `views/admin_pending.ejs` - Page title and headers  
- `views/pending_students_report.ejs` - Report title
- `views/reports.ejs` - Report link text
- `admin_user_guide.html` - Documentation
- `docs/admin-manual.md` - Documentation

### Search/Replace Strategy
- "Pending Students" → "Registered Students"
- "Pending Student" → "Registered Student"
- "pending students" → "registered students"
- Keep code comments and variable names as-is for clarity

---

## 2. Admin Notes Improvements

### A. Text Wrapping in SweetAlert Input

**Problem**: Long notes extend horizontally without wrapping

**Solution**: Configure textarea with proper CSS

```javascript
Swal.fire({
  input: 'textarea',
  inputAttributes: {
    style: 'width: 100%; white-space: pre-wrap; word-wrap: break-word;'
  }
})
```

### B. Show Note Author and Timestamp

**Current Structure**:
```javascript
{
  content: "Note text here",
  createdAt: "2025-10-29T10:00:00Z"
}
```

**New Structure**:
```javascript
{
  content: "Note text here",
  createdAt: "2025-10-29T10:00:00Z",
  authorId: 123,
  authorName: "Admin User"
}
```

**Display Format**:
```
Note text here

— Added by Admin User on Oct 29, 2025 10:00 AM
```

**Files to Modify**:
- `views/student_profile.ejs` - Note display and creation
- `routes/admin.js` - Add authorId and authorName when creating notes
- Database: `profile.adminNotes` array in users table (JSON field)

---

## 3. Send Certificate from Roster

### Implementation
Add "Send Certificate" button to roster Actions column

**Location**: `views/view_class.ejs` - Student Roster section

**Button**:
```html
<button class="btn btn-sm btn-success btn-send-cert-roster" 
        data-student-id="<%= s.id %>" 
        data-student-name="<%= s.name %>"
        data-class-id="<%= klass.id %>">
  <i class="bi bi-award"></i> Send Certificate
</button>
```

**JavaScript Handler**:
```javascript
document.addEventListener('click', async (e) => {
  const btn = e.target.closest('.btn-send-cert-roster');
  if (!btn) return;
  
  const studentId = btn.dataset.studentId;
  const className = btn.dataset.studentName;
  const classId = btn.dataset.classId;
  
  // Confirm dialog
  const result = await Swal.fire({
    title: 'Send Certificate?',
    text: `Send course completion certificate to ${studentName}?`,
    showCancelButton: true
  });
  
  if (!result.isConfirmed) return;
  
  // POST to /admin/students/:id/send-certificate
  const response = await fetch(`/admin/students/${studentId}/send-certificate`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ classId })
  });
  
  // Show success/error
});
```

**Route**: `routes/admin.js`
```javascript
router.post('/students/:id/send-certificate', async (req, res) => {
  const studentId = req.params.id;
  const { classId } = req.body;
  
  // Generate certificate PDF
  // Email to student
  // Mark as sent in profile
  
  res.json({ ok: true });
});
```

---

## 4. Show Student Names on Event RSVP

### Current Issue
Event RSVP lists show user IDs but not names

### Solution
Join with users table to get names

**File**: Routes handling event RSVPs (likely `routes/events.js` or `routes/admin.js`)

**Query Enhancement**:
```javascript
// Before
const rsvps = event.rsvps || [];

// After
const rsvps = await Promise.all((event.rsvps || []).map(async (rsvpId) => {
  const user = await userModel.findById(rsvpId);
  return {
    id: rsvpId,
    name: user?.name || 'Unknown',
    email: user?.email
  };
}));
```

**View Update**: Display names in RSVP list
```html
<% rsvps.forEach(rsvp => { %>
  <li><%= rsvp.name %> (<%= rsvp.email %>)</li>
<% }) %>
```

---

## 5. Alumni Students System

### Overview
Multi-phase lifecycle for students post-graduation

### Student Lifecycle Phases

```
┌─────────────┐
│  Registered │ (status='pending')
│  Student    │
└──────┬──────┘
       │ Approved
       ▼
┌─────────────┐
│   Active    │ (status='approved', enrolled in classes)
│   Student   │
└──────┬──────┘
       │ Course Ends
       ▼
┌─────────────┐
│ Post-Course │ (1 week grace period)
│   Access    │ Full access to materials
└──────┬──────┘
       │ After 1 week
       ▼
┌─────────────┐
│   Alumni    │ (status='alumni')
│   Student   │ Lecture-only access
└──────┬──────┘
       │ Re-enrolls
       ▼
┌─────────────┐
│   Active    │ (status='approved')
│   Student   │ Full access restored
└─────────────┘
```

### A. Database Changes

**User Model** - Add new status value:
```javascript
// status can be: 'pending', 'approved', 'alumni'
```

**Class Model** - Track course end date:
```javascript
// Already exists: endDate field
```

### B. Access Control Logic

**File**: `routes/student.js` - Class viewing route

```javascript
router.get('/classes/:id', async (req, res) => {
  const student = req.session.user;
  const klass = await classModel.findClassById(req.params.id);
  
  // Check if student is alumni
  const isAlumni = student.status === 'alumni';
  
  // Check if within 1-week grace period
  const endDate = new Date(klass.endDate);
  const now = new Date();
  const daysAfterEnd = Math.floor((now - endDate) / (1000 * 60 * 60 * 24));
  const withinGracePeriod = daysAfterEnd >= 0 && daysAfterEnd <= 7;
  
  // Determine access level
  let accessLevel = 'full'; // lectures, tests, sims, assignments
  
  if (isAlumni && !withinGracePeriod) {
    accessLevel = 'lectures-only';
    // Filter out tests, sims, assignments
    klass.tests = [];
    klass.simulations = [];
    klass.assignments = [];
  }
  
  res.render('view_class', { 
    klass, 
    studentView: true,
    accessLevel,
    withinGracePeriod
  });
});
```

### C. Auto-Transition to Alumni Status

**Background Job** (or cron route):
```javascript
async function transitionToAlumni() {
  const classes = await classModel.getAllClasses();
  const now = new Date();
  
  for (const klass of classes) {
    const endDate = new Date(klass.endDate);
    const daysAfterEnd = Math.floor((now - endDate) / (1000 * 60 * 60 * 24));
    
    // If course ended more than 7 days ago
    if (daysAfterEnd > 7) {
      // Get all active students in this class
      const studentIds = klass.studentIds || [];
      
      for (const studentId of studentIds) {
        const student = await userModel.findById(studentId);
        
        // If still active, transition to alumni
        if (student.status === 'approved') {
          // Check if enrolled in any OTHER active classes
          const otherActiveClasses = await classModel.getAllClasses().filter(c => {
            const cEndDate = new Date(c.endDate);
            return cEndDate > now && c.studentIds.includes(studentId);
          });
          
          // Only transition if NO other active enrollments
          if (otherActiveClasses.length === 0) {
            await userModel.updateUser(studentId, { status: 'alumni' });
          }
        }
      }
    }
  }
}

// Run daily via cron or scheduled job
```

### D. Re-enrollment Logic

**When student enrolls in new class**:
```javascript
async function enrollStudent(classId, studentId) {
  const student = await userModel.findById(studentId);
  
  // If alumni, restore to active status
  if (student.status === 'alumni') {
    await userModel.updateUser(studentId, { status: 'approved' });
  }
  
  // Add to class
  await classModel.addStudent(classId, studentId);
}
```

### E. Alumni Section in Admin

**New View**: `views/admin_alumni.ejs`

Similar to pending students page but for alumni

**Route**:
```javascript
router.get('/alumni', async (req, res) => {
  const users = await userModel.getAll();
  const alumniStudents = users.filter(u => u.status === 'alumni');
  
  res.render('admin_alumni', { 
    alumni: alumniStudents,
    title: 'Alumni Students'
  });
});
```

**Dashboard Tile**: Add "Alumni Students" tile to admin dashboard

---

## 6. Historical Gradebook

### Overview
Allow admins to manually enter past course grades for students who took courses before the LMS existed or at other institutions.

### A. Database Structure

**User Profile Field**: `historicalGrades`

```javascript
profile: {
  // ... existing fields
  historicalGrades: [
    {
      id: 1,
      courseName: "CASP Prep Course",
      schoolYear: "2024",
      cohort: "Fall 2024",
      startDate: "2024-09-01",
      endDate: "2024-12-15",
      clockHours: 720,
      clockHoursCompleted: 720,
      grades: [
        { type: 'test', name: 'Midterm Exam', score: 85 },
        { type: 'test', name: 'Final Exam', score: 92 },
        { type: 'assignment', name: 'Project 1', score: 88 }
      ],
      overallGrade: 88,
      letterGrade: 'B+',
      notes: 'Completed before LMS system',
      addedBy: 'Admin Name',
      addedAt: '2025-10-29T10:00:00Z'
    }
  ]
}
```

### B. Admin Interface

**Student Profile Section**: Add "Historical Gradebook" section

```html
<section class="cardish">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <div class="section-title mb-0">Historical Gradebook</div>
    <button type="button" class="btn btn-sm btn-primary" id="addHistoricalCourseBtn">
      <i class="bi bi-plus-circle"></i> Add Past Course
    </button>
  </div>
  
  <% if (!student.profile?.historicalGrades?.length) { %>
    <div class="alert alert-info">No historical courses recorded.</div>
  <% } else { %>
    <% student.profile.historicalGrades.forEach(course => { %>
      <div class="card mb-3">
        <div class="card-header">
          <strong><%= course.courseName %></strong>
          <span class="text-muted">(<%= course.schoolYear %>)</span>
          <button class="btn btn-sm btn-outline-danger float-end delete-historical-course" 
                  data-course-id="<%= course.id %>">Delete</button>
          <button class="btn btn-sm btn-outline-primary float-end me-2 edit-historical-course" 
                  data-course-id="<%= course.id %>">Edit</button>
        </div>
        <div class="card-body">
          <div class="row">
            <div class="col-md-6">
              <strong>Period:</strong> <%= course.startDate %> to <%= course.endDate %><br>
              <strong>Cohort:</strong> <%= course.cohort %><br>
              <strong>Clock Hours:</strong> <%= course.clockHoursCompleted %> / <%= course.clockHours %>
            </div>
            <div class="col-md-6">
              <strong>Overall Grade:</strong> <%= course.overallGrade %>% (<%= course.letterGrade %>)<br>
              <strong>Added by:</strong> <%= course.addedBy %> on <%= new Date(course.addedAt).toLocaleDateString() %>
            </div>
          </div>
          
          <% if (course.grades?.length) { %>
            <table class="table table-sm mt-3">
              <thead>
                <tr>
                  <th>Type</th>
                  <th>Name</th>
                  <th>Score</th>
                </tr>
              </thead>
              <tbody>
                <% course.grades.forEach(grade => { %>
                  <tr>
                    <td class="text-capitalize"><%= grade.type %></td>
                    <td><%= grade.name %></td>
                    <td><%= grade.score %>%</td>
                  </tr>
                <% }) %>
              </tbody>
            </table>
          <% } %>
          
          <% if (course.notes) { %>
            <div class="mt-2">
              <strong>Notes:</strong> <%= course.notes %>
            </div>
          <% } %>
        </div>
      </div>
    <% }) %>
  <% } %>
</section>
```

**Add/Edit Modal**:
```javascript
document.getElementById('addHistoricalCourseBtn').addEventListener('click', async () => {
  const { value: formValues } = await Swal.fire({
    title: 'Add Historical Course',
    width: 800,
    html: `
      <div class="text-start">
        <div class="mb-3">
          <label class="form-label">Course Name</label>
          <input id="courseName" class="form-control" placeholder="e.g., CASP Prep Course">
        </div>
        <div class="row mb-3">
          <div class="col-md-6">
            <label class="form-label">School Year</label>
            <input id="schoolYear" class="form-control" placeholder="e.g., 2024">
          </div>
          <div class="col-md-6">
            <label class="form-label">Cohort</label>
            <input id="cohort" class="form-control" placeholder="e.g., Fall 2024">
          </div>
        </div>
        <div class="row mb-3">
          <div class="col-md-6">
            <label class="form-label">Start Date</label>
            <input type="date" id="startDate" class="form-control">
          </div>
          <div class="col-md-6">
            <label class="form-label">End Date</label>
            <input type="date" id="endDate" class="form-control">
          </div>
        </div>
        <div class="row mb-3">
          <div class="col-md-6">
            <label class="form-label">Clock Hours (Total)</label>
            <input type="number" id="clockHours" class="form-control" placeholder="e.g., 720">
          </div>
          <div class="col-md-6">
            <label class="form-label">Clock Hours Completed</label>
            <input type="number" id="clockHoursCompleted" class="form-control">
          </div>
        </div>
        <div class="row mb-3">
          <div class="col-md-6">
            <label class="form-label">Overall Grade (%)</label>
            <input type="number" id="overallGrade" class="form-control" min="0" max="100">
          </div>
          <div class="col-md-6">
            <label class="form-label">Letter Grade</label>
            <select id="letterGrade" class="form-control">
              <option>A+</option>
              <option>A</option>
              <option>A-</option>
              <option>B+</option>
              <option>B</option>
              <option>B-</option>
              <option>C+</option>
              <option>C</option>
              <option>C-</option>
              <option>D</option>
              <option>F</option>
            </select>
          </div>
        </div>
        <div class="mb-3">
          <label class="form-label">Notes</label>
          <textarea id="notes" class="form-control" rows="3"></textarea>
        </div>
        <div id="gradesContainer">
          <label class="form-label">Individual Grades (Optional)</label>
          <button type="button" class="btn btn-sm btn-outline-primary mb-2" id="addGradeRow">
            <i class="bi bi-plus"></i> Add Grade
          </button>
          <div id="gradeRows"></div>
        </div>
      </div>
    `,
    focusConfirm: false,
    showCancelButton: true,
    confirmButtonText: 'Save Course',
    didOpen: () => {
      document.getElementById('addGradeRow').addEventListener('click', () => {
        const container = document.getElementById('gradeRows');
        const row = document.createElement('div');
        row.className = 'row mb-2';
        row.innerHTML = `
          <div class="col-md-4">
            <select class="form-control grade-type">
              <option value="test">Test</option>
              <option value="assignment">Assignment</option>
              <option value="lab">Lab</option>
            </select>
          </div>
          <div class="col-md-5">
            <input type="text" class="form-control grade-name" placeholder="Name">
          </div>
          <div class="col-md-2">
            <input type="number" class="form-control grade-score" placeholder="Score" min="0" max="100">
          </div>
          <div class="col-md-1">
            <button type="button" class="btn btn-sm btn-outline-danger remove-grade">×</button>
          </div>
        `;
        row.querySelector('.remove-grade').addEventListener('click', () => row.remove());
        container.appendChild(row);
      });
    },
    preConfirm: () => {
      // Collect form data
      const grades = [];
      document.querySelectorAll('#gradeRows .row').forEach(row => {
        grades.push({
          type: row.querySelector('.grade-type').value,
          name: row.querySelector('.grade-name').value,
          score: Number(row.querySelector('.grade-score').value)
        });
      });
      
      return {
        courseName: document.getElementById('courseName').value,
        schoolYear: document.getElementById('schoolYear').value,
        cohort: document.getElementById('cohort').value,
        startDate: document.getElementById('startDate').value,
        endDate: document.getElementById('endDate').value,
        clockHours: Number(document.getElementById('clockHours').value),
        clockHoursCompleted: Number(document.getElementById('clockHoursCompleted').value),
        overallGrade: Number(document.getElementById('overallGrade').value),
        letterGrade: document.getElementById('letterGrade').value,
        notes: document.getElementById('notes').value,
        grades
      };
    }
  });
  
  if (formValues) {
    // POST to /admin/students/:id/historical-grades
    const response = await fetch(`/admin/students/${studentId}/historical-grades`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(formValues)
    });
    
    if (response.ok) {
      location.reload();
    }
  }
});
```

### C. Backend Routes

**routes/admin.js**:
```javascript
router.post('/students/:id/historical-grades', async (req, res) => {
  const studentId = req.params.id;
  const courseData = req.body;
  const adminName = req.session.user.name;
  
  const student = await userModel.findById(studentId);
  const profile = student.profile || {};
  const historicalGrades = profile.historicalGrades || [];
  
  // Generate ID
  const newId = historicalGrades.length ? Math.max(...historicalGrades.map(c => c.id)) + 1 : 1;
  
  historicalGrades.push({
    id: newId,
    ...courseData,
    addedBy: adminName,
    addedAt: new Date().toISOString()
  });
  
  await userModel.updateProfile(studentId, {
    ...profile,
    historicalGrades
  });
  
  res.json({ ok: true });
});

router.delete('/students/:id/historical-grades/:courseId', async (req, res) => {
  const { id: studentId, courseId } = req.params;
  
  const student = await userModel.findById(studentId);
  const profile = student.profile || {};
  const historicalGrades = (profile.historicalGrades || []).filter(c => c.id !== Number(courseId));
  
  await userModel.updateProfile(studentId, {
    ...profile,
    historicalGrades
  });
  
  res.json({ ok: true });
});
```

### D. Student/Teacher View

**Read-Only Display**: Same table structure but no edit/delete buttons

**Add to**:
- Student dashboard/profile
- Teacher student view
- Admin can edit, others can only view

---

## Implementation Priority

1. **High Priority** (Quick wins):
   - [ ] Change "Pending" to "Registered Students" (terminology)
   - [ ] Fix note text wrapping in SweetAlert
   - [ ] Add note author/date display
   - [ ] Send certificate button on roster

2. **Medium Priority** (Moderate effort):
   - [ ] Show student names on event RSVP
   - [ ] Historical gradebook feature

3. **Low Priority** (Complex, requires careful planning):
   - [ ] Alumni student system
   - [ ] Post-course access grace period
   - [ ] Auto-transition logic

---

## Testing Checklist

### Terminology Change
- [ ] All admin views show "Registered Students"
- [ ] Dashboard tiles updated
- [ ] Reports renamed
- [ ] Help text updated
- [ ] No instances of "Pending Students" remain in UI

### Notes
- [ ] Long notes wrap properly
- [ ] Author name displays
- [ ] Timestamp displays in readable format
- [ ] New notes capture current admin's info

### Certificates
- [ ] Button appears on roster
- [ ] Confirmation dialog works
- [ ] Certificate generates and emails
- [ ] Error handling works

### Historical Gradebook
- [ ] Admin can add courses
- [ ] Admin can edit courses
- [ ] Admin can delete courses
- [ ] Student can view (read-only)
- [ ] Teacher can view (read-only)
- [ ] Data persists correctly

### Alumni System
- [ ] Students transition to alumni after 7 days
- [ ] Alumni see lectures only
- [ ] Grace period allows full access
- [ ] Re-enrollment restores full access
- [ ] Dashboard shows alumni count

---

*Implementation Plan Created: October 29, 2025*
