# Step 2 Multi-Program Enrollment & Demographics Update

## Overview
This feature restructures the Step 2 section to support **multiple program enrollments**, each with its own program details and tuition breakdown. Additionally, the **Grievance Acknowledged** field has been moved from the Program section to the Demographics section for better organization.

## Part 1: Grievance Acknowledged Migration

### Changes Made

#### 1. Removed from Programs Section
**Location:** `views/student_profile.ejs`

**Before:** Located at the bottom of the admin Program Enrollments section
**After:** Completely removed from Programs section

#### 2. Added to Demographics Section
**Location:** `views/student_profile.ejs` (Lines ~654-672)

**New Section Title:** "Demographics & Personal Info"

**New Field:**
```html
<div class="col-md-6">
  <label class="form-label" for="grievanceAcknowledged">Grievance Acknowledged</label>
  <select id="grievanceAcknowledged" name="grievanceAcknowledged" class="form-select">
    <option value="false">No</option>
    <option value="true">Yes</option>
  </select>
</div>
```

**Layout:**
- Row 1: Gender (col-md-6) | Race (col-md-6)
- Row 2: Grievance Acknowledged (col-md-6)
- Submit Button: "Save Demographics"

#### 3. Backend Update
**Location:** `routes/admin.js` - POST `/students/:id/demographics`

**Updated Logic:**
```javascript
const grievanceAcknowledged = req.body.grievanceAcknowledged === 'true' || req.body.grievanceAcknowledged === true;

await userModel.updateProfile(id, { gender, race, grievanceAcknowledged });
```

## Part 2: Step 2 Multi-Program Structure

### Overview
Step 2 has been completely restructured from a single program/tuition form to a **dynamic list** where admins can add multiple program enrollments, each with:
- Program details (course, dates, times)
- Individual tuition breakdown
- Separate total cost per program

### Frontend Implementation

#### Location
**File:** `views/student_profile.ejs` (Lines ~900-1050)

#### UI Components

**Header Section:**
```html
<div class="d-flex justify-content-between align-items-center mb-3">
  <div class="section-title mb-0">Step 2: Program Enrollments & Tuition</div>
  <button type="button" class="btn btn-sm btn-success" onclick="addStep2ProgramEntry()">
    <i class="bi bi-plus-circle"></i> Add Program
  </button>
</div>
```

**Program Entry Card:**
Each program enrollment displays as a card containing:

1. **Header:**
   - Program number and course name
   - Remove button (trash icon)

2. **Program Details Section:**
   - Course Name (required)
   - Class Time
   - Start Date
   - End Date
   - Class Days

3. **Tuition Breakdown Section:**
   - Tuition ($)
   - Registration Fee ($)
   - Books/Supplies ($)
   - Equipment ($)
   - Misc. Fees ($)
   - Total Cost ($) - Bold

**Empty State:**
```html
<div class="alert alert-info mb-0">
  <i class="bi bi-info-circle"></i> No program enrollments yet. Click "Add Program" to create one.
</div>
```

**Submit Button:**
```html
<button type="submit" class="btn btn-primary">
  <i class="bi bi-send"></i> Send Program Details to Student
</button>
```

### Data Structure

#### New Format (step2Programs Array)
```javascript
{
  profile: {
    step2Programs: [
      {
        course: "Security+",
        startDate: "2025-10-20",
        endDate: "2025-10-24",
        classTime: "8:00pm",
        classDays: "Mon-Fri",
        tuition: {
          tuition: "2500.00",
          registrationFee: "100.00",
          books: "150.00",
          equipment: "200.00",
          miscFees: "50.00",
          totalCost: "3000.00"
        }
      },
      {
        course: "Network+",
        startDate: "2025-11-01",
        endDate: "2025-11-30",
        classTime: "6:00pm",
        classDays: "Tue-Thu",
        tuition: {
          tuition: "2800.00",
          registrationFee: "100.00",
          books: "175.00",
          equipment: "225.00",
          miscFees: "50.00",
          totalCost: "3350.00"
        }
      }
    ]
  }
}
```

#### Old Format (Backwards Compatible)
```javascript
{
  profile: {
    course: "Security+",
    program: {
      startDate: "2025-10-20",
      endDate: "2025-10-24",
      classTime: "8:00pm",
      classDays: "33"
    },
    tuition: {
      tuition: "2500.00",
      registrationFee: "100.00",
      books: "150.00",
      equipment: "200.00",
      miscFees: "50.00",
      totalCost: "3000.00"
    }
  }
}
```

**Migration:** The view automatically converts old structure to new array format on render.

### JavaScript Functions

#### Location
**File:** `views/student_profile.ejs` (Lines ~2327-2478)

#### Functions

**`addStep2ProgramEntry()`**
- Creates new program entry card with all fields
- Increments counter for unique field names
- Removes empty state if present
- Appends new entry to container
- Auto-focuses on course name input

**`removeStep2ProgramEntry(index)`**
- Shows confirmation dialog (SweetAlert2)
- Removes entry from DOM on confirmation
- Shows empty state if no entries remain

**Counter Management:**
```javascript
let step2ProgramCounter = document.querySelectorAll('.step2-program-entry').length;
```

### Backend Implementation

#### Route Handler
**File:** `routes/admin.js`
**Route:** `POST /admin/students/:id/step2`
**Lines:** ~2630-2677

#### Request Processing

**Input:**
```javascript
{
  step2Programs: {
    0: {
      course: "Security+",
      startDate: "2025-10-20",
      endDate: "2025-10-24",
      classTime: "8:00pm",
      classDays: "Mon-Fri",
      tuition: {
        tuition: "2500.00",
        registrationFee: "100.00",
        // ... other tuition fields
      }
    },
    1: {
      // ... second program
    }
  }
}
```

**Processing Logic:**
1. Extract `step2Programs` from request body
2. Convert object to array format
3. Map each program entry with trimmed values
4. Filter out entries without course name
5. Save as `profile.step2Programs` array

**Code:**
```javascript
let programsArray = [];
if (step2Programs && typeof step2Programs === 'object') {
  programsArray = Object.values(step2Programs).map(prog => ({
    course: prog.course?.trim() || '',
    startDate: prog.startDate?.trim() || '',
    endDate: prog.endDate?.trim() || '',
    classTime: prog.classTime?.trim() || '',
    classDays: prog.classDays?.trim() || '',
    tuition: {
      tuition: prog.tuition?.tuition || '',
      registrationFee: prog.tuition?.registrationFee || '',
      books: prog.tuition?.books || '',
      equipment: prog.tuition?.equipment || '',
      miscFees: prog.tuition?.miscFees || '',
      totalCost: prog.tuition?.totalCost || ''
    }
  })).filter(prog => prog.course);
}

await userModel.updateProfile(id, {
  step2Programs: programsArray
});
```

**Email Notification:**
After saving, sends enrollment step 2 email to student with token link (unchanged).

### Form Field Naming

**Pattern:** `step2Programs[index][field]` and `step2Programs[index][tuition][field]`

**Examples:**
```html
<input name="step2Programs[0][course]" value="Security+">
<input name="step2Programs[0][startDate]" value="2025-10-20">
<input name="step2Programs[0][tuition][tuition]" value="2500.00">
<input name="step2Programs[1][course]" value="Network+">
```

### UI/UX Features

#### Styling
- **Cards:** Light gray background (`bg-light`)
- **Headers:** Primary blue color for course titles
- **Currency Inputs:** Dollar sign prefix in input groups
- **Small Inputs:** Compact input groups (`input-group-sm`)
- **Bold Total:** Total cost field is bold

#### Icons
- **Program Icon:** `bi-mortarboard-fill`
- **Add Icon:** `bi-plus-circle`
- **Remove Icon:** `bi-trash`
- **Send Icon:** `bi-send`
- **Info Icon:** `bi-info-circle`

#### Validation
- Course name marked as `required`
- Backend filters empty entries
- Only programs with course names are saved

### Backwards Compatibility

#### Migration Strategy
**View-Level Migration:**
```ejs
<% 
  const step2Programs = (student.profile?.step2Programs || []);
  // Backwards compatibility: migrate old structure if exists
  if (!step2Programs.length && student.profile?.program) {
    step2Programs.push({
      course: student.profile?.course || '',
      startDate: student.profile?.program?.startDate || '',
      endDate: student.profile?.program?.endDate || '',
      classTime: student.profile?.program?.classTime || '',
      classDays: student.profile?.program?.classDays || '',
      tuition: {
        tuition: student.profile?.tuition?.tuition || '',
        // ... other tuition fields
      }
    });
  }
%>
```

**Benefits:**
- No database migration required
- Old data automatically converted on render
- Both structures supported
- Gradual transition as profiles are edited

### Benefits

1. **Multiple Programs:** Students can enroll in multiple courses with different schedules
2. **Individual Tuition:** Each program has its own tuition breakdown
3. **Clear Organization:** Separate cards for each program
4. **Flexible Dates:** Independent start/end dates per program
5. **Complete Financial Info:** Detailed cost breakdown per program
6. **Better Tracking:** Clear view of all enrolled programs
7. **User-Friendly:** Intuitive add/remove interface

### Use Cases

#### Example 1: Multiple Certifications
Student enrolls in:
- Security+ (Oct-Nov, $3000)
- Network+ (Dec-Jan, $3350)
- CySA+ (Feb-Mar, $3500)

Each has different:
- Start/end dates
- Class times
- Tuition amounts
- Books/equipment costs

#### Example 2: Staggered Enrollment
Student starts with:
- CompTIA A+ (Immediate, $2500)

Later adds:
- Security+ (3 months later, $3000)
- Network+ (6 months later, $3350)

Admin can add programs as student progresses.

### Testing Scenarios

**Test 1: Add Single Program**
1. Click "Add Program"
2. Fill in all fields
3. Submit form
4. Verify data saved in `step2Programs` array

**Test 2: Add Multiple Programs**
1. Click "Add Program" three times
2. Fill in different details for each
3. Submit form
4. Verify all three saved in array

**Test 3: Remove Program**
1. Have multiple programs
2. Click remove on one
3. Confirm deletion
4. Verify removed from form
5. Submit to persist

**Test 4: Empty State**
1. Remove all programs
2. Verify empty state message appears
3. Click "Add Program"
4. Verify empty state removed

**Test 5: Backwards Compatibility**
1. Load student with old structure
2. Verify old data displays in new format
3. Edit and save
4. Verify converted to new structure

**Test 6: Validation**
1. Try to submit empty course field
2. Verify required validation
3. Fill course name
4. Verify submission works

## Demographics Changes Summary

### Before:
- **Section:** "Demographics"
- **Fields:** Gender, Race
- **Location:** Separate section

### After:
- **Section:** "Demographics & Personal Info"
- **Fields:** Gender, Race, Grievance Acknowledged
- **Location:** Combined section
- **Button:** "Save Demographics"

## Files Modified

### 1. `views/student_profile.ejs`
- **Lines ~488-498:** Removed grievance from Programs section
- **Lines ~640-648:** Removed grievance from student read-only
- **Lines ~644-672:** Added grievance to Demographics form
- **Lines ~900-1050:** Complete Step 2 restructure with multi-program support
- **Lines ~2327-2478:** JavaScript functions for Step 2 program management

### 2. `routes/admin.js`
- **Lines ~2137-2141:** Added grievance to demographics route
- **Lines ~2630-2677:** Complete rewrite of step2 route for array handling

## Dependencies

- **Bootstrap 5.3.3:** UI components, forms, input groups
- **Bootstrap Icons:** Icon set
- **SweetAlert2:** Confirmation dialogs
- **Express.js:** Backend routing
- **MySQL:** Database storage (JSON field)
- **EJS:** Template engine

## Migration Notes

### For Existing Students

**Automatic View Migration:**
- Old format displays correctly in new UI
- No data loss
- No manual migration needed

**Data Conversion:**
- Happens on first edit after update
- Old structure → new array structure
- Original data preserved in `program` and `tuition` fields

**Timeline:**
- Deploy update
- Existing students show old data in new format
- On next save, converts to new structure
- Both structures remain supported

## Future Enhancements

### Potential Improvements:
- **Auto-calculate Total:** Sum tuition fields automatically
- **Copy Program:** Duplicate existing program as template
- **Program Templates:** Pre-defined program/tuition combos
- **Status Tracking:** Active/Completed/Withdrawn per program
- **Payment Tracking:** Link payments to specific programs
- **Certificate Generation:** Tie certificates to specific programs
- **Bulk Import:** CSV import for multiple programs
- **Program Comparison:** Side-by-side view of all programs
- **Historical View:** Track all past program enrollments

---

**Implemented:** October 14, 2025
**Status:** ✅ Active and Running
**Server:** Port 3012
