# Template Selection and Preview on Document Creation

## Overview
The document creation page now allows admins to select from saved templates or upload custom PDFs. Templates can be previewed before selection, showing all configured fields.

## Features Added

### 1. Template Selection Interface
- **Visual template cards** displaying all available templates
- **Radio button selection** for choosing a template
- **"Upload Custom PDF" option** for documents without templates
- **Automatic PDF upload toggle** - hides upload when template selected
- **Visual feedback** - selected template highlighted with blue border

### 2. Template Preview Modal
- **Preview button** on each template card
- **Detailed template information**:
  - Template name and description
  - PDF filename
  - Number of fields
  - Complete field list with details
- **Field table** showing:
  - Field label
  - Field type (signature, initials, text, date, checkbox)
  - Signer type (student/staff) with color badges
  - Page number
  - Required status
- **Quick selection** - "Use This Template" button in preview

### 3. Smart Form Behavior
- **Conditional PDF upload**: Hidden when template is selected
- **Template-based creation**: Uses template's PDF and fields automatically
- **Custom upload option**: Still available for non-template documents
- **Validation**: Ensures either template or PDF upload is provided

## User Interface

### Template Card Layout

```html
┌─────────────────────────────────┐
│ 📄 Enrollment Agreement      ○  │
│                                  │
│ Standard enrollment document    │
│ for new students                │
│                                  │
│ 📄 enrollment.pdf               │
│ ☑ 12 fields                     │
│                                  │
│ [👁 Preview]                    │
└─────────────────────────────────┘
```

### Preview Modal Structure

```
┌──────────────────────────────────────────┐
│ 👁 Template Preview: Enrollment Agreement│
│                                 [Close]  │
├──────────────────────────────────────────┤
│ Description: Standard enrollment...      │
│ PDF File: enrollment.pdf                 │
│ Number of Fields: 12                     │
│                                          │
│ Fields:                                  │
│ ┌────────────────────────────────────┐  │
│ │Label      Type   Signer  Page Req │  │
│ ├────────────────────────────────────┤  │
│ │Student... sign.. student  1   ✓   │  │
│ │Date       date   student  1   ✓   │  │
│ │Staff...   sign.. staff    2   ✓   │  │
│ └────────────────────────────────────┘  │
│                                          │
│ ℹ This template will use the existing   │
│   PDF and field configuration.          │
│                                          │
│         [Close]  [✓ Use This Template]  │
└──────────────────────────────────────────┘
```

## Implementation Details

### Route Changes

#### GET /document-signing/create/:studentId

**Before:**
```javascript
res.render('admin_signature_document_create', {
  user: req.session.user,
  role: req.session.role,
  student: student[0]
});
```

**After:**
```javascript
const templates = await signatureDocumentModel.getAllTemplates();

res.render('admin_signature_document_create', {
  user: req.session.user,
  role: req.session.role,
  student: student[0],
  templates: templates
});
```

#### POST /document-signing/upload

**New Logic:**
1. Check if `template_id` is provided in request body
2. **If template selected**:
   - Load template from database
   - Use template's PDF file reference
   - Create document with template data
   - Copy all fields from template to new document
   - Log as "created_from_template"
3. **If no template**:
   - Require file upload
   - Create document with uploaded file
   - Log as "document_uploaded"

**Code:**
```javascript
if (template_id && template_id !== '') {
  // Load template
  const template = await signatureDocumentModel.getTemplateById(parseInt(template_id));
  
  // Use template's PDF
  documentData = {
    student_id: parseInt(student_id),
    created_by: req.session.user.id,
    original_filename: template.original_filename,
    stored_filename: template.stored_filename,
    file_path: template.file_path,
    title: title,
    description: description || template.description,
    signing_order: signing_order || 'any_order'
  };
  
  documentId = await signatureDocumentModel.createDocument(documentData);
  
  // Copy fields
  for (const field of template.fields) {
    await signatureDocumentModel.addField({
      document_id: documentId,
      // ... field properties
    });
  }
}
```

### View Changes

#### Template Cards Section

Added before PDF upload area:
```html
<% if (templates && templates.length > 0) { %>
<div class="mb-4">
  <label class="form-label fw-bold">
    <i class="bi bi-collection"></i> Use a Template (Optional)
  </label>
  <div class="alert alert-info">
    Select a pre-configured template or upload your own PDF below.
  </div>
  <div class="row g-3" id="templatesList">
    <% templates.forEach(template => { %>
      <div class="col-md-6 col-lg-4">
        <div class="card h-100 template-card" data-template-id="<%= template.id %>">
          <!-- Template card content -->
        </div>
      </div>
    <% }); %>
    <!-- Custom upload option card -->
  </div>
</div>
<% } %>
```

#### JavaScript Functions

**updateTemplateSelection()**
- Updates visual styling for selected template
- Shows/hides PDF upload section
- Sets file input required/optional

**showTemplatePreview(template)**
- Creates and displays Bootstrap modal
- Renders template details in table
- Provides "Use This Template" button

**selectTemplate(templateId)**
- Programmatically selects a template
- Called from preview modal button

### CSS Styling

```css
.template-card {
  transition: all 0.2s;
  border: 2px solid transparent;
}

.template-card:hover {
  transform: translateY(-4px);
  box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

/* Selected state */
.template-card[style*="border-color: var(--primary-color)"] {
  box-shadow: 0 4px 12px rgba(30, 144, 255, 0.2);
}
```

## Workflow

### Creating Document from Template

1. **Admin navigates** to create page for student
2. **Template cards display** if any templates exist
3. **Admin clicks** on template card or radio button
4. **Visual feedback**: Card highlighted with blue border
5. **PDF upload section hides** automatically
6. **Admin clicks Preview** (optional) to see fields
7. **Preview modal opens** showing all template details
8. **Admin clicks "Use This Template"** (or closes modal)
9. **Admin fills** title, description, signing order
10. **Admin submits** form
11. **System creates** document using template PDF
12. **System copies** all fields from template
13. **System redirects** to edit page for further customization

### Creating Document without Template

1. **Admin navigates** to create page
2. **Admin clicks** "Upload Custom PDF" card
3. **PDF upload section remains visible**
4. **Admin uploads** PDF file
5. **Admin fills** form details
6. **Admin submits** form
7. **System uploads** new PDF
8. **System creates** empty document
9. **System redirects** to edit page to add fields

## Benefits

### For Admins
- **Faster document creation** - reuse existing configurations
- **Consistency** - ensure all required fields are included
- **Preview before use** - see exactly what's in the template
- **Still flexible** - can upload custom PDFs when needed
- **Visual selection** - easy to browse and choose
- **No re-work** - fields automatically added

### For System
- **Reduced storage** - templates share PDF files
- **Better organization** - standardized documents
- **Audit trail** - tracks template usage
- **Field standardization** - consistent field placement

## API Endpoints Used

### GET /document-signing/template/:templateId
- Returns template with fields
- Used for preview modal
- Already existed from template feature

### POST /document-signing/upload
- Enhanced to accept `template_id` parameter
- Creates document from template if provided
- Falls back to file upload if no template

## Database Impact

### Tables Used
- `signature_document_templates` - Template metadata
- `signature_template_fields` - Template field definitions
- `signature_documents` - Created documents
- `signature_fields` - Copied fields from template

### Field Copying
When creating from template:
```sql
-- For each field in template:
INSERT INTO signature_fields (
  document_id, field_type, signer_type, field_label,
  page_number, x_position, y_position, width, height,
  is_required, placeholder_text
) VALUES (
  new_document_id, 
  template_field.field_type,
  template_field.signer_type,
  -- ... etc
);
```

### Audit Log
```sql
INSERT INTO signature_audit_log (
  document_id, user_id, action, details
) VALUES (
  new_document_id, 
  admin_id, 
  'document_created_from_template',
  'Document "X" created from template "Y"'
);
```

## Testing Scenarios

### Test Template Selection
1. ✅ Navigate to create page with templates
2. ✅ See template cards displayed
3. ✅ Click template card - radio button selected
4. ✅ See blue border on selected template
5. ✅ Verify PDF upload section hidden
6. ✅ Submit form - document created successfully

### Test Template Preview
1. ✅ Click "Preview" button on template
2. ✅ Modal opens with template details
3. ✅ See all fields in table
4. ✅ Click "Use This Template" - modal closes, template selected
5. ✅ Click "Close" - modal closes, selection unchanged

### Test Custom Upload
1. ✅ Click "Upload Custom PDF" card
2. ✅ Verify PDF upload section visible
3. ✅ Upload PDF file
4. ✅ Submit form - document uploaded successfully

### Test Form Validation
1. ✅ Select template - file upload not required
2. ✅ Deselect template - file upload required again
3. ✅ Try submit without file or template - error shown

### Test Template Usage
1. ✅ Create document from template
2. ✅ Go to edit page
3. ✅ Verify all template fields present
4. ✅ Verify PDF is template's PDF
5. ✅ Fields can be moved/edited/deleted

## Edge Cases Handled

1. **No templates exist** - Template section doesn't display
2. **Template deleted** - Error shown if selected
3. **Template file missing** - Error shown on creation
4. **Multiple templates** - All displayed, user can choose
5. **Large template list** - Responsive grid layout
6. **Template with no fields** - Still creates document, admin can add fields

## Security Considerations

1. **Admin-only access** - requireAdmin middleware
2. **Template validation** - Template exists check
3. **Student validation** - Student exists check
4. **File sharing** - Templates share PDFs (read-only)
5. **Audit logging** - All actions logged
6. **Input sanitization** - Title, description sanitized

## Performance

- **Template loading**: One query on page load
- **Preview**: Single AJAX request per preview
- **Field copying**: Batch insert for all fields
- **File sharing**: No file duplication, instant creation
- **Response time**: < 1 second for template-based creation

## Future Enhancements

1. **Template search/filter** - Find templates by name
2. **Template categories** - Group related templates
3. **Template preview PDF** - Show actual PDF preview
4. **Inline field preview** - Visual field placement
5. **Template statistics** - Usage count, last used
6. **Favorite templates** - Mark frequently used
7. **Template cloning** - Duplicate for modification
8. **Template versions** - Track changes over time

## Files Modified

1. **routes/documentSigning.js**
   - Updated GET /create/:studentId - fetch templates
   - Updated POST /upload - handle template selection

2. **views/admin_signature_document_create.ejs**
   - Added template cards section
   - Added preview modal HTML
   - Added JavaScript for selection/preview
   - Added CSS for template cards

## Related Documentation

- [DOCUMENT_TEMPLATES_FEATURE.md](./DOCUMENT_TEMPLATES_FEATURE.md) - Template creation and management
- [DOCUMENT_SIGNING_README.md](./DOCUMENT_SIGNING_README.md) - Overall document signing system
- [RESIZABLE_FIELDS_AND_PROGRESS_EMAILS.md](./RESIZABLE_FIELDS_AND_PROGRESS_EMAILS.md) - Field editing features
