const express = require('express');
const router = express.Router();

const classModel = require('../models/classModel');
const userModel = require('../models/userModel');
const db = require('../models/db');
const discussionModel = require('../models/discussionModel');
const preRegModel = require('../models/preRegModel');

const eventModel = require('../models/eventModel');
const rsvpModel = require('../models/rsvpModel');
const interviewModel = require('../models/interviewModel');
const calendarEventModel = require('../models/calendarEventModel');
const careerAdvisorInterviewModel = require('../models/careerAdvisorInterviewModel');
const careerAdvisorSettings = require('../models/careerAdvisorSettings');
const emailTemplates = require('../utils/emailTemplates');
const marketingTemplates = require('../data/marketingTemplates.json');
const marketingSubjects = Object.fromEntries(
  Object.entries(marketingTemplates).map(([k, v]) => [k, { subject: v.subject }])
);
const announcementModel = require('../models/announcementModel');
const testModel = require('../models/testModel');
const dripCampaign = require('../utils/dripCampaign');
const dropdowns = require('../utils/dropdownStore');
const courseModel = require('../models/courseModel');
const leadModel = require('../models/leadModel');
const partnerModel = require('../models/partnerModel');
const resourceLibrary = require('../utils/resourceLibrary');
const mentorModel = require('../models/mentorModel');
const todoModel = require('../models/todoModel');
const mergeDb = require('../models/db');
const notificationSettings = require('../utils/notificationSettings');
const signatureDocumentModel = require('../models/signatureDocumentModel');
const testProgressModel = require('../models/testProgressModel');
const scheduleModel = require('../models/scheduleModel');
const { checkAlumniMiddleware, checkAllStudentsAlumniStatus } = require('../middleware/alumniAutoTransition');

const multer = require('multer');
const crypto = require('crypto');
const path = require('path');

const upload = multer();

const fs = require('fs');
const branding = require('../branding.json');
const brandingPath = path.join(__dirname, '..', 'branding.json');

const CONFIG_PATH = path.join(__dirname, '..', 'data', 'signature-docs.json');
let signatureDocsConfig = {};

function loadSignatureDocsConfig() {
  try {
    signatureDocsConfig = JSON.parse(fs.readFileSync(CONFIG_PATH, 'utf8'));
  } catch (err) {
    console.error('Failed to load signature-docs.json:', err.message);
    signatureDocsConfig = {};
  }
}
loadSignatureDocsConfig();

// Storage for test media uploads
const mediaStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, path.join(__dirname, '../uploads'));
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + '-' + file.originalname);
  }
});
const mediaUpload = multer({ storage: mediaStorage });
const brandingStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, path.join(__dirname, '../uploads'));
  },
  filename: (req, file, cb) => {
    cb(null, 'logo-' + Date.now() + path.extname(file.originalname));
  }
});
const brandingUpload = multer({ storage: brandingStorage });
const nodemailer = require('nodemailer');
const PDFDocument = require('pdfkit');
const transporter = nodemailer.createTransport({
  host: 'mdts-apps.com',
  port: 465,
  secure: true,
  auth: {
   user: 'noreply@mdts-apps.com',
    pass: 'c@r,5ysPI@&s'
  }
});

let marked;
try {
  marked = require('marked');
} catch (err) {
  console.warn('[admin manual] marked package not found, using plain text manual rendering');
  marked = null;
}

function generateTeacherTempPassword() {
  const letters = 'ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz';
  const upper = 'ABCDEFGHJKLMNPQRSTUVWXYZ';
  const digits = '23456789';
  const symbols = '!@#$%';
  const randomFrom = (source, length) => Array.from({ length }, () => source[Math.floor(Math.random() * source.length)]).join('');
  const segment1 = randomFrom(letters, 6);
  const segment2 = randomFrom(upper, 2);
  const segment3 = randomFrom(digits, 2);
  const segment4 = symbols[Math.floor(Math.random() * symbols.length)];
  const segment5 = randomFrom(letters + digits, 2);
  return `${segment1}${segment2}${segment3}${segment4}${segment5}`;
}

// Uploader for admin-managed student documents/certifications
const docsStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, path.join(__dirname, '../docs/stxd'));
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + '-' + file.originalname);
  }
});
const docsUpload = multer({
  storage: docsStorage,
  limits: { fileSize: 20 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png', '.doc', '.docx'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type'));
  }
}).fields([
  { name: 'docFiles', maxCount: 10 },
  { name: 'certFiles', maxCount: 10 },
  { name: 'toSignFiles', maxCount: 10 }
]);

// Separate uploader for admin "to-sign" docs (reuse same storage/filters)
const toSignUpload = multer({
  storage: docsStorage,
  limits: { fileSize: 20 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png', '.doc', '.docx'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type'));
  }
}).array('toSignFiles', 10);

// Uploader for required documents (ID and transcript) - admin use
const requiredDocsUpload = multer({
  storage: docsStorage,
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type. Only PDF, JPG, and PNG files are allowed.'));
  }
}).fields([
  { name: 'idDocument', maxCount: 1 },
  { name: 'transcriptDocument', maxCount: 1 }
]);

// Uploader for earned certificates - admin use
const certificatesUpload = multer({
  storage: docsStorage,
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type. Only PDF, JPG, and PNG files are allowed.'));
  }
}).array('certificates', 10);

// Uploader for general document uploads to student profile (admin use)
const generalUpload = multer({
  storage: docsStorage,
  limits: { fileSize: 20 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png', '.doc', '.docx'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type'));
  }
}).array('uploadFiles', 10);

const partnerDocsStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    const dir = path.join(__dirname, '../uploads/partner-docs');
    fs.mkdirSync(dir, { recursive: true });
    cb(null, dir);
  },
  filename: (req, file, cb) => {
    const unique = Date.now() + '-' + Math.round(Math.random() * 1e9) + path.extname(file.originalname);
    cb(null, unique);
  }
});
const partnerDocsUpload = multer({
  storage: partnerDocsStorage,
  limits: { fileSize: 20 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png', '.doc', '.docx'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type'));
  }
});

// Syllabus uploader for admin
const syllabusStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    const dir = path.join(__dirname, '../uploads/syllabi');
    fs.mkdirSync(dir, { recursive: true });
    cb(null, dir);
  },
  filename: (req, file, cb) => {
    const unique = Date.now() + '-' + Math.round(Math.random() * 1e9) + path.extname(file.originalname);
    cb(null, unique);
  }
});
const syllabusUpload = multer({
  storage: syllabusStorage,
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.doc', '.docx'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type. Only PDF, DOC, and DOCX files are allowed.'));
  }
}).single('syllabus');

router.use((req, res, next) => {
  if (!req.session || req.session.role !== 'admin') return res.status(403).send('Forbidden');
  next();
});

router.get('/manual', (req, res) => {
  const mdPath = path.join(__dirname, '..', 'docs', 'admin-manual.md');
  try {
    const markdown = fs.readFileSync(mdPath, 'utf-8');
    const { marked } = require('marked');
    marked.setOptions({
      headerIds: true,
      mangle: false
    });
    const htmlContent = marked.parse(markdown);
    const fullHtml = `
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Administrator Manual - MDTS LMS</title>
  <link rel="stylesheet" href="/styles.css">
  <style>
    body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif; line-height: 1.6; max-width: 900px; margin: 40px auto; padding: 0 20px; }
    h1 { color: #2c3e50; border-bottom: 3px solid #3498db; padding-bottom: 10px; }
    h2 { color: #34495e; margin-top: 30px; border-bottom: 2px solid #ecf0f1; padding-bottom: 8px; }
    h3 { color: #555; margin-top: 20px; }
    img { max-width: 100%; height: auto; border: 1px solid #ddd; border-radius: 4px; margin: 15px 0; }
    code { background: #f4f4f4; padding: 2px 6px; border-radius: 3px; font-family: 'Courier New', monospace; }
    pre { background: #f4f4f4; padding: 15px; border-radius: 5px; overflow-x: auto; }
    ul, ol { margin: 10px 0; padding-left: 30px; }
    li { margin: 5px 0; }
    blockquote { border-left: 4px solid #3498db; padding-left: 15px; color: #666; margin: 15px 0; }
    a { color: #3498db; text-decoration: none; }
    a:hover { text-decoration: underline; }
    .back-link { margin-bottom: 20px; }
    .back-link a { display: inline-block; padding: 8px 15px; background: #3498db; color: white; border-radius: 4px; }
    .back-link a:hover { background: #2980b9; text-decoration: none; }
  </style>
</head>
<body>
  <div class="back-link">
    <a href="/admin">← Back to Dashboard</a>
  </div>
  ${htmlContent}
</body>
</html>
    `;
    res.send(fullHtml);
  } catch (err) {
    console.error('Failed to load admin manual', err);
    res.status(404).send('User guide not found');
  }
});

router.get('/view-as/:role', async (req, res) => {
  if (!req.session || req.session.role !== 'admin') return res.status(403).send('Forbidden');
  const role = req.params.role;

  if (role === 'teacher') {
    const requestedId = Number(req.query.userId);
    let teacher = null;
    if (requestedId) {
      const candidate = await userModel.findById(requestedId);
      if (candidate && candidate.role === 'teacher') teacher = candidate;
    }
    if (!teacher) {
      const teachers = await userModel.getByRole('teacher');
      teacher = teachers[0] || null;
    }
    if (!teacher) {
      return res.render('admin_preview_unavailable', { role: 'teacher' });
    }

    const classes = await classModel.byTeacher(teacher.id);
    const users = await userModel.getAll();
    const students = users.filter(u => u.role === 'student');
    const pendingStudents = students.filter(u => u.status === 'pending' || !u.status);
    const approvedStudents = students.filter(u => u.status === 'approved');
    const classSizes = classes.map(c => ({ name: c.name, size: (c.studentIds || []).length }));
    const weekly = { Monday:[], Tuesday:[], Wednesday:[], Thursday:[], Friday:[], Saturday:[], Sunday:[] };
    classes.forEach(k => (k.schedule || []).forEach(s => {
      if (weekly[s.day]) weekly[s.day].push({ className: k.name, start: s.start, end: s.end });
    }));

    return res.render('teacher_dashboard', {
      teacher,
      classes,
      weekly,
      pendingCount: pendingStudents.length,
      approvedCount: approvedStudents.length,
      classSizes,
      previewContext: {
        role: 'teacher',
        name: teacher.name || teacher.username || `Teacher #${teacher.id}`,
        returnUrl: '/admin',
        switchUrl: '/admin/view-as/teacher'
      }
    });
  }

  if (role === 'student') {
    const requestedId = Number(req.query.userId);
    let student = null;
    if (requestedId) {
      const candidate = await userModel.findById(requestedId);
      if (candidate && candidate.role === 'student') student = candidate;
    }
    if (!student) {
      const students = await userModel.getByRole('student');
      student = students[0] || null;
    }
    if (!student) {
      return res.render('admin_preview_unavailable', { role: 'student' });
    }

    const classes = await classModel.getAllClasses();
    const myClasses = classes.filter(k => (k.studentIds || []).includes(student.id));

    const events = [];
    const now = new Date();
    const threeWeeksFromNow = new Date();
    threeWeeksFromNow.setDate(now.getDate() + 21);

    const latestGrades = [];
    const currentGrades = {};
    const tasks = [];

    myClasses.forEach(klass => {
      const studentGrades = (klass.grades || []).filter(
        g => g.studentId === student.id && g.testId !== undefined
      );
      if (studentGrades.length) {
        const avg = Math.round(studentGrades.reduce((sum, g) => sum + (g.score || 0), 0) / studentGrades.length);
        currentGrades[klass.id] = avg;
        studentGrades.sort((a, b) => new Date(b.gradedAt || 0) - new Date(a.gradedAt || 0));
        const latest = studentGrades[0];
        const testInfo = (klass.tests || []).find(t => t.id === latest.testId);
        latestGrades.push({
          className: klass.name,
          testTitle: testInfo ? testInfo.title : `Test ${latest.testId}`,
          score: latest.score,
          gradedAt: latest.gradedAt
        });
      }

      (klass.schedule || []).forEach(s => {
        const dayIndex = ['Sunday','Monday','Tuesday','Wednesday','Thursday','Friday','Saturday'].indexOf(s.day);
        if (dayIndex === -1) return;
        let date = new Date(now);
        while (date <= threeWeeksFromNow) {
          if (date.getDay() === dayIndex) {
            const [sh, sm] = String(s.start || '0:0').split(':').map(Number);
            events.push({
              title: `${klass.name} (Class)`,
              start: new Date(date.getFullYear(), date.getMonth(), date.getDate(), sh, sm || 0),
              url: `/student/classes/${klass.id}`
            });
          }
          date.setDate(date.getDate() + 1);
        }
      });

      (klass.tests || []).forEach(test => {
        if (!test.dueDate) return;
        const due = new Date(test.dueDate);
        if (due >= now && due <= threeWeeksFromNow) {
          events.push({
            title: `${klass.name} - ${test.title} (Due)`,
            start: due,
            url: `/student/classes/${klass.id}/tests/${test.id}`
          });
        }
        const graded = (klass.grades || []).some(g => g.studentId === student.id && g.testId === test.id);
        if (due >= now && !graded) {
          tasks.push({
            className: klass.name,
            title: test.title,
            dueDate: test.dueDate,
            classId: klass.id,
            testId: test.id
          });
        }
      });
    });

    let pendingDocuments = [];
    try {
      pendingDocuments = await signatureDocumentModel.getPendingDocumentsForUser(student.id);
    } catch (err) {
      console.error('Preview pending docs error', err);
    }

    let incompleteTest = null;
    try {
      const incompleteTests = await testProgressModel.getIncompleteTests(student.id);
      if (incompleteTests.length > 0) {
        const progress = incompleteTests[0];
        const klass = myClasses.find(c => c.id === progress.class_id);
        if (klass) {
          const test = (klass.tests || []).find(t => t.id === progress.test_id);
          if (test) {
            incompleteTest = {
              className: klass.name,
              testTitle: test.title,
              classId: progress.class_id,
              testId: progress.test_id,
              lastSaved: progress.last_saved_at
            };
          }
        }
      }
    } catch (err) {
      console.error('Preview incomplete tests error', err);
    }

    return res.render('student_dashboard', {
      user: student,
      classes: myClasses,
      events,
      latestGrades,
      tasks,
      currentGrades,
      pendingDocuments,
      incompleteTest,
      previewContext: {
        role: 'student',
        name: student.name || student.username || `Student #${student.id}`,
        returnUrl: '/admin',
        switchUrl: '/admin/view-as/student'
      }
    });
  }

  return res.redirect('/admin');
});

router.use(express.json());

// Cache report chart metadata (tables/columns) for quicker loads
let chartMetaCache = { tables: {}, fetchedAt: 0 };
const CHART_META_TTL_MS = 5 * 60 * 1000;

async function loadChartMeta() {
  const now = Date.now();
  if (chartMetaCache.fetchedAt && now - chartMetaCache.fetchedAt < CHART_META_TTL_MS) {
    return chartMetaCache.tables;
  }

  const [tables] = await db.query("SHOW TABLES LIKE 'mdtslms\\_%'");
  const tableNames = tables.map(t => Object.values(t)[0]);

  const entries = await Promise.all(tableNames.map(async (name) => {
    const [cols] = await db.query('SHOW COLUMNS FROM ??', [name]);
    return [name, cols.map(c => c.Field)];
  }));

  chartMetaCache = { tables: Object.fromEntries(entries), fetchedAt: now };
  return chartMetaCache.tables;
}

router.get('/chart/meta', async (_req, res) => {
  try {
    const tables = await loadChartMeta();
    res.json({ tables });
  } catch (e) {
    console.error('chart/meta error', e);
    res.status(500).json({ tables: {} });
  }
});

// Report Charts page (Chart.js builder)
router.get('/report-charts', async (req, res) => {
  res.render('admin_report_charts', { user: req.session.user });
});

// Data endpoint for ad-hoc chart queries
router.post('/charts/data', async (req, res) => {
  try {
    const { table, x, yColumns, agg, limit, bucketDate, dateRange, dateColumn } = req.body || {};
    const safeLimit = Math.min(Math.max(parseInt(limit || 100, 10), 1), 1000);
    if (!table || !x) return res.status(400).json({ error: 'Missing table or x column' });

    // Load schema and validate identifiers (prefixed tables only)
    const tables = await loadChartMeta();
    const allowedTables = Object.keys(tables);
    if (!allowedTables.includes(table)) return res.status(400).json({ error: 'Invalid table' });
    const colNames = tables[table] || [];
    if (!colNames.includes(x)) return res.status(400).json({ error: 'Invalid x column' });

    const yList = Array.isArray(yColumns) ? yColumns.filter(c => colNames.includes(c)) : [];
    const useCount = !yList.length || String(agg || '').toLowerCase() === 'count';

    // Build SQL
    const selectX = bucketDate ? 'DATE(??)' : '??';
    // Optional WHERE by date range
    let where = '';
    let whereParams = [];
    const dr = String(dateRange || '').toLowerCase();
    const dc = (dateColumn || x);
    const hasDateFilter = ['week','month','quarter','year','30d','90d','365d'].includes(dr);
    if (hasDateFilter) {
      if (!colNames.includes(dc)) return res.status(400).json({ error: 'Invalid date column' });
      const now = new Date();
      const days = dr === 'week' ? 7 : dr === 'month' ? 30 : dr === 'quarter' ? 90 : 365;
      const start = new Date(now.getTime() - days*24*60*60*1000);
      const fmt = (d) => d.toISOString().slice(0,19).replace('T',' ');
      where = 'WHERE ?? BETWEEN ? AND ?';
      whereParams = [dc, fmt(start), fmt(now)];
    }
    let sql, params;
    if (useCount) {
      sql = `SELECT ${selectX} as label, COUNT(*) as value FROM ?? ${where} GROUP BY ${selectX} ORDER BY label LIMIT ?`;
      params = [x, table, ...whereParams, x, safeLimit];
    } else {
      const fn = ['sum', 'avg'].includes(String(agg).toLowerCase()) ? String(agg).toUpperCase() : 'SUM';
      const selects = yList.map(() => `${fn}(??) as ??`).join(', ');
      sql = `SELECT ${selectX} as label, ${selects} FROM ?? ${where} GROUP BY ${selectX} ORDER BY label LIMIT ?`;
      params = [x, ...yList.flatMap(c => [c, c]), table, ...whereParams, x, safeLimit];
    }
    const [rows] = await db.query(sql, params);
    return res.json({ rows, x, yColumns: yList, agg: useCount ? 'count' : (String(agg).toLowerCase() || 'sum') });
  } catch (e) {
    console.error('charts/data error', e);
    return res.status(500).json({ error: 'Query failed' });
  }
});

// Save a chart configuration to the admin's dashboard (max 8)
router.post('/charts/save', async (req, res) => {
  try {
    const userId = req.session.user && req.session.user.id;
    if (!userId) return res.status(401).json({ error: 'Not authenticated' });
    const admin = await userModel.findById(userId);
    if (!admin || admin.role !== 'admin') return res.status(403).json({ error: 'Forbidden' });

    const cfg = req.body || {};
    const required = ['title','table','x','chartType'];
    for (const k of required){ if (!cfg[k]) return res.status(400).json({ error: `Missing ${k}` }); }
    cfg.yColumns = Array.isArray(cfg.yColumns) ? cfg.yColumns : [];
    cfg.limit = Math.min(Math.max(parseInt(cfg.limit || 100, 10), 1), 1000);
    cfg.bucketDate = !!cfg.bucketDate;
    cfg.radius = Math.min(Math.max(parseInt(cfg.radius || 10, 10), 0), 50);
    cfg.dateRange = cfg.dateRange || '';
    cfg.agg = (cfg.agg || 'count');
    cfg.id = Date.now();
    cfg.createdAt = new Date().toISOString();

    const profile = admin.profile || {};
    const arr = Array.isArray(profile.dashboardCharts) ? profile.dashboardCharts : [];
    if (arr.length >= 8) return res.status(400).json({ error: 'Maximum of 8 charts reached' });
    arr.push(cfg);
    await userModel.updateProfile(userId, { dashboardCharts: arr });
    return res.json({ ok: true, chart: cfg });
  } catch (e) {
    console.error('charts/save error', e);
    return res.status(500).json({ error: 'Failed to save chart' });
  }
});

// List saved charts for current admin
router.get('/charts/list', async (req, res) => {
  try {
    const userId = req.session.user && req.session.user.id;
    if (!userId) return res.status(401).json({ error: 'Not authenticated' });
    const admin = await userModel.findById(userId);
    const charts = (admin && admin.profile && Array.isArray(admin.profile.dashboardCharts)) ? admin.profile.dashboardCharts : [];
    return res.json({ charts });
  } catch (e) {
    console.error('charts/list error', e);
    return res.status(500).json({ error: 'Failed to load charts' });
  }
});

// Reorder dashboard charts for current admin
router.post('/charts/reorder', async (req, res) => {
  try {
    const userId = req.session.user && req.session.user.id;
    if (!userId) return res.status(401).json({ error: 'Not authenticated' });
    const admin = await userModel.findById(userId);
    if (!admin || admin.role !== 'admin') return res.status(403).json({ error: 'Forbidden' });
    const ids = Array.isArray(req.body.ids) ? req.body.ids.map(id => parseInt(id, 10)) : [];
    const charts = (admin.profile && Array.isArray(admin.profile.dashboardCharts)) ? admin.profile.dashboardCharts : [];
    const ordered = ids.map(id => charts.find(c => c.id === id)).filter(Boolean);
    if (ordered.length !== charts.length) return res.status(400).json({ error: 'Invalid ids' });
    await userModel.updateProfile(userId, { dashboardCharts: ordered });
    return res.json({ ok: true });
  } catch (e) {
    console.error('charts/reorder error', e);
    return res.status(500).json({ error: 'Failed to reorder charts' });
  }
});

// Delete a dashboard chart
router.delete('/charts/:id', async (req, res) => {
  try {
    const userId = req.session.user && req.session.user.id;
    if (!userId) return res.status(401).json({ error: 'Not authenticated' });
    const admin = await userModel.findById(userId);
    if (!admin || admin.role !== 'admin') return res.status(403).json({ error: 'Forbidden' });
    const id = parseInt(req.params.id, 10);
    const charts = (admin.profile && Array.isArray(admin.profile.dashboardCharts)) ? admin.profile.dashboardCharts : [];
    const filtered = charts.filter(c => c.id !== id);
    await userModel.updateProfile(userId, { dashboardCharts: filtered });
    return res.json({ ok: true });
  } catch (e) {
    console.error('charts/delete error', e);
    return res.status(500).json({ error: 'Failed to delete chart' });
  }
});

// Download CSV template for test uploads
router.get('/tests/template.csv', (_req, res) => {
  const header = [
    'Question','Answer','Explanation','Picture',
    'OptionA','OptionB','OptionC','OptionD','OptionE','OptionF','OptionG',
    'Test','Content Type','Title','Item Type','Path'
  ].join(',') + '\n';
  res.setHeader('Content-Type', 'text/csv; charset=utf-8');
  res.setHeader('Content-Disposition', 'attachment; filename="test_template.csv"');
  res.send(header);
});

// ----- dropdown options -----
router.get('/dropdowns', (req, res) => {
  const data = dropdowns.getAll();
  res.render('admin_dropdowns', { user: req.session.user, dropdowns: data, saved: req.query.saved });
});

router.post('/dropdowns/add', (req, res) => {
  const { type, value } = req.body;
  dropdowns.add(type, value && value.trim());
  res.redirect('/admin/dropdowns?saved=1');
});

router.post('/dropdowns/delete', (req, res) => {
  const { type, value } = req.body;
  dropdowns.remove(type, value);
  res.redirect('/admin/dropdowns?saved=1');
});

// ----- Financials / Course Pricing -----
router.get('/financials', async (req, res) => {
  try {
    // Sync courses from dropdowns first
    const dropdownData = dropdowns.getAll();
    await courseModel.syncFromDropdowns(dropdownData.courses || []);
    
    // Get all courses
    const courses = await courseModel.getAllCourses();
    
    res.render('admin_financials', {
      user: req.session.user,
      courses,
      saved: req.query.saved,
      branding
    });
  } catch (err) {
    console.error('Error loading financials:', err);
    res.status(500).send('Error loading financials page');
  }
});

router.post('/financials/update/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const { program_hours, program_cost, credential_awarded } = req.body;
    
    await courseModel.updateCourse(id, {
      program_hours: parseInt(program_hours) || 0,
      program_cost: parseFloat(program_cost) || 0,
      credential_awarded: credential_awarded || 'Certificate'
    });
    
    res.json({ ok: true });
  } catch (err) {
    console.error('Error updating course:', err);
    res.status(500).json({ ok: false, error: 'Failed to update course' });
  }
});

router.post('/financials/bulk-update', async (req, res) => {
  try {
    const { courses } = req.body;
    
    if (!Array.isArray(courses)) {
      return res.status(400).json({ ok: false, error: 'Invalid data format' });
    }
    
    for (const course of courses) {
      if (course.id) {
        await courseModel.updateCourse(course.id, {
          program_hours: parseInt(course.program_hours) || 0,
          program_cost: parseFloat(course.program_cost) || 0,
          credential_awarded: course.credential_awarded || 'Certificate'
        });
      }
    }
    
    res.json({ ok: true });
  } catch (err) {
    console.error('Error bulk updating courses:', err);
    res.status(500).json({ ok: false, error: 'Failed to update courses' });
  }
});

// ----- Financial Forecasting -----
router.get('/financials/forecast', async (req, res) => {
  try {
    const courses = await courseModel.getAllCourses();
    const classModel = require('../models/classModel');
    
    // Create price lookup map with fuzzy matching helper
    const priceMap = {};
    courses.forEach(course => {
      priceMap[course.program_name] = parseFloat(course.program_cost) || 0;
    });
    
    // Fuzzy match helper function
    const fuzzyMatchCourse = (courseName) => {
      if (!courseName) return null;
      const cleanName = courseName.trim().toLowerCase();
      
      // Try exact match first
      const exactMatch = courses.find(c => c.program_name.toLowerCase() === cleanName);
      if (exactMatch) return exactMatch;
      
      // Try partial match
      const partialMatch = courses.find(c => 
        cleanName.includes(c.program_name.toLowerCase()) || 
        c.program_name.toLowerCase().includes(cleanName)
      );
      if (partialMatch) return partialMatch;
      
      // Try fuzzy match (remove common words and special chars)
      const normalize = (str) => str.toLowerCase()
        .replace(/\+/g, 'plus')
        .replace(/[^a-z0-9]/g, '')
        .replace(/\s+/g, '');
      
      const normalizedSearch = normalize(courseName);
      const fuzzyMatch = courses.find(c => {
        const normalized = normalize(c.program_name);
        return normalized === normalizedSearch || 
               normalized.includes(normalizedSearch) ||
               normalizedSearch.includes(normalized);
      });
      
      return fuzzyMatch;
    };
    
    // Parse comma-separated courses and calculate total price
    const parseMultipleCourses = (courseString) => {
      if (!courseString) return { total: 0, courses: [], matched: [] };
      
      const courseNames = courseString.split(',').map(c => c.trim()).filter(c => c);
      let total = 0;
      const matched = [];
      
      courseNames.forEach(courseName => {
        const course = fuzzyMatchCourse(courseName);
        if (course) {
          total += parseFloat(course.program_cost) || 0;
          matched.push({ name: course.program_name, price: parseFloat(course.program_cost) || 0 });
        } else {
          // Try direct lookup
          const directPrice = priceMap[courseName] || 0;
          if (directPrice > 0) {
            total += directPrice;
            matched.push({ name: courseName, price: directPrice });
          }
        }
      });
      
      return { total, courses: courseNames, matched };
    };
    
    // Get all classes to check enrollments
    const allClasses = await classModel.getAllClasses();
    
    // Helper function to check if name should be auto-ignored
    const shouldAutoIgnore = (name) => {
      if (!name) return false;
      const normalized = name.toLowerCase().replace(/[^a-z]/g, '');
      
      // List of names to auto-ignore
      const ignoreNames = [
        'carltonpayneiii',
        'carltonpayne',
        'rachelsnyder',
        'gabriellepayne',
        'olga'
      ];
      
      return ignoreNames.some(ignoreName => 
        normalized.includes(ignoreName) || ignoreName.includes(normalized)
      );
    };
    
    // Get interested students (pre-registered) - DON'T filter, just mark
    let interested = await preRegModel.getAll();
    interested = interested.map(student => {
      const email = (student.email || '').toLowerCase();
      const name = (student.name || '').toLowerCase();
      const isTestAccount = email.includes('test') || name.includes('test') || name.includes('demo');
      const isAutoIgnored = shouldAutoIgnore(name);
      
      return {
        ...student,
        excludeFromForecast: student.excludeFromForecast || isTestAccount || isAutoIgnored,
        isTestAccount
      };
    });
    
    // Get registered students (pending approval)
    const [registeredRows] = await db.query(`
      SELECT u.id, u.name, u.email, u.createdAt, u.profile
      FROM mdtslms_users u
      WHERE u.status = 'pending'
      ORDER BY u.createdAt DESC
    `);
    
    // Get accepted students
    const [acceptedRows] = await db.query(`
      SELECT u.id, u.name, u.email, u.createdAt, u.profile, u.acceptedAt
      FROM mdtslms_users u
      WHERE u.status = 'approved'
      ORDER BY u.acceptedAt DESC
    `);
    
    // Parse profile JSON and extract course - DON'T filter
    const parseStudents = (rows) => {
      return rows.map(row => {
        let profile = {};
        if (row.profile && typeof row.profile === 'string') {
          try {
            profile = JSON.parse(row.profile);
          } catch (e) {
            console.error('Error parsing profile:', e);
          }
        } else if (row.profile && typeof row.profile === 'object') {
          profile = row.profile;
        }
        
        const email = (row.email || '').toLowerCase();
        const name = (row.name || '').toLowerCase();
        const isTestAccount = email.includes('test') || name.includes('test') || name.includes('demo');
        const isAutoIgnored = shouldAutoIgnore(name);
        
        return {
          ...row,
          course: profile.course || '',
          excludeFromForecast: profile.excludeFromForecast || isTestAccount || isAutoIgnored,
          isTestAccount
        };
      });
    };
    
    const registeredParsed = parseStudents(registeredRows);
    const acceptedParsed = parseStudents(acceptedRows);
    
    // Helper function to get all enrolled classes for a student
    const getStudentClasses = (studentId) => {
      return allClasses.filter(klass => 
        (klass.studentIds || []).includes(studentId)
      );
    };
    
    // Helper function to calculate total price for multiple courses
    const calculateStudentPrice = (courseName, studentId, isAccepted) => {
      // For pre-registered and pending students, parse comma-separated courses
      if (!isAccepted) {
        const result = parseMultipleCourses(courseName);
        return { 
          price: result.total, 
          courses: result.matched.map(m => m.name), 
          classNames: [],
          originalCourses: result.courses
        };
      }
      
      // For accepted students, check all enrolled classes
      const enrolledClasses = getStudentClasses(studentId);
      
      if (enrolledClasses.length === 0) {
        // No classes enrolled, parse comma-separated courses from profile
        const result = parseMultipleCourses(courseName);
        return { 
          price: result.total, 
          courses: result.matched.map(m => m.name), 
          classNames: [],
          originalCourses: result.courses
        };
      }
      
      // For each enrolled class, try to find matching course pricing with fuzzy matching
      let totalPrice = 0;
      const enrolledCourses = [];
      const classNames = [];
      
      enrolledClasses.forEach(klass => {
        classNames.push(klass.name);
        
        // Use fuzzy matching
        let matchingCourse = fuzzyMatchCourse(klass.name);
        
        // Try shortName if main name didn't match
        if (!matchingCourse && klass.shortName) {
          matchingCourse = fuzzyMatchCourse(klass.shortName);
        }
        
        // Try description if still no match
        if (!matchingCourse && klass.description) {
          matchingCourse = fuzzyMatchCourse(klass.description);
        }
        
        if (matchingCourse) {
          const price = parseFloat(matchingCourse.program_cost) || 0;
          totalPrice += price;
          enrolledCourses.push(matchingCourse.program_name);
        } else {
          // Fallback: try direct price map lookup
          const fallbackPrice = priceMap[klass.name] || priceMap[klass.shortName] || 0;
          if (fallbackPrice > 0) {
            totalPrice += fallbackPrice;
            enrolledCourses.push(klass.name);
          }
        }
      });
      
      // If still no courses found, fall back to profile course with comma parsing
      if (totalPrice === 0 && courseName) {
        const result = parseMultipleCourses(courseName);
        totalPrice = result.total;
        enrolledCourses.push(...result.matched.map(m => m.name));
      }
      
      return { price: totalPrice, courses: enrolledCourses, classNames: classNames, originalCourses: [] };
    };
    
    // Calculate forecasts
    const calculateForecast = (students, categoryName, isAccepted = false) => {
      const studentData = students.map(student => {
        const courseName = student.course || '';
        const createdDate = new Date(student.createdAt || student.acceptedAt);
        
        let price = 0;
        let coursesList = [courseName];
        let classNamesList = [];
        let originalCourses = [];
        
        if (isAccepted) {
          const result = calculateStudentPrice(courseName, student.id, true);
          price = result.price;
          coursesList = result.courses.length > 0 ? result.courses : [courseName];
          classNamesList = result.classNames || [];
          originalCourses = result.originalCourses || [];
        } else {
          const result = calculateStudentPrice(courseName, student.id, false);
          price = result.price;
          coursesList = result.courses.length > 0 ? result.courses : [courseName];
          originalCourses = result.originalCourses || [];
        }
        
        // If excluded, set price to 0 for calculations but keep course info
        const effectivePrice = student.excludeFromForecast ? 0 : price;
        
        return {
          id: student.id,
          name: student.name,
          email: student.email,
          course: courseName,
          courses: coursesList,
          classNames: classNamesList,
          originalCourses: originalCourses,
          price: effectivePrice,
          actualPrice: price, // Keep actual price for display
          createdAt: createdDate,
          status: categoryName,
          excludeFromForecast: student.excludeFromForecast || false,
          isTestAccount: student.isTestAccount || false
        };
      });
      
      // Calculate totals only from non-excluded students
      const activeStudents = studentData.filter(s => !s.excludeFromForecast);
      const total = activeStudents.reduce((sum, s) => sum + s.price, 0);
      
      // Calculate time-based projections (only non-excluded)
      const now = new Date();
      const oneWeekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
      const oneMonthAgo = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000);
      const oneYearAgo = new Date(now.getTime() - 365 * 24 * 60 * 60 * 1000);
      
      const weekRevenue = activeStudents.filter(s => s.createdAt >= oneWeekAgo).reduce((sum, s) => sum + s.price, 0);
      const monthRevenue = activeStudents.filter(s => s.createdAt >= oneMonthAgo).reduce((sum, s) => sum + s.price, 0);
      const yearRevenue = activeStudents.filter(s => s.createdAt >= oneYearAgo).reduce((sum, s) => sum + s.price, 0);
      
      return {
        category: categoryName,
        students: studentData, // Include all students (excluded will be greyed out in UI)
        count: activeStudents.length, // Only count active students
        total: total,
        week: weekRevenue,
        month: monthRevenue,
        year: yearRevenue
      };
    };
    
    const interestedForecast = calculateForecast(interested, 'Interested', false);
    const registeredForecast = calculateForecast(registeredParsed, 'Registered', false);
    const acceptedForecast = calculateForecast(acceptedParsed, 'Accepted', true);
    
    // Calculate overall totals
    const overallTotal = interestedForecast.total + registeredForecast.total + acceptedForecast.total;
    const overallWeek = interestedForecast.week + registeredForecast.week + acceptedForecast.week;
    const overallMonth = interestedForecast.month + registeredForecast.month + acceptedForecast.month;
    const overallYear = interestedForecast.year + registeredForecast.year + acceptedForecast.year;
    
    // Calculate 12-month trend (monthly breakdown)
    const now = new Date();
    const monthlyTrend = [];
    const allActiveStudents = [
      ...interestedForecast.students.filter(s => !s.excludeFromForecast),
      ...registeredForecast.students.filter(s => !s.excludeFromForecast),
      ...acceptedForecast.students.filter(s => !s.excludeFromForecast)
    ];
    
    for (let i = 11; i >= 0; i--) {
      const monthDate = new Date(now.getFullYear(), now.getMonth() - i, 1);
      const monthStart = new Date(monthDate.getFullYear(), monthDate.getMonth(), 1);
      const monthEnd = new Date(monthDate.getFullYear(), monthDate.getMonth() + 1, 0, 23, 59, 59);
      
      const monthRevenue = allActiveStudents.filter(s => {
        const createdAt = new Date(s.createdAt);
        return createdAt >= monthStart && createdAt <= monthEnd;
      }).reduce((sum, s) => sum + s.price, 0);
      
      const monthCount = allActiveStudents.filter(s => {
        const createdAt = new Date(s.createdAt);
        return createdAt >= monthStart && createdAt <= monthEnd;
      }).length;
      
      monthlyTrend.push({
        month: monthDate.toLocaleString('en-US', { month: 'short', year: 'numeric' }),
        monthYear: monthDate.toLocaleString('en-US', { month: 'long', year: 'numeric' }),
        revenue: monthRevenue,
        count: monthCount
      });
    }
    
    res.render('admin_financials_forecast', {
      user: req.session.user,
      branding,
      interested: interestedForecast,
      registered: registeredForecast,
      accepted: acceptedForecast,
      overall: {
        total: overallTotal,
        week: overallWeek,
        month: overallMonth,
        year: overallYear,
        count: interestedForecast.count + registeredForecast.count + acceptedForecast.count
      },
      monthlyTrend: monthlyTrend
    });
  } catch (err) {
    console.error('Error loading financial forecast:', err);
    res.status(500).send('Error loading forecast page');
  }
});

// Toggle exclude from forecast flag for a student
router.post('/students/:id/toggle-forecast-exclusion', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    
    // Try finding in users table first (registered/accepted students)
    let student = await userModel.findById(studentId);
    let isPreReg = false;
    
    // If not found, check pre-registration table (interested students)
    if (!student) {
      const preRegRows = await db.query('SELECT * FROM mdtslms_pre_registrations WHERE id = ?', [studentId]);
      if (preRegRows.length > 0) {
        student = preRegRows[0];
        isPreReg = true;
      }
    }
    
    if (!student) {
      // Student not found in either table - just return success to avoid UI errors
      return res.json({ 
        ok: true, 
        excluded: true,
        message: 'Student record not found, but continuing'
      });
    }
    
    // Update in appropriate table
    if (isPreReg) {
      // For pre-registration, ensure column exists and toggle it
      try {
        await db.query('ALTER TABLE mdtslms_pre_registrations ADD COLUMN excludeFromForecast TINYINT(1) DEFAULT 0');
      } catch (e) {
        // Column might already exist, that's okay
      }
      
      // Get current value
      const currentValue = student.excludeFromForecast || 0;
      const newValue = currentValue ? 0 : 1;
      
      await db.query(
        'UPDATE mdtslms_pre_registrations SET excludeFromForecast = ? WHERE id = ?',
        [newValue, studentId]
      );
      
      res.json({ 
        ok: true, 
        excluded: newValue === 1,
        message: newValue === 1 ? 'Student excluded from forecast' : 'Student included in forecast'
      });
    } else {
      // For users table, use profile
      let profile = student.profile || {};
      if (typeof profile === 'string') {
        try {
          profile = JSON.parse(profile);
        } catch (e) {
          profile = {};
        }
      }
      
      // Toggle the flag
      profile.excludeFromForecast = !profile.excludeFromForecast;
      
      await userModel.updateProfile(studentId, profile);
      
      res.json({ 
        ok: true, 
        excluded: profile.excludeFromForecast,
        message: profile.excludeFromForecast ? 'Student excluded from forecast' : 'Student included in forecast'
      });
    }
  } catch (err) {
    console.error('Error toggling forecast exclusion:', err);
    res.status(500).json({ ok: false, error: 'Failed to update student' });
  }
});

router.get('/branding', (req, res) => {
  res.render('admin_branding', {
    user: req.session.user,
    branding,
    saved: req.query.saved
  });
});

router.post('/branding', brandingUpload.fields([{ name: 'primaryLogo', maxCount: 1 }, { name: 'directorSignature', maxCount: 1 }]), (req, res) => {
  const { primaryColor, secondaryColor, font } = req.body;
  if (req.files && Array.isArray(req.files.primaryLogo) && req.files.primaryLogo[0]) {
    const f = req.files.primaryLogo[0];
    branding.primaryLogo = '/uploads/' + f.filename;
  }
  if (req.files && Array.isArray(req.files.directorSignature) && req.files.directorSignature[0]) {
    const f = req.files.directorSignature[0];
    branding.directorSignature = '/uploads/' + f.filename;
  }
    // Prefer drawn signature if provided
  if (req.body && req.body.directorSignatureData && /^data:image\//.test(req.body.directorSignatureData)) {
    branding.directorSignature = req.body.directorSignatureData;
  }
  if (primaryColor) branding.primaryColor = primaryColor;
  if (secondaryColor) branding.secondaryColor = secondaryColor;
  if (font) {
    branding.font = font;
    // Map to Google Fonts import if recognized from our list
    const fontMap = {
      'Inter':'Inter:wght@400;600;700', 'Roboto':'Roboto:wght@400;500;700', 'Open Sans':'Open+Sans:wght@400;600;700',
      'Poppins':'Poppins:wght@400;600;700','Lato':'Lato:wght@400;700','Nunito':'Nunito:wght@400;700','Montserrat':'Montserrat:wght@400;700',
      'Source Sans Pro':'Source+Sans+Pro:wght@400;700','Work Sans':'Work+Sans:wght@400;700','Rubik':'Rubik:wght@400;700','Mulish':'Mulish:wght@400;700',
      'Raleway':'Raleway:wght@400;700','Titillium Web':'Titillium+Web:wght@400;700','Urbanist':'Urbanist:wght@400;700','PT Sans':'PT+Sans:wght@400;700',
      'Quicksand':'Quicksand:wght@400;700','Heebo':'Heebo:wght@400;700','Manrope':'Manrope:wght@400;700','Merriweather':'Merriweather:wght@400;700',
      'IBM Plex Sans':'IBM+Plex+Sans:wght@400;700'
    };
    branding.fontImport = fontMap[font] || '';
  }
  fs.writeFileSync(brandingPath, JSON.stringify(branding, null, 2));
  req.app.locals.branding = branding;
  res.redirect('/admin/branding?saved=1');
});

router.get('/email-templates', (req, res) => {
  emailTemplates.load();
  const key = req.query.key || Object.keys(emailTemplates.templates)[0];
  const template = key ? emailTemplates.templates[key] : { subject: '', html: '' };
  res.render('admin_email_templates', {
    user: req.session.user,
    templates: emailTemplates.templates,
    selectedKey: key,
    template,
    saved: req.query.saved
  });
});

router.post('/email-templates/save', (req, res) => {
  const { key, subject, html } = req.body;
  if (key) emailTemplates.saveTemplate(key, { subject, html });
  res.redirect(`/admin/email-templates?key=${encodeURIComponent(key)}&saved=1`);
});

router.post('/email-templates/ai', async (req, res) => {
  const { prompt } = req.body || {};
  const apiKey = process.env.OPENAI_API_KEY;
  if (!apiKey) return res.status(400).json({ error: 'Missing API key' });
  try {
    const response = await fetch('https://api.openai.com/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${apiKey}`
      },
      body: JSON.stringify({
        model: 'gpt-3.5-turbo',
        messages: [
          { role: 'system', content: 'You generate personalized HTML email templates that use {{name}} placeholders.' },
          { role: 'user', content: prompt }
        ]
      })
    });
    const data = await response.json();
    const html = data.choices?.[0]?.message?.content?.trim() || '';
    res.json({ html });
  } catch (e) {
    console.error('AI error', e);
    res.status(500).json({ error: 'AI request failed' });
  }
});

router.get('/drip-campaigns', (req, res) => {
  const campaigns = dripCampaign.loadCampaigns();
  const historyEmail = req.query.email;
  const historyCampaign = historyEmail ? campaigns.find(c => c.email === historyEmail) : null;
  res.render('admin_drip_campaigns', {
    user: req.session.user,
    campaigns,
    created: req.query.created,
    historyEmail,
    historyCampaign
  });
});

// ===== Interactions / Leads Tracking =====
router.get('/interactions', async (req, res) => {
  const leads = await leadModel.getAll();
  res.render('admin_interactions', { user: req.session.user, leads, saved: req.query.saved, created: req.query.created });
});

router.post('/interactions/add', async (req, res) => {
  const { name, email, phone, interestPercent, source, note } = req.body;
  try {
    const lead = await leadModel.upsertByEmail({ name, email, phone, interestPercent: Number(interestPercent || 0), source: source || 'admin' });
    if (lead?.id && note) {
      await leadModel.addContact(lead.id, { direction: 'outbound', method: 'other', note });
    }
    res.redirect('/admin/interactions?created=1');
  } catch (e) {
    console.error('Add lead error', e);
    res.redirect('/admin/interactions?saved=0');
  }
});

router.post('/interactions/:id/contact', async (req, res) => {
  const id = Number(req.params.id);
  const { direction, method, note, interestPercent } = req.body;
  try {
    if (interestPercent !== undefined && interestPercent !== '') {
      await leadModel.updateInterest(id, Number(interestPercent));
    }
    await leadModel.addContact(id, { direction: direction || 'outbound', method: method || 'other', note: note || '' });
    res.redirect('/admin/interactions?saved=1');
  } catch (e) {
    console.error('Log contact error', e);
    res.redirect('/admin/interactions?saved=0');
  }
});

router.delete('/interactions/:id', async (req, res) => {
  try {
    const id = Number(req.params.id);
    if (!Number.isInteger(id)) return res.status(400).json({ error: 'invalid id' });
    const lead = await leadModel.getById(id);
    if (!lead) return res.status(404).json({ error: 'not found' });
    await leadModel.deleteLead(id);
    res.json({ ok: true });
  } catch (e) {
    console.error('delete lead error', e);
    res.status(500).json({ error: 'failed' });
  }
});

router.delete('/interactions/contacts/:contactId', async (req, res) => {
  try {
    const contactId = Number(req.params.contactId);
    if (!Number.isInteger(contactId)) return res.status(400).json({ error: 'invalid id' });
    const ok = await leadModel.deleteContact(contactId);
    if (!ok) return res.status(404).json({ error: 'not found' });
    res.json({ ok: true });
  } catch (e) {
    console.error('delete lead contact error', e);
    res.status(500).json({ error: 'failed' });
  }
});
// Send email to a lead (AJAX from interactions page)
router.post('/interactions/:id/email', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const lead = await leadModel.getById(id);
    if (!lead || !lead.email) return res.status(404).json({ error: 'Lead not found' });
    const { subject, message } = req.body || {};
    if (!subject || !message) return res.status(400).json({ error: 'Missing subject or message' });
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: lead.email,
      subject,
      html: message,
      text: message.replace(/<[^>]*>/g, '')
    });
    try { await leadModel.addContact(id, { direction: 'outbound', method: 'email', note: (subject||'').slice(0,120) }); } catch(_) {}
    return res.json({ ok: true });
  } catch (e) {
    console.error('Lead email error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

// Professional leads report
router.get('/interactions/report', async (req, res) => {
  const leads = await leadModel.getAll();
  res.render('leads_report', { user: req.session.user, leads, generatedAt: new Date() });
});

router.get('/interactions/export', async (_req, res) => {
  const leads = await leadModel.getAll();
  const header = 'Name,Email,Phone,Interest %,Source,Created At,Updated At,Last Contacted,Contact Count\n';
  const rows = leads.map(l => [
    (l.name || '').replace(/,/g, ' '),
    (l.email || '').replace(/,/g, ' '),
    (l.phone || '').replace(/,/g, ' '),
    l.interestPercent || 0,
    (l.source || '').replace(/,/g, ' '),
    l.createdAt ? new Date(l.createdAt).toISOString() : '',
    l.updatedAt ? new Date(l.updatedAt).toISOString() : '',
    l.lastContacted ? new Date(l.lastContacted).toISOString() : '',
    l.contactCount || 0
  ].join(',')).join('\n');
  res.setHeader('Content-Type', 'text/csv');
  res.setHeader('Content-Disposition', 'attachment; filename="interactions.csv"');
  res.send(header + rows + '\n');
});

router.post('/drip-campaigns', (req, res) => {
  const { email, phone, name, segment, program, enrollmentStatus } = req.body;
  if (email) dripCampaign.addCampaign({ email, phone, name, segment, program, enrollmentStatus });
  res.redirect('/admin/drip-campaigns?created=1');
});

router.get('/', async (req, res) => {
  req.time && req.time('Start dashboard');
  
  const users = await userModel.getAll();
  req.time && req.time('getAll users');
  
  const classes = await classModel.getAllClasses();
  req.time && req.time('getAllClasses');
  
  const preregs = await preRegModel.getAll();
  req.time && req.time('getAll preregs');
  
  const teachers = users.filter(u => u.role === 'teacher');
  const students = users.filter(u => u.role === 'student');
  const mapCalendarPerson = (person) => ({
    id: person.id,
    name: person.name || person.username || person.email || `User #${person.id}`,
    email: person.email || ''
  });
  const isActive = (person) => person.active === undefined || person.active === null || !!person.active;
  const calendarAdmins = users
    .filter((u) => u.role === 'admin' && isActive(u))
    .map(mapCalendarPerson);
  const calendarTeachers = teachers
    .filter((t) => isActive(t))
    .map(mapCalendarPerson);
  const pendingStudents = students.filter(u => u.status === 'pending' || !u.status);
  const approvedStudents = students.filter(u => u.status === 'approved');
  // Pre-registered in last 14 days
  const now = new Date();
  const cutoff = new Date(now.getTime() - 14 * 24 * 60 * 60 * 1000);
  const prereg30Count = preregs.filter(p => p.createdAt && new Date(p.createdAt) >= cutoff).length;

  // ----- course counts -----
  const courseCountsMap = students.reduce((acc, s) => {
    const course = (s.profile && s.profile.course) || 'Unknown';
    acc[course] = (acc[course] || 0) + 1;
    return acc;
  }, {});
  const courseCounts = Object.entries(courseCountsMap).map(([course, count]) => ({ course, count }));

  // ----- state counts -----
  const stateCounts = {};
  students.forEach(s => {
    const state = s.profile && s.profile.address && s.profile.address.state;
    if (state) stateCounts[state] = (stateCounts[state] || 0) + 1;
  });
  const studentsByState = Object.entries(stateCounts).map(([state, count]) => ({ state, count }));

  // ----- affiliate counts -----
  const affiliateCounts = students.reduce((acc, s) => {
    const program = (s.profile && s.profile.affiliateProgram) || 'Unspecified';
    acc[program] = (acc[program] || 0) + 1;
    return acc;
  }, {});


const signupsLast7Days  = buildDailySeries(students, 7);    // day-by-day for past 7 days
const signupsLast30Days = buildDailySeries(students, 30);   // optional
const signupsLast365    = buildDailySeries(students, 365);  // optional

  // const sum = a => a.reduce((n, x) => n + x.count, 0);
  // const signupsTotals = {
  //   week:  sum(signupsDailyWeek),
  //   month: sum(signupsDailyMonth),
  //   year:  sum(signupsDailyYear)
  // };

  const classSizes = classes.map(c => ({ name: c.name, size: (c.studentIds || []).length }));

  req.time && req.time('Data processing');

  // Load saved dashboard charts for this admin
  let customCharts = [];
  try {
    const me = await userModel.findById(req.session.user.id);
    if (me && me.profile && Array.isArray(me.profile.dashboardCharts)) customCharts = me.profile.dashboardCharts;
  } catch(_) {}
  
  req.time && req.time('Custom charts');

  // Get pending staff signatures
  let pendingStaffSignatures = [];
  try {
    pendingStaffSignatures = await signatureDocumentModel.getPendingStaffSignatures();
  } catch(err) {
    console.error('Error fetching pending staff signatures:', err);
  }
  
  req.time && req.time('Staff signatures');

  const alumniStudents = students.filter(u => u.status === 'alumni');

  // Load alumni transition timestamp
  let alumniTransitionInfo = null;
  try {
    const fs = require('fs');
    const path = require('path');
    const timestampPath = path.join(__dirname, '../data/alumniTransitionTimestamp.json');
    if (fs.existsSync(timestampPath)) {
      const data = fs.readFileSync(timestampPath, 'utf8');
      alumniTransitionInfo = JSON.parse(data);
      // Calculate days since last transition
      if (alumniTransitionInfo.lastTransitionDate) {
        const lastDate = new Date(alumniTransitionInfo.lastTransitionDate);
        const now = new Date();
        const diffTime = Math.abs(now - lastDate);
        const diffDays = Math.floor(diffTime / (1000 * 60 * 60 * 24));
        alumniTransitionInfo.daysSince = diffDays;
      }
    }
  } catch (err) {
    console.error('Error reading alumni transition timestamp:', err);
  }
  
  req.time && req.time('Before render');

  res.render('admin_dashboard', {
    user: req.session.user,
    classes,
    teachers,
    students,
    prereg30Count,
    pendingCount: pendingStudents.length,
    approvedCount: approvedStudents.length,
    alumniCount: alumniStudents.length,
    classSizes,
    courseCounts,
    studentsByState,
    affiliateCounts,
    signupsLast7Days,
    signupsLast30Days,
    signupsLast365,
    customCharts,
    calendarPeople: { admins: calendarAdmins, teachers: calendarTeachers },
    pendingStaffSignatures,
    alumniTransitionInfo
  });
});

router.get('/announcements', async (req, res) => {
  const announcements = await announcementModel.forAdmin();
  res.render('admin_announcements', { user: req.session.user, announcements });
});

async function renderPending(_req, res) {
  const users = await userModel.getAll();
  const pending = users
    .filter(u => u.role === 'student' && (u.status === 'pending' || !u.status))
    .sort((a, b) => {
      // Sort by appliedAt date, newest first (descending)
      const dateA = a.appliedAt ? new Date(a.appliedAt).getTime() : 0;
      const dateB = b.appliedAt ? new Date(b.appliedAt).getTime() : 0;
      return dateB - dateA;
    });
  
  // Enrich with lead information
  let leadsByEmail = new Map();
  let preRegsByEmail = new Map();
  try {
    const leads = await leadModel.getAll();
    leadsByEmail = new Map(leads.map(l => [l.email?.toLowerCase().trim(), l]));
  } catch (e) { console.error('Load leads for pending failed', e); }
  
  try {
    const preregs = await preRegModel.getAll();
    preRegsByEmail = new Map(preregs.map(p => [p.email?.toLowerCase().trim(), p]));
  } catch (e) { console.error('Load preregs for pending failed', e); }
  
  // Attach lead and pre-reg data to each pending student
  const enrichedPending = pending.map(student => {
    const email = student.email?.toLowerCase().trim();
    const lead = email ? leadsByEmail.get(email) : null;
    const preReg = email ? preRegsByEmail.get(email) : null;
    return {
      ...student,
      lead,
      preReg,
      leadId: lead?.id || null
    };
  });
  
  let adminsForActions = [];
  try {
    const admins = await userModel.listAdmins();
    adminsForActions = admins.map((admin) => ({
      id: admin.id,
      name: admin.name || admin.username || admin.email || `Admin #${admin.id}`,
      email: admin.email || ''
    }));
  } catch (e) {
    console.error('Load admins for pending actions failed', e);
  }
  res.render('admin_pending', { 
    pending: enrichedPending, 
    admins: adminsForActions,
    user: { ...(_req.session.user || {}), role: _req.session.role },
    branding: _req.app.locals.branding || {}
  });
}

function buildDailySeries(people, days, tz = 'America/New_York') {
  const fmtDay = new Intl.DateTimeFormat('en-CA', { timeZone: tz, year: 'numeric', month: '2-digit', day: '2-digit' });
  const fmtDow = new Intl.DateTimeFormat('en-US', { timeZone: tz, weekday: 'short' });

  const todayUTC = new Date();
  const keys = [];
  for (let i = days - 1; i >= 0; i--) {
    const d = new Date(todayUTC);
    d.setUTCDate(d.getUTCDate() - i);
    keys.push(d);
  }

  const counts = Object.fromEntries(keys.map(d => [fmtDay.format(d), 0]));

  for (const s of people) {
    if (!s.appliedAt) continue;
        const dateObj = new Date(s.appliedAt);
    if (isNaN(dateObj)) continue; // skip invalid dates
    const bucket = fmtDay.format(dateObj);
    if (bucket in counts) counts[bucket] += 1;
  }

  return keys.map(d => {
    const date = fmtDay.format(d);           // YYYY-MM-DD
    const dow  = fmtDow.format(d);           // Mon/Tue/...
    return { date, dow, count: counts[date] };
  });
}

router.get('/approvals', renderPending);
router.get('/students/pending', renderPending);

// Alumni students route
router.get('/students/alumni', async (req, res) => {
  try {
    const users = await userModel.getAll();
    const alumni = users.filter(u => u.role === 'student' && u.status === 'alumni');
    
    // Enhance alumni data with last course information
    const classes = await classModel.getAllClasses();
    const enhancedAlumni = alumni.map(student => {
      // Find most recent class they were enrolled in
      let lastCourse = null;
      let lastEndDate = null;
      
      classes.forEach(klass => {
        if ((klass.studentIds || []).includes(student.id)) {
          const endDate = klass.endDate ? new Date(klass.endDate) : null;
          if (endDate && (!lastEndDate || endDate > lastEndDate)) {
            lastEndDate = endDate;
            lastCourse = klass.title || klass.name;
          }
        }
      });
      
      // Use placeholder date of 11/04/2025 for existing alumni without a date
      const defaultDate = '2025-11-04T00:00:00.000Z';
      
      return {
        ...student,
        lastCourseName: lastCourse,
        becameAlumniAt: student.becameAlumniAt || student.profile?.becameAlumniAt || defaultDate
      };
    });
    
    res.render('admin_alumni', { alumni: enhancedAlumni, user: { ...(req.session.user || {}), role: req.session.role } });
  } catch (err) {
    console.error('Alumni route error:', err);
    res.status(500).send('Error loading alumni');
  }
});

// Test Engine Students route - shows all users from mdtsapps_myclass.users
router.get('/students/testengine', async (req, res) => {
  try {
    const users = await userModel.getAllTestEngine();
    res.render('admin_testengine_students', { 
      students: users,
      user: { ...(req.session.user || {}), role: req.session.role },
      branding: req.app.locals.branding || {}
    });
  } catch (err) {
    console.error('Test Engine Students route error:', err);
    res.status(500).send('Error loading test engine students');
  }
});

// Deactivate user route
router.post('/students/testengine/:id/deactivate', async (req, res) => {
  try {
    const userId = Number(req.params.id);
    await userModel.setActiveTestEngine(userId, false);
    res.redirect('/admin/students/testengine');
  } catch (err) {
    console.error('Error deactivating user:', err);
    res.status(500).send('Error deactivating user');
  }
});

// Activate user route
router.post('/students/testengine/:id/activate', async (req, res) => {
  try {
    const userId = Number(req.params.id);
    await userModel.setActiveTestEngine(userId, true);
    res.redirect('/admin/students/testengine');
  } catch (err) {
    console.error('Error activating user:', err);
    res.status(500).send('Error activating user');
  }
});

// Registered students route (pending status)
router.get('/students/registered', async (req, res) => {
  try {
    const users = await userModel.getAll();
    const registered = users.filter(u => u.role === 'student' && u.status === 'pending');
    res.render('admin_students_by_status', { 
      students: registered, 
      statusTitle: 'Registered Students',
      statusType: 'registered',
      user: { ...(req.session.user || {}), role: req.session.role }
    });
  } catch (err) {
    console.error('Registered students route error:', err);
    res.status(500).send('Error loading registered students');
  }
});

// Accepted students route (approved status)
router.get('/students/accepted', async (req, res) => {
  try {
    const users = await userModel.getAll();
    const accepted = users.filter(u => u.role === 'student' && u.status === 'approved');
    res.render('admin_students_by_status', { 
      students: accepted, 
      statusTitle: 'Accepted Students',
      statusType: 'accepted',
      user: { ...(req.session.user || {}), role: req.session.role }
    });
  } catch (err) {
    console.error('Accepted students route error:', err);
    res.status(500).send('Error loading accepted students');
  }
});

// Denied students route (denied status)
router.get('/students/denied', async (req, res) => {
  try {
    const users = await userModel.getAll();
    const denied = users.filter(u => u.role === 'student' && u.status === 'denied');
    res.render('admin_students_by_status', { 
      students: denied, 
      statusTitle: 'Denied Students',
      statusType: 'denied',
      user: { ...(req.session.user || {}), role: req.session.role }
    });
  } catch (err) {
    console.error('Denied students route error:', err);
    res.status(500).send('Error loading denied students');
  }
});

// Global student search endpoint
router.get('/students/search', async (req, res) => {
  try {
    const query = (req.query.q || '').toLowerCase().trim();
    
    if (!query || query.length < 2) {
      return res.json([]);
    }
    
    // Get all users (students and pre-registrations)
    const allUsers = await userModel.getAll();
    const allPreregs = await preRegModel.getAll();
    
    // Search in registered students
    const students = allUsers.filter(u => u.role === 'student');
    const matchedStudents = students.filter(student => {
      const name = (student.name || '').toLowerCase();
      const email = (student.email || '').toLowerCase();
      const username = (student.username || '').toLowerCase();
      
      return name.includes(query) || 
             email.includes(query) || 
             username.includes(query);
    }).map(student => ({
      id: student.id,
      name: student.name,
      email: student.email,
      username: student.username,
      status: student.status,
      role: student.role,
      type: 'student'
    }));
    
    // Search in pre-registrations (interested students)
    const matchedPreregs = allPreregs.filter(prereg => {
      const name = (prereg.name || '').toLowerCase();
      const email = (prereg.email || '').toLowerCase();
      const phone = (prereg.phone || '').toLowerCase();
      
      return name.includes(query) || 
             email.includes(query) || 
             phone.includes(query);
    }).map(prereg => ({
      id: prereg.id,
      name: prereg.name,
      email: prereg.email,
      username: null,
      status: 'interested',
      role: 'pre-registration',
      type: 'interested'
    }));
    
    // Combine and limit results
    const allResults = [...matchedStudents, ...matchedPreregs].slice(0, 20);
    
    res.json(allResults);
  } catch (error) {
    console.error('Student search error:', error);
    res.status(500).json({ error: 'Search failed' });
  }
});

// Reactivate alumni student
router.post('/students/:id/reactivate', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    if (student.status !== 'alumni') {
      return res.status(400).json({ error: 'Student is not an alumni' });
    }
    
    // Reactivate to approved status
    await userModel.setStatus(id, 'approved');
    
    console.log(`Alumni student ${id} reactivated to approved status by admin ${req.session.user.id}`);
    res.json({ ok: true, message: 'Student reactivated successfully' });
    
  } catch (e) {
    console.error('Reactivate student error:', e);
    res.status(500).json({ error: 'Failed to reactivate student' });
  }
});

// Update alumni date
router.post('/students/:id/update-alumni-date', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const { alumniDate } = req.body;
    
    const student = await userModel.findById(id);
    if (!student || student.role !== 'student' || student.status !== 'alumni') {
      return res.status(404).json({ error: 'Alumni student not found' });
    }
    
    // Update profile with new alumni date
    const profile = student.profile || {};
    profile.becameAlumniAt = alumniDate ? new Date(alumniDate).toISOString() : null;
    await userModel.updateProfile(id, profile);
    
    console.log(`Alumni date updated for student ${id} by admin ${req.session.user.id}`);
    res.json({ ok: true, message: 'Alumni date updated successfully' });
    
  } catch (e) {
    console.error('Update alumni date error:', e);
    res.status(500).json({ error: 'Failed to update alumni date' });
  }
});

// Move student to alumni
router.post('/students/:id/move-to-alumni', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    if (student.status === 'alumni') {
      return res.status(400).json({ error: 'Student is already alumni' });
    }
    
    // Move to alumni status
    await userModel.setStatus(id, 'alumni');
    
    // Store the timestamp when they became alumni
    const profile = student.profile || {};
    profile.becameAlumniAt = new Date().toISOString();
    await userModel.updateProfile(id, profile);
    
    console.log(`Student ${id} moved to alumni by admin ${req.session.user.id}`);
    res.json({ ok: true, message: 'Student moved to alumni successfully' });
    
  } catch (e) {
    console.error('Move to alumni error:', e);
    res.status(500).json({ error: 'Failed to move student to alumni' });
  }
});

// Re-enroll alumni student (Alumni -> Pending)
router.post('/students/:id/reenroll', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    if (student.status !== 'alumni') {
      return res.status(400).json({ error: 'Only alumni students can be re-enrolled' });
    }
    
    // Change status to pending for re-enrollment
    await userModel.setStatus(id, 'pending');
    
    // Update profile with re-enrollment info
    const profile = student.profile || {};
    profile.reenrolledAt = new Date().toISOString();
    profile.reenrolledBy = req.session.user.id;
    profile.previousAlumniStatus = true;
    await userModel.updateProfile(id, profile);
    
    // Also activate the student in mdtsapps_myclass.users table by email
    // This sets status to 'active' and StartDate to current date
    if (student.email) {
      try {
        const testEngineResult = await userModel.activateTestEngineByEmail(student.email);
        if (testEngineResult) {
          console.log(`Test engine user activated for email ${student.email}`);
        }
      } catch (teError) {
        console.error('Error activating test engine user:', teError);
        // Don't fail the re-enrollment if test engine update fails
      }
    }
    
    console.log(`Alumni student ${id} re-enrolled (status changed to pending) by admin ${req.session.user.id}`);
    res.json({ ok: true, message: 'Student has been moved to pending status for re-enrollment' });
    
  } catch (e) {
    console.error('Re-enroll student error:', e);
    res.status(500).json({ error: 'Failed to re-enroll student' });
  }
});

// Bulk check all students for alumni transition (manual trigger)
router.post('/students/bulk-check-alumni', async (req, res) => {
  try {
    if (!req.session.user || req.session.user.role !== 'admin') {
      return res.status(403).json({ error: 'Unauthorized' });
    }
    
    const result = await checkAllStudentsAlumniStatus();
    
    console.log(`Bulk alumni check completed by admin ${req.session.user.id}:`, result);
    res.json({ 
      ok: true, 
      message: `Checked ${result.checked} students, transitioned ${result.transitioned} to alumni`,
      ...result
    });
    
  } catch (e) {
    console.error('Bulk alumni check error:', e);
    res.status(500).json({ error: 'Failed to check alumni status' });
  }
});

// Student Lookup - search across all student types
router.get('/students/lookup', async (req, res) => {
  try {
    const query = req.query.q?.trim().toLowerCase();
    
    if (!query) {
      return res.json({ results: [] });
    }

    const results = [];

    // Search in registered/approved/denied/alumni students (users table)
    try {
      const allStudents = await userModel.getAll();
      const matchingStudents = allStudents.filter(student => {
        // Search in various fields
        const firstName = (student.profile?.firstName || student.name || '').toLowerCase();
        const lastName = (student.profile?.lastName || '').toLowerCase();
        const email = (student.email || '').toLowerCase();
        const phone = (student.profile?.phone || '').toLowerCase();
        const studentId = (student.profile?.studentId || student.id?.toString() || '').toLowerCase();
        const username = (student.username || '').toLowerCase();

        return firstName.includes(query) ||
               lastName.includes(query) ||
               email.includes(query) ||
               phone.includes(query) ||
               studentId.includes(query) ||
               username.includes(query);
      });

      matchingStudents.forEach(student => {
        results.push({
          id: student.id,
          firstName: student.profile?.firstName || student.name || '',
          lastName: student.profile?.lastName || '',
          email: student.email || '',
          phone: student.profile?.phone || '',
          studentId: student.profile?.studentId || student.id,
          status: student.status || 'unknown',
          source: 'student'
        });
      });
    } catch (err) {
      console.error('Error searching students:', err);
    }

    // Search in interested students (pre-registrations)
    try {
      const allPreRegs = await preRegModel.getAll();
      const matchingPreRegs = allPreRegs.filter(preReg => {
        const firstName = (preReg.firstName || '').toLowerCase();
        const lastName = (preReg.lastName || '').toLowerCase();
        const email = (preReg.email || '').toLowerCase();
        const phone = (preReg.phone || '').toLowerCase();

        return firstName.includes(query) ||
               lastName.includes(query) ||
               email.includes(query) ||
               phone.includes(query);
      });

      matchingPreRegs.forEach(preReg => {
        results.push({
          id: preReg.id,
          firstName: preReg.firstName || '',
          lastName: preReg.lastName || '',
          email: preReg.email || '',
          phone: preReg.phone || '',
          studentId: null,
          status: 'interested',
          source: 'preReg'
        });
      });
    } catch (err) {
      console.error('Error searching pre-registrations:', err);
    }

    // Sort results by last name, then first name
    results.sort((a, b) => {
      const lastNameCompare = (a.lastName || '').localeCompare(b.lastName || '');
      if (lastNameCompare !== 0) return lastNameCompare;
      return (a.firstName || '').localeCompare(b.firstName || '');
    });

    res.json({ results });
  } catch (error) {
    console.error('Student lookup error:', error);
    res.status(500).json({ error: 'Search failed', results: [] });
  }
});

router.post('/approve/:id', async (req, res) => {
  const user = await userModel.setStatus(Number(req.params.id), 'approved');
  if (user && user.email) {
    const name = (user.profile && user.profile.firstName) || user.name || 'Student';
    try {
      const { subject, html, text } = emailTemplates.render('studentApproved', { name });
      await transporter.sendMail({
        from: 'no-reply@mdts-apps.com',
        to: user.email,
        subject,
        html,
        text
      });
      try { await userModel.setLastContacted(user.id, new Date()); } catch(_) {}
    } catch (e) {
      console.error('Error sending approval email', e);
    }
  }
  res.redirect('/admin/students/pending');
});


// Render form to create a new event
router.get('/create-event', async (req, res) => {
  const events = await eventModel.getAllEvents();
  // Filter to only show future events (today or later)
  const now = new Date();
  now.setHours(0, 0, 0, 0);
  const futureEvents = events.filter(e => {
    const eventDate = new Date(e.eventDate);
    return eventDate >= now;
  });
  res.render('create_event', { user: req.session.user, events: futureEvents, editEvent: null });
});

// Handle event creation
router.post('/create-event', mediaUpload.single('attachment'), async (req, res) => {
  const { name, date, description, autoShowPdf, eventId } = req.body;
  const attachment = req.file ? `/uploads/${req.file.filename}` : null;
  
  // Extract just the date portion (YYYY-MM-DD) from datetime-local input
  const eventDate = date ? date.split('T')[0] : date;
  
  if (eventId) {
    // Update existing event
    const existingEvent = await eventModel.getEventById(eventId);
    await eventModel.updateEvent(eventId, { 
      name, 
      eventDate, 
      description, 
      attachment: attachment || existingEvent.attachment,
      autoShowPdf: autoShowPdf === 'on'
    });
  } else {
    // Create new event
    await eventModel.createEvent({ 
      name, 
      eventDate, 
      description, 
      attachment,
      autoShowPdf: autoShowPdf === 'on'
    });
  }
  res.redirect('/admin/create-event');
});

// Edit event (load form with event data)
router.get('/edit-event/:id', async (req, res) => {
  const events = await eventModel.getAllEvents();
  // Filter to only show future events (today or later)
  const now = new Date();
  now.setHours(0, 0, 0, 0);
  const futureEvents = events.filter(e => {
    const eventDate = new Date(e.eventDate);
    return eventDate >= now;
  });
  const editEvent = await eventModel.getEventById(req.params.id);
  res.render('create_event', { user: req.session.user, events: futureEvents, editEvent });
});

// Delete event
router.post('/delete-event/:id', async (req, res) => {
  await eventModel.deleteEvent(req.params.id);
  res.json({ success: true });
});

// View RSVPs for events
router.get('/event-rsvps', async (req, res) => {
  const rsvps = await rsvpModel.getAllRSVPs();
  res.render('event_rsvps', { user: req.session.user, rsvps });
});

// Contact an RSVP (AJAX)
router.post('/rsvps/:id/contact', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const rsvp = await rsvpModel.getById(id);
    if (!rsvp || !rsvp.email) return res.status(404).json({ error: 'RSVP not found' });
    const { subject, message } = req.body || {};
    if (!subject || !message) return res.status(400).json({ error: 'Missing subject or message' });
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: rsvp.email,
      subject,
      html: message,
      text: message.replace(/<[^>]*>/g, '')
    });
    return res.json({ ok: true });
  } catch (e) {
    console.error('RSVP contact error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

// Delete an RSVP
router.post('/rsvps/:id/delete', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const ok = await rsvpModel.deleteById(id);
    if (!ok) return res.status(404).json({ error: 'Not found' });
    return res.json({ ok: true });
  } catch (e) {
    console.error('RSVP delete error', e);
    return res.status(500).json({ error: 'Failed to delete' });
  }
});

const pad2 = (num) => String(num).padStart(2, '0');
const toDateKey = (value) => {
  if (!value) return null;
  const d = value instanceof Date ? value : new Date(value);
  if (Number.isNaN(d.getTime())) return null;
  return `${d.getFullYear()}-${pad2(d.getMonth() + 1)}-${pad2(d.getDate())}`;
};

const toTimeKey = (value) => {
  if (!value) return null;
  const d = value instanceof Date ? value : new Date(value);
  if (Number.isNaN(d.getTime())) return null;
  return `${pad2(d.getHours())}:${pad2(d.getMinutes())}`;
};

const toDateInput = (value) => {
  if (!value) return null;
  const d = value instanceof Date ? value : new Date(value);
  if (Number.isNaN(d.getTime())) return null;
  return `${d.getFullYear()}-${pad2(d.getMonth() + 1)}-${pad2(d.getDate())}`;
};

const toTimeInput = (value) => {
  if (!value) return null;
  const d = value instanceof Date ? value : new Date(value);
  if (Number.isNaN(d.getTime())) return null;
  return `${pad2(d.getHours())}:${pad2(d.getMinutes())}`;
};

function normalizeRangeStart(value) {
  if (!value) return null;
  const d = value instanceof Date ? new Date(value) : new Date(value);
  if (Number.isNaN(d.getTime())) return null;
  return d;
}

function normalizeRangeEnd(value, fallbackStart) {
  if (value) {
    const d = value instanceof Date ? new Date(value) : new Date(value);
    if (!Number.isNaN(d.getTime())) return d;
  }
  return fallbackStart ? new Date(fallbackStart) : null;
}

async function buildCalendarItems(rangeStart, rangeEnd, types) {
  const typeSet = types || new Set(['events', 'interviews', 'classes', 'tests', 'assignments', 'custom']);
  const items = [];

  const addItem = (entry) => {
    if (!entry) return;
    const start = normalizeRangeStart(entry.start);
    if (!start) return;
    const end = normalizeRangeEnd(entry.end, start);
    const allDay = Boolean(entry.allDay);
    items.push({
      id: entry.id,
      type: entry.type,
      title: entry.title,
      description: entry.description || null,
      start: start.toISOString(),
      end: end ? end.toISOString() : null,
      allDay,
      startDate: toDateKey(start),
      endDate: toDateKey(end || start),
      startTime: allDay ? null : toTimeKey(start),
      endTime: (!allDay && end) ? toTimeKey(end) : null,
      link: entry.link || null,
      notifyCount: Number(entry.notifyCount || 0)
    });
  };

  const within = (start, end) => {
    const startMs = start ? start.getTime() : Number.NEGATIVE_INFINITY;
    const endMs = end ? end.getTime() : Number.POSITIVE_INFINITY;
    const windowStart = rangeStart ? rangeStart.getTime() : Number.NEGATIVE_INFINITY;
    const windowEnd = rangeEnd ? rangeEnd.getTime() : Number.POSITIVE_INFINITY;
    return endMs >= windowStart && startMs <= windowEnd;
  };

  if (typeSet.has('custom')) {
    try {
      const customs = await calendarEventModel.listBetween(rangeStart, rangeEnd);
      let attendeeMap = new Map();
      let peopleData = null;
      if (customs.length) {
        attendeeMap = await calendarEventModel.getAttendeeIdsForEvents(customs.map((ev) => ev.id));
        // Load people data to get names
        peopleData = await loadCalendarPeople();
      }
      for (const ev of customs) {
        const start = ev.start || null;
        const end = ev.end || start;
        if (!start || !within(start, end)) continue;
        let adjustedEnd = end;
        if (ev.allDay) {
          adjustedEnd = new Date(end);
          adjustedEnd.setHours(23, 59, 59, 999);
        }
        // Get attendee names
        const attendeeIds = attendeeMap.get(ev.id) || [];
        const attendeeNames = peopleData 
          ? attendeeIds.map(id => peopleData.byId.get(id)).filter(Boolean).map(person => person.name)
          : [];
        
        addItem({
          id: `custom-${ev.id}`,
          type: 'custom',
          title: ev.title,
          description: ev.description,
          start,
          end: adjustedEnd,
          allDay: ev.allDay,
          link: null,
          notifyCount: attendeeIds.length,
          attendeeNames: attendeeNames
        });
      }
    } catch (err) {
      console.error('custom calendar events load failed', err);
    }
  }

  if (typeSet.has('events')) {
    try {
      const events = await eventModel.getAllEvents();
      for (const e of events) {
        if (!e.eventDate) continue;
        
        // Parse the event date - handle both Date objects and string dates
        let start;
        if (e.eventDate instanceof Date) {
          start = new Date(e.eventDate);
        } else {
          // Convert to ISO date string format (YYYY-MM-DD)
          const dateStr = String(e.eventDate).split('T')[0];
          start = new Date(`${dateStr}T00:00:00`);
        }
        
        // Check if date is valid
        if (isNaN(start.getTime())) {
          console.error('Invalid event date:', e.eventDate, 'for event:', e.name);
          continue;
        }
        
        const end = new Date(start);
        end.setHours(23, 59, 59, 999);
        if (!within(start, end)) continue;
        addItem({
          id: `event-${e.id}`,
          type: 'event',
          title: e.name,
          description: e.description,
          start,
          end,
          allDay: true,
          link: '/admin/events'
        });
      }
    } catch (err) {
      console.error('events load failed', err);
    }
  }

  if (typeSet.has('interviews')) {
    try {
      const interviews = await interviewModel.listBetween(rangeStart, rangeEnd);
      for (const it of interviews) {
        const start = it.scheduledAt ? new Date(it.scheduledAt) : (it.scheduledat ? new Date(it.scheduledat) : null);
        if (!start) continue;
        const durationMs = (Number(it.durationMinutes) || 30) * 60000;
        const end = new Date(start.getTime() + durationMs);
        if (!within(start, end)) continue;
        addItem({
          id: `interview-${it.id}`,
          type: 'interview',
          title: `Interview: ${(it.studentName || '').trim()}`.trim() || 'Interview',
          description: null,
          start,
          end,
          allDay: false,
          link: '/admin/interviews'
        });
      }
    } catch (err) {
      console.error('interviews load failed', err);
    }
    
    // Add career advisor interviews
    try {
      const careerInterviews = careerAdvisorInterviewModel.getForCalendar(
        rangeStart ? rangeStart.toISOString().split('T')[0] : null,
        rangeEnd ? rangeEnd.toISOString().split('T')[0] : null
      );
      
      for (const ci of careerInterviews) {
        // Parse the time (e.g., "9:00 AM") and create proper Date objects
        const [timeStr, period] = ci.time.split(' ');
        const [hours, minutes] = timeStr.split(':');
        let hour = parseInt(hours, 10);
        
        if (period === 'PM' && hour !== 12) {
          hour += 12;
        } else if (period === 'AM' && hour === 12) {
          hour = 0;
        }
        
        const start = new Date(`${ci.date}T${String(hour).padStart(2, '0')}:${minutes || '00'}:00`);
        const end = new Date(start.getTime() + 45 * 60000); // 45 minutes
        
        if (!within(start, end)) continue;
        
        addItem({
          id: `career-advisor-${ci.id}`,
          type: 'career-advisor',
          title: `Career Advisor: ${ci.studentName}`,
          description: `Interest: ${ci.interestArea}${ci.advisorName ? ' | Advisor: ' + ci.advisorName : ''}`,
          start,
          end,
          allDay: false,
          link: '/admin/career-advisor-interviews'
        });
      }
    } catch (err) {
      console.error('career advisor interviews load failed', err);
    }
  }

  if (typeSet.has('classes') || typeSet.has('tests') || typeSet.has('assignments')) {
    try {
      const classes = await classModel.getAllClasses();
      for (const c of classes) {
        if (typeSet.has('classes') && c.schedule && Array.isArray(c.schedule)) {
          // Generate individual class sessions based on schedule
          const dayMap = { 'Sunday': 0, 'Monday': 1, 'Tuesday': 2, 'Wednesday': 3, 'Thursday': 4, 'Friday': 5, 'Saturday': 6 };
          
          c.schedule.forEach(scheduleItem => {
            const dayOfWeek = dayMap[scheduleItem.day];
            if (dayOfWeek !== undefined) {
              // Generate dates for this day of week within the window
              const windowStart = rangeStart || new Date();
              const windowEnd = rangeEnd || new Date(windowStart.getTime() + 90 * 24 * 60 * 60 * 1000);
              
              let currentDate = new Date(windowStart);
              currentDate.setDate(currentDate.getDate() - currentDate.getDay() + dayOfWeek);
              if (currentDate < windowStart) currentDate.setDate(currentDate.getDate() + 7);
              
              while (currentDate <= windowEnd) {
                // Check if within class start/end dates if they exist
                if (c.startDate) {
                  const classStart = new Date(c.startDate);
                  if (currentDate < classStart) {
                    currentDate.setDate(currentDate.getDate() + 7);
                    continue;
                  }
                }
                if (c.endDate) {
                  const classEnd = new Date(c.endDate);
                  if (currentDate > classEnd) break;
                }
                
                // Create start/end times for this session
                let sessionStart, sessionEnd;
                if (scheduleItem.start && scheduleItem.end) {
                  const [startHour, startMin] = scheduleItem.start.split(':').map(Number);
                  const [endHour, endMin] = scheduleItem.end.split(':').map(Number);
                  sessionStart = new Date(currentDate);
                  sessionStart.setHours(startHour, startMin || 0, 0, 0);
                  sessionEnd = new Date(currentDate);
                  sessionEnd.setHours(endHour, endMin || 0, 0, 0);
                } else {
                  sessionStart = new Date(currentDate);
                  sessionStart.setHours(0, 0, 0, 0);
                  sessionEnd = new Date(currentDate);
                  sessionEnd.setHours(23, 59, 59, 999);
                }
                
                if (within(sessionStart, sessionEnd)) {
                  addItem({
                    id: `class-${c.id}-${currentDate.toISOString().split('T')[0]}`,
                    type: 'class',
                    title: `${c.name}${c.cohort ? ' - ' + c.cohort : ''}`,
                    description: `Time: ${scheduleItem.start || ''} - ${scheduleItem.end || ''}`,
                    start: sessionStart,
                    end: sessionEnd,
                    allDay: false,
                    link: `/admin/classes/${c.id}`
                  });
                }
                
                currentDate.setDate(currentDate.getDate() + 7);
              }
            }
          });
        }

        if (typeSet.has('tests') && Array.isArray(c.tests)) {
          for (const t of c.tests) {
            if (!t.dueDate) continue;
            const start = new Date(t.dueDate);
            const end = new Date(start);
            end.setMinutes(end.getMinutes() + 30);
            if (!within(start, end)) continue;
            addItem({
              id: `test-${c.id}-${t.id || start.getTime()}`,
              type: 'test',
              title: `Test: ${t.title || 'Test'} (${c.name})`,
              description: null,
              start,
              end,
              allDay: false,
              link: `/admin/classes/${c.id}#tests`
            });
          }
        }

        if (typeSet.has('assignments') && Array.isArray(c.assignments)) {
          for (const a of c.assignments) {
            if (!a.dueDate) continue;
            const start = new Date(a.dueDate);
            const end = new Date(start);
            end.setMinutes(end.getMinutes() + 30);
            if (!within(start, end)) continue;
            addItem({
              id: `assign-${c.id}-${a.id || start.getTime()}`,
              type: 'assignment',
              title: `Assignment: ${a.title || 'Assignment'} (${c.name})`,
              description: null,
              start,
              end,
              allDay: false,
              link: `/admin/classes/${c.id}#assignments`
            });
          }
        }
      }
    } catch (err) {
      console.error('classes load failed', err);
    }
  }

  return items;
}

function isTruthy(value) {
  if (typeof value === 'boolean') return value;
  if (typeof value === 'string') {
    const normalized = value.trim().toLowerCase();
    return ['1', 'true', 'yes', 'on'].includes(normalized);
  }
  return Boolean(value);
}

function parseDateTimeInput(date, time, { endOfDay = false } = {}) {
  if (!date) return null;
  const [y, m, d] = String(date).split('-').map((part) => Number(part));
  if (![y, m, d].every((n) => Number.isInteger(n) && n > 0)) return null;
  if (endOfDay) {
    return new Date(y, m - 1, d, 23, 59, 59, 0);
  }
  if (!time) {
    return new Date(y, m - 1, d, 0, 0, 0, 0);
  }
  const [hh, mm] = String(time).split(':').map((part) => Number(part));
  if (![hh, mm].every((n) => Number.isInteger(n) && n >= 0)) return null;
  return new Date(y, m - 1, d, hh, mm, 0, 0);
}

function parseCalendarEventBody(body = {}) {
  const title = (body.title || '').trim();
  const description = (body.description || '').trim();
  const startDate = body.startDate || body.start_date;
  const endDate = body.endDate || body.end_date;
  const allDay = isTruthy(body.allDay);
  const startTime = allDay ? '' : (body.startTime || body.start_time || '');
  const endTime = allDay ? '' : (body.endTime || body.end_time || '');
  const adminIds = toIdArray(body.adminIds || body.admin_ids || body.admins);
  const teacherIds = toIdArray(body.teacherIds || body.teacher_ids || body.teachers);

  if (!title) {
    return { error: 'Missing title' };
  }
  if (!startDate) {
    return { error: 'Missing start date' };
  }

  const start = parseDateTimeInput(startDate, startTime);
  if (!start) {
    return { error: 'Invalid start date/time' };
  }

  let end = null;
  if (allDay) {
    const endCandidate = parseDateTimeInput(endDate || startDate, null, { endOfDay: true });
    end = endCandidate || new Date(start.getFullYear(), start.getMonth(), start.getDate(), 23, 59, 59, 0);
  } else if (endDate || endTime) {
    end = parseDateTimeInput(endDate || startDate, endTime || startTime);
    if (!end) return { error: 'Invalid end date/time' };
  }

  if (!end) {
    end = new Date(start.getTime() + 60 * 60 * 1000);
  }

  if (end.getTime() < start.getTime()) {
    return { error: 'End must be after start' };
  }

  return {
    title,
    description: description || null,
    start,
    end,
    allDay,
    adminIds,
    teacherIds
  };
}

function formatCustomEventResponse(event, attendeeInfo = {}) {
  if (!event) return null;
  const adminIds = Array.isArray(attendeeInfo.adminIds) ? attendeeInfo.adminIds.map((id) => Number(id)) : [];
  const teacherIds = Array.isArray(attendeeInfo.teacherIds) ? attendeeInfo.teacherIds.map((id) => Number(id)) : [];
  const notifyCount = adminIds.length + teacherIds.length;
  
  // Collect attendee names
  const people = attendeeInfo.people || { byId: new Map() };
  const attendeeNames = [...adminIds, ...teacherIds]
    .map(id => people.byId.get(id))
    .filter(Boolean)
    .map(person => person.name);
  
  return {
    id: event.id,
    title: event.title,
    description: event.description || '',
    allDay: Boolean(event.allDay),
    start: event.start ? event.start.toISOString() : null,
    end: event.end ? event.end.toISOString() : null,
    startDate: toDateInput(event.start),
    endDate: toDateInput(event.end),
    startTime: event.allDay ? '' : toTimeInput(event.start),
    endTime: event.allDay ? '' : toTimeInput(event.end),
    adminIds,
    teacherIds,
    attendeeNames,
    notifyCount
  };
}

function parseCustomEventId(param) {
  const match = /^custom-(\d+)$/.exec(String(param));
  if (!match) return null;
  return Number(match[1]);
}

function toIdArray(value) {
  if (value === undefined || value === null) return [];
  const arr = Array.isArray(value) ? value : [value];
  return arr
    .map((v) => Number(v))
    .filter((id) => Number.isInteger(id) && id > 0);
}

async function loadCalendarPeople() {
  const [admins, teachers] = await Promise.all([
    userModel.listAdmins().catch(() => []),
    userModel.listTeachers().catch(() => [])
  ]);
  const adminMap = new Map(admins.map((a) => [a.id, a]));
  const teacherMap = new Map(teachers.map((t) => [t.id, t]));
  const byId = new Map([...adminMap, ...teacherMap]);
  return { admins, teachers, adminMap, teacherMap, byId };
}

function classifyAttendees(ids = [], people) {
  const adminIds = [];
  const teacherIds = [];
  ids.forEach((id) => {
    if (people.adminMap.has(id)) adminIds.push(id);
    if (people.teacherMap.has(id)) teacherIds.push(id);
  });
  return { adminIds, teacherIds };
}

async function syncCustomEventAttendees(eventId, adminIds, teacherIds, previousIds = []) {
  const people = await loadCalendarPeople();
  const combined = [...new Set([...adminIds, ...teacherIds])].filter((id) => people.byId.has(id));
  await calendarEventModel.replaceAttendees(eventId, combined);
  const classification = classifyAttendees(combined, people);
  const previousSet = new Set((previousIds || []).map((id) => Number(id)));
  const addedUsers = combined
    .filter((id) => !previousSet.has(id))
    .map((id) => people.byId.get(id))
    .filter(Boolean);
  return { classification, addedUsers, people };
}

function describeEventTiming(event) {
  if (!event || !event.start) return 'Date to be determined';
  const start = new Date(event.start);
  const end = event.end ? new Date(event.end) : null;
  const dateFormatter = new Intl.DateTimeFormat('en-US', { dateStyle: 'full' });
  const timeFormatter = new Intl.DateTimeFormat('en-US', { timeStyle: 'short' });

  if (event.allDay) {
    let message = dateFormatter.format(start);
    if (event.endDate && event.endDate !== event.startDate) {
      const endDateObj = end && !Number.isNaN(end.getTime()) ? end : new Date(`${event.endDate}T00:00:00`);
      message += ` – ${dateFormatter.format(endDateObj)}`;
    }
    return `${message} (All day)`;
  }

  const startText = `${dateFormatter.format(start)} ${timeFormatter.format(start)}`;
  if (!end || Number.isNaN(end.getTime())) return startText;

  if (event.startDate && event.endDate && event.startDate !== event.endDate) {
    return `${startText} – ${dateFormatter.format(end)} ${timeFormatter.format(end)}`;
  }

  return `${startText} – ${timeFormatter.format(end)}`;
}

function escapeHtml(value = '') {
  return String(value).replace(/[&<>"']/g, (ch) => ({
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    "'": '&#39;'
  }[ch] || ch));
}

function plainTextFromHtml(html = '') {
  return String(html)
    .replace(/<style[\s\S]*?<\/style>/gi, ' ')
    .replace(/<script[\s\S]*?<\/script>/gi, ' ')
    .replace(/<[^>]*>/g, ' ')
    .replace(/\s+/g, ' ')
    .trim();
}

async function sendEventAssignmentEmails(eventInfo, recipients, origin, actor) {
  if (!Array.isArray(recipients) || !recipients.length) return;
  const actorName = actor?.name || actor?.username || 'MDTS Admin';
  const whenText = describeEventTiming(eventInfo);
  const subject = `Added to calendar event: ${eventInfo.title}`;

  const linkMarkup = origin ? `<p><a href="${origin}/admin">Open the admin calendar</a></p>` : '';

  for (const person of recipients) {
    if (!person || !person.email) continue;
    const greetingName = person.name || person.username || 'there';
    const safeDescription = eventInfo.description
      ? `<p><strong>Description:</strong><br>${escapeHtml(eventInfo.description).replace(/\n/g, '<br>')}</p>`
      : '';
    const html = `
      <p>Hello ${escapeHtml(greetingName)},</p>
      <p>You were added to the calendar event <strong>${escapeHtml(eventInfo.title)}</strong>.</p>
      <p><strong>When:</strong> ${escapeHtml(whenText)}</p>
      ${safeDescription}
      <p>Added by ${escapeHtml(actorName)}.</p>
      ${linkMarkup}
    `;

    let text = `Hello ${greetingName},\n\n`;
    text += `You were added to the calendar event "${eventInfo.title}".\n`;
    text += `When: ${whenText}\n`;
    if (eventInfo.description) {
      text += `Description:\n${eventInfo.description}\n`;
    }
    text += `Added by: ${actorName}\n`;
    if (origin) text += `View calendar: ${origin}/admin\n`;

    try {
      await transporter.sendMail({
        to: person.email,
        from: 'MD Technical School <noreply@mdts-apps.com>',
        subject,
        text,
        html
      });
    } catch (err) {
      console.error('calendar event email send failed', err);
    }
  }
}

// Alias route for interested students (pre-registrations)
router.get('/students/interested', async (req, res) => {
  const allPreregs = await preRegModel.getAll();
  
  // Get all student accounts to filter out those who have fully registered
  const allUsers = await userModel.getAll();
  const studentEmails = new Set(
    allUsers
      .filter(u => u.role === 'student')
      .map(u => u.email?.toLowerCase().trim())
      .filter(Boolean)
  );
  
  // Filter out pre-registrations that have corresponding student accounts
  const preregs = allPreregs.filter(p => {
    const preEmail = p.email?.toLowerCase().trim();
    return !preEmail || !studentEmails.has(preEmail);
  });
  
  let leadsByEmail = new Map();
  let adminsForActions = [];
  try {
    const leads = await leadModel.getAll();
    leadsByEmail = new Map(leads.map(l => [l.email?.toLowerCase().trim(), l]));
  } catch (e) { console.error('Load leads for interested students failed', e); }
  try {
    const admins = await userModel.listAdmins();
    adminsForActions = admins.map((admin) => ({
      id: admin.id,
      name: admin.name || admin.username || admin.email || `Admin #${admin.id}`,
      email: admin.email || ''
    }));
  } catch (e) { console.error('Load admins for interested students failed', e); }
  const enriched = preregs.map(p => ({ ...p, leadId: (p.email && leadsByEmail.get(p.email)?.id) || null }));
  const dropdownData = dropdowns.getAll();
  res.render('admin_pre_registered', { 
    preregs: enriched, 
    user: { ...(req.session.user || {}), role: req.session.role }, 
    templates: marketingTemplates, 
    admins: adminsForActions,
    courses: dropdownData.courses || [],
    affiliatePrograms: dropdownData.affiliatePrograms || []
  });
});

router.get('/pre-registrations', async (req, res) => {
  const allPreregs = await preRegModel.getAll();
  
  // Get all student accounts to filter out those who have fully registered
  const allUsers = await userModel.getAll();
  const studentEmails = new Set(
    allUsers
      .filter(u => u.role === 'student')
      .map(u => u.email?.toLowerCase().trim())
      .filter(Boolean)
  );
  
  // Filter out pre-registrations that have corresponding student accounts
  const preregs = allPreregs.filter(p => {
    const preregEmail = p.email?.toLowerCase().trim();
    return !preregEmail || !studentEmails.has(preregEmail);
  });
  
  let leadsByEmail = new Map();
  let adminsForActions = [];
  try {
    const leads = await leadModel.getAll();
    leadsByEmail = new Map(leads.map(l => [l.email, l]));
  } catch (e) { console.error('Load leads for preregs failed', e); }
  try {
    const admins = await userModel.listAdmins();
    adminsForActions = admins.map((admin) => ({
      id: admin.id,
      name: admin.name || admin.username || admin.email || `Admin #${admin.id}`,
      email: admin.email || ''
    }));
  } catch (e) { console.error('Load admins for prereg actions failed', e); }
  const enriched = preregs.map(p => ({ ...p, leadId: (p.email && leadsByEmail.get(p.email)?.id) || null }));
  const dropdownData = dropdowns.getAll();
  res.render('admin_pre_registered', { 
    preregs: enriched, 
    user: { ...(req.session.user || {}), role: req.session.role }, 
    templates: marketingSubjects, 
    admins: adminsForActions,
    courses: dropdownData.courses || [],
    affiliatePrograms: dropdownData.affiliatePrograms || []
  });
});

// Update pre-registration data
router.put('/pre-registrations/:id', async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    const {
      name,
      email,
      phone,
      phoneCarrier,
      address,
      zip,
      state,
      serving,
      branch,
      course,
      applicantType,
      referral,
      referralEmail
    } = req.body;

    // Validate required fields
    if (!name || !email || !phone || !zip || !state || !course || !applicantType) {
      return res.status(400).json({ error: 'Missing required fields' });
    }

    // Update the pre-registration record
    const result = await preRegModel.update(id, {
      name: name.trim(),
      email: email.trim(),
      phone: phone.trim(),
      phoneCarrier: phoneCarrier ? phoneCarrier.trim() : null,
      address: address ? address.trim() : null,
      zip: zip.trim(),
      state: state.trim(),
      serving: serving === true || serving === 'true' || serving === 'yes',
      branch: (serving === true || serving === 'true' || serving === 'yes') && branch ? branch.trim() : null,
      course: course.trim(),
      applicantType: applicantType.trim(),
      referral: referral ? referral.trim() : null,
      referralEmail: referralEmail ? referralEmail.trim() : null
    });

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: 'Pre-registration not found' });
    }

    // Return updated record
    const updatedRecord = await preRegModel.findById(id);
    res.json({ success: true, record: updatedRecord });
  } catch (error) {
    console.error('Error updating pre-registration:', error);
    res.status(500).json({ error: 'Failed to update pre-registration' });
  }
});

// Delete pre-registration
router.delete('/pre-registrations/:id', async (req, res) => {
  try {
    const id = parseInt(req.params.id);

    if (!id || isNaN(id)) {
      return res.status(400).json({ error: 'Invalid pre-registration ID' });
    }

    // Check if the pre-registration exists
    const existingRecord = await preRegModel.findById(id);
    if (!existingRecord) {
      return res.status(404).json({ error: 'Pre-registration not found' });
    }

    // Delete the pre-registration record
    const result = await preRegModel.deleteById(id);

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: 'Pre-registration not found' });
    }

    res.json({ success: true, message: 'Pre-registration deleted successfully' });
  } catch (error) {
    console.error('Error deleting pre-registration:', error);
    res.status(500).json({ error: 'Failed to delete pre-registration' });
  }
});

async function createPendingStudentFromPreReg(preReg) {
  if (!preReg) throw new Error('Missing pre-registration record');

  const sanitizeUsername = (value = '') =>
    String(value)
      .toLowerCase()
      .replace(/[^a-z0-9._-]/g, '')
      .slice(0, 40);

  let usernameBase = '';
  if (preReg.email) {
    usernameBase = sanitizeUsername(preReg.email.split('@')[0] || '');
  }
  if (!usernameBase && preReg.name) {
    usernameBase = sanitizeUsername(preReg.name);
  }
  if (!usernameBase) {
    usernameBase = `student${Date.now()}`;
  }

  let username = usernameBase;
  let attempts = 0;
  while (attempts < 10) {
    const existingUser = await userModel.findByUsername(username);
    if (!existingUser) break;
    attempts += 1;
    username = `${usernameBase.slice(0, 30)}${Math.floor(100 + Math.random() * 900)}`;
  }
  if (attempts >= 10) {
    username = `${usernameBase.slice(0, 24)}${Date.now()}`;
  }

  const tempPassword = Math.random().toString(36).slice(-8) + Math.random().toString(36).slice(-8);
  const nameParts = (preReg.name || '').trim().split(/\s+/);
  const firstName = nameParts[0] || '';
  const lastName = nameParts.slice(1).join(' ') || '';

  const student = await userModel.createStudent({
    username,
    name: preReg.name || '',
    email: preReg.email || '',
    password: tempPassword,
    studentId: null,
    firstName,
    lastName,
    suffix: '',
    address: preReg.address || '',
    city: '',
    state: preReg.state || '',
    zip: preReg.zip || '',
    course: preReg.course || '',
    affiliateProgram: preReg.applicantType || '',
    phones: {
      primary: preReg.phone || '',
      carrier: preReg.phoneCarrier || ''
    },
    ssn: '',
    emergencyContact: {},
    admissionDate: new Date().toISOString().split('T')[0],
    startDate: '',
    endDate: '',
    classTime: '',
    classDays: '',
    tuition: {},
    grievanceAck: false,
    financialAid: false,
    referralName: preReg.referral || '',
    referralEmail: preReg.referralEmail || '',
    gender: '',
    race: ''
  });

  if (preReg.serving && preReg.branch) {
    const profile = student.profile || {};
    profile.militaryService = {
      serving: true,
      branch: preReg.branch
    };
    await userModel.updateProfile(student.id, profile);
  }

  return { student };
}

async function sendPendingStatusEmail(student, preReg) {
  if (!student?.email) return false;
  const schoolName = branding?.schoolName || 'MD Technical School';
  const subject = `${schoolName} Enrollment Update – Application Pending`;
  const safeSchool = escapeHtml(schoolName);
  const safeName = escapeHtml(preReg?.name || student?.name || 'there');
  const htmlBody = `
    <p>Hello ${safeName},</p>
    <p>Thank you for pre-registering with ${safeSchool}. We've created your student account and your application is now <strong>pending</strong> review by our admissions team.</p>
    <p>We'll email you as soon as the review is complete. If we need additional information, an administrator will reach out directly.</p>
    <p>If you have any questions, just reply to this message and we'll be happy to help.</p>
  `;
  const wrappedHtml = emailTemplates.wrapHtml(subject, htmlBody);
  const textBody = plainTextFromHtml(wrappedHtml);
  await transporter.sendMail({
    from: `${schoolName} <noreply@mdts-apps.com>`,
    to: student.email,
    subject,
    html: wrappedHtml,
    text: textBody
  });
  return true;
}

async function handlePreRegRegistration(req, res) {
  try {
    const preRegId = parseInt(req.params.id, 10);
    if (!preRegId) {
      return res.status(400).json({ error: 'Invalid pre-registration ID' });
    }

    const preReg = await preRegModel.findById(preRegId);
    if (!preReg) {
      return res.status(404).json({ error: 'Pre-registration not found' });
    }

    if (preReg.email) {
      const existing = await userModel.findByEmail(preReg.email);
      if (existing && existing.role === 'student') {
        return res.status(400).json({
          error: 'Student account already exists',
          studentId: existing.id
        });
      }
    }

    const { student } = await createPendingStudentFromPreReg(preReg);

    let emailSent = false;
    try {
      emailSent = await sendPendingStatusEmail(student, preReg);
    } catch (err) {
      console.error('Pending status email failed', err);
    }

    try {
      await preRegModel.setLastContacted(preRegId, new Date());
    } catch (err) {
      console.error('Failed to mark pre-registration as contacted', err);
    }

    res.json({
      success: true,
      studentId: student.id,
      pendingStatusEmailSent: emailSent,
      message: emailSent
        ? 'Student registered and pending status email sent.'
        : 'Student registered. Pending status email could not be sent.'
    });
  } catch (error) {
    console.error('Error creating student from pre-registration:', error);
    res.status(500).json({ error: 'Failed to create student account' });
  }
}

router.post('/pre-registrations/:id/create-student', handlePreRegRegistration);
router.post('/pre-registrations/:id/register', handlePreRegRegistration);

router.get('/students.json', async (req, res) => {
  try {
    const emailQuery = (req.query.email || '').trim();
    if (emailQuery) {
      const student = await userModel.findByEmail(emailQuery);
      if (!student || student.role !== 'student') {
        return res.json({ student: null });
      }
      return res.json({
        student: {
          id: student.id,
          name: student.name || '',
          email: student.email || '',
          status: student.status || '',
          profile: student.profile || null
        }
      });
    }

    const users = await userModel.getAll();
    const students = users
      .filter((u) => u.role === 'student')
      .map((student) => ({
        id: student.id,
        name: student.name || '',
        email: student.email || '',
        status: student.status || '',
        profile: student.profile || null
      }));
    res.json({ students });
  } catch (e) {
    console.error('students.json error', e);
    res.status(500).json({ students: [] });
  }
});

// Calendar Highlights - Classes starting this/next week and MDTS events (Lunch & Learns, IT Workshops)
router.get('/calendar/highlights', async (req, res) => {
  try {
    const weekParam = req.query.week || 'this'; // 'this' or 'next'
    const now = new Date();
    
    // Calculate week start (Monday) and end (Sunday)
    const dayOfWeek = now.getDay();
    const daysToMonday = dayOfWeek === 0 ? 6 : dayOfWeek - 1;
    
    let weekStart, weekEnd;
    if (weekParam === 'next') {
      // Next week's Monday
      weekStart = new Date(now);
      weekStart.setDate(now.getDate() - daysToMonday + 7);
      weekStart.setHours(0, 0, 0, 0);
      weekEnd = new Date(weekStart);
      weekEnd.setDate(weekStart.getDate() + 6);
      weekEnd.setHours(23, 59, 59, 999);
    } else {
      // This week's Monday
      weekStart = new Date(now);
      weekStart.setDate(now.getDate() - daysToMonday);
      weekStart.setHours(0, 0, 0, 0);
      weekEnd = new Date(weekStart);
      weekEnd.setDate(weekStart.getDate() + 6);
      weekEnd.setHours(23, 59, 59, 999);
    }
    
    const highlights = {
      weekStart: weekStart.toISOString(),
      weekEnd: weekEnd.toISOString(),
      weekLabel: weekParam === 'next' ? 'Next Week' : 'This Week',
      classesStarting: [],
      mdtsEvents: []
    };
    
    // Get classes starting this/next week
    try {
      const classes = await classModel.getAllClasses();
      for (const c of classes) {
        if (!c.startDate) continue;
        
        // Parse start date
        let classStart;
        if (c.startDate instanceof Date) {
          classStart = new Date(c.startDate);
        } else {
          const dateStr = String(c.startDate).split('T')[0];
          classStart = new Date(`${dateStr}T00:00:00`);
        }
        
        if (isNaN(classStart.getTime())) continue;
        
        // Check if class starts within the week
        if (classStart >= weekStart && classStart <= weekEnd) {
          // Get roster (studentIds)
          const studentIds = Array.isArray(c.studentIds) ? c.studentIds : [];
          const enrollments = Array.isArray(c.studentEnrollments) ? c.studentEnrollments : [];
          
          // Gather all student IDs from enrollments as well
          const enrollmentIds = enrollments.map(e => e.studentId || e.id).filter(Boolean);
          const allStudentIds = [...new Set([...studentIds, ...enrollmentIds])].filter(id => id && id > 0);
          
          // Fetch student details
          let roster = [];
          if (allStudentIds.length > 0) {
            try {
              const placeholders = allStudentIds.map(() => '?').join(',');
              const [students] = await db.query(
                `SELECT id, name, email, status, profile FROM mdtslms_users WHERE id IN (${placeholders})`,
                allStudentIds
              );
              roster = students.map(s => {
                let profile = {};
                try { profile = s.profile ? JSON.parse(s.profile) : {}; } catch(_) {}
                return {
                  id: s.id,
                  name: s.name || profile.fullName || 'Unknown',
                  email: s.email || profile.email || '',
                  phone: profile.phone || '',
                  status: s.status
                };
              });
            } catch (err) {
              console.error('Error fetching class roster:', err);
            }
          }
          
          highlights.classesStarting.push({
            id: c.id,
            type: 'class',
            name: c.name || 'Unnamed Class',
            cohort: c.cohort || '',
            startDate: classStart.toISOString().split('T')[0],
            startDateFormatted: classStart.toLocaleDateString('en-US', { weekday: 'short', month: 'short', day: 'numeric' }),
            schedule: c.schedule || [],
            teacherId: c.teacherId,
            studentCount: roster.length,
            roster: roster,
            link: `/admin/classes/${c.id}`
          });
        }
      }
    } catch (err) {
      console.error('Error loading classes for highlights:', err);
    }
    
    // Get MDTS Events (Lunch & Learns, IT Workshops) within the week
    try {
      const events = await eventModel.getAllEvents();
      for (const e of events) {
        if (!e.eventDate) continue;
        
        // Parse event date
        let eventDate;
        if (e.eventDate instanceof Date) {
          eventDate = new Date(e.eventDate);
        } else {
          const dateStr = String(e.eventDate).split('T')[0];
          eventDate = new Date(`${dateStr}T00:00:00`);
        }
        
        if (isNaN(eventDate.getTime())) continue;
        
        // Check if event is within the week
        if (eventDate >= weekStart && eventDate <= weekEnd) {
          // Check if it's a Lunch & Learn or IT Workshop (case-insensitive)
          const eventName = (e.name || '').toLowerCase();
          const isLunchAndLearn = eventName.includes('lunch') && eventName.includes('learn');
          const isITWorkshop = eventName.includes('workshop') || eventName.includes('it ');
          
          if (isLunchAndLearn || isITWorkshop) {
            // Get RSVPs for this event
            let rsvpList = [];
            try {
              const [rsvps] = await db.query(
                `SELECT id, fullName, email, phone, city, state, branch, program, createdAt 
                 FROM mdtslms_event_rsvps WHERE eventId = ?`,
                [e.id]
              );
              rsvpList = rsvps.map(r => ({
                id: r.id,
                name: r.fullName || 'Unknown',
                email: r.email || '',
                phone: r.phone || '',
                location: [r.city, r.state].filter(Boolean).join(', '),
                branch: r.branch || '',
                program: r.program || '',
                rsvpDate: r.createdAt
              }));
            } catch (err) {
              console.error('Error fetching event RSVPs:', err);
            }
            
            highlights.mdtsEvents.push({
              id: e.id,
              type: 'event',
              name: e.name,
              description: e.description || '',
              eventDate: eventDate.toISOString().split('T')[0],
              eventDateFormatted: eventDate.toLocaleDateString('en-US', { weekday: 'short', month: 'short', day: 'numeric' }),
              rsvpCount: rsvpList.length,
              rsvpList: rsvpList,
              link: `/admin/events`
            });
          }
        }
      }
    } catch (err) {
      console.error('Error loading events for highlights:', err);
    }
    
    // Sort by date
    highlights.classesStarting.sort((a, b) => new Date(a.startDate) - new Date(b.startDate));
    highlights.mdtsEvents.sort((a, b) => new Date(a.eventDate) - new Date(b.eventDate));
    
    res.json(highlights);
  } catch (err) {
    console.error('Calendar highlights error:', err);
    res.status(500).json({ error: 'Failed to load highlights' });
  }
});

// Get class roster for a specific class
router.get('/classes/:id/roster', async (req, res) => {
  try {
    const classId = parseInt(req.params.id, 10);
    if (isNaN(classId)) return res.status(400).json({ error: 'Invalid class ID' });
    
    const classData = await classModel.findClassById(classId);
    if (!classData) return res.status(404).json({ error: 'Class not found' });
    
    const studentIds = Array.isArray(classData.studentIds) ? classData.studentIds : [];
    const enrollments = Array.isArray(classData.studentEnrollments) ? classData.studentEnrollments : [];
    const enrollmentIds = enrollments.map(e => e.studentId || e.id).filter(Boolean);
    const allStudentIds = [...new Set([...studentIds, ...enrollmentIds])].filter(id => id && id > 0);
    
    let roster = [];
    if (allStudentIds.length > 0) {
      const placeholders = allStudentIds.map(() => '?').join(',');
      const [students] = await db.query(
        `SELECT id, name, email, status, profile FROM mdtslms_users WHERE id IN (${placeholders})`,
        allStudentIds
      );
      
      // Get all emails to check test engine status
      const emails = students.map(s => s.email).filter(Boolean);
      let testEngineUsers = [];
      if (emails.length > 0) {
        const emailPlaceholders = emails.map(() => '?').join(',');
        const [teUsers] = await db.query(
          `SELECT id, email, status, StartDate FROM mdtsapps_myclass.users WHERE email IN (${emailPlaceholders})`,
          emails
        );
        testEngineUsers = teUsers;
      }
      
      roster = students.map(s => {
        let profile = {};
        try { profile = s.profile ? JSON.parse(s.profile) : {}; } catch(_) {}
        
        // Check test engine status for this student
        const teUser = testEngineUsers.find(te => te.email && te.email.toLowerCase() === (s.email || '').toLowerCase());
        
        return {
          id: s.id,
          name: s.name || profile.fullName || 'Unknown',
          email: s.email || profile.email || '',
          phone: profile.phone || '',
          status: s.status,
          testEngine: teUser ? {
            id: teUser.id,
            exists: true,
            active: teUser.status === 'active',
            startDate: teUser.StartDate
          } : {
            exists: false,
            active: false
          }
        };
      });
    }
    
    res.json({
      classId: classData.id,
      className: classData.name,
      cohort: classData.cohort || '',
      roster: roster
    });
  } catch (err) {
    console.error('Class roster error:', err);
    res.status(500).json({ error: 'Failed to load roster' });
  }
});

// Activate test engine account by email
router.post('/test-engine/activate-by-email', async (req, res) => {
  try {
    const { email } = req.body;
    if (!email) return res.status(400).json({ error: 'Email is required' });
    
    const result = await userModel.activateTestEngineByEmail(email);
    if (result) {
      res.json({ ok: true, message: 'Test engine account activated', user: result });
    } else {
      res.status(404).json({ error: 'No test engine account found for this email' });
    }
  } catch (err) {
    console.error('Activate test engine error:', err);
    res.status(500).json({ error: 'Failed to activate test engine account' });
  }
});

// Get RSVP list for a specific event
router.get('/events/:id/rsvps', async (req, res) => {
  try {
    const eventId = parseInt(req.params.id, 10);
    if (isNaN(eventId)) return res.status(400).json({ error: 'Invalid event ID' });
    
    const event = await eventModel.getEventById(eventId);
    if (!event) return res.status(404).json({ error: 'Event not found' });
    
    const [rsvps] = await db.query(
      `SELECT id, fullName, email, phone, city, state, branch, program, createdAt 
       FROM mdtslms_event_rsvps WHERE eventId = ?`,
      [eventId]
    );
    
    const rsvpList = rsvps.map(r => ({
      id: r.id,
      name: r.fullName || 'Unknown',
      email: r.email || '',
      phone: r.phone || '',
      location: [r.city, r.state].filter(Boolean).join(', '),
      branch: r.branch || '',
      program: r.program || '',
      rsvpDate: r.createdAt
    }));
    
    res.json({
      eventId: event.id,
      eventName: event.name,
      eventDate: event.eventDate,
      rsvpList: rsvpList
    });
  } catch (err) {
    console.error('Event RSVPs error:', err);
    res.status(500).json({ error: 'Failed to load RSVPs' });
  }
});

// Calendar data feed for admin dashboard
router.get('/calendar/data', async (req, res) => {
  try {
    const start = req.query.start ? new Date(req.query.start) : new Date();
    const end = req.query.end
      ? new Date(req.query.end)
      : new Date(start.getFullYear(), start.getMonth() + 1, 0, 23, 59, 59, 999);
    const rawTypes = String(req.query.types || 'events,interviews,classes,tests,assignments,custom')
      .split(',')
      .map((s) => s.trim())
      .filter(Boolean);
    const items = await buildCalendarItems(start, end, new Set(rawTypes));
    res.json({ items });
  } catch (e) {
    console.error('calendar data error', e);
    res.status(500).json({ error: 'failed' });
  }
});

router.post('/calendar/events', async (req, res) => {
  try {
    const payload = parseCalendarEventBody(req.body);
    if (payload.error) {
      return res.status(400).json({ error: payload.error });
    }

    const {
      adminIds = [],
      teacherIds = [],
      title,
      description,
      start,
      end,
      allDay
    } = payload;

    const created = await calendarEventModel.create({
      title,
      description,
      start,
      end,
      allDay,
      createdBy: req.session.user?.id || null
    });

    const attendeeSync = await syncCustomEventAttendees(created.id, adminIds, teacherIds, []);
    const eventResponse = formatCustomEventResponse(created, attendeeSync.classification);
    res.json({ ok: true, event: eventResponse });

    const origin = `${req.protocol}://${req.get('host')}`;
    sendEventAssignmentEmails(eventResponse, attendeeSync.addedUsers, origin, req.session.user)
      .catch((err) => console.error('calendar event notification error', err));
  } catch (e) {
    console.error('calendar event create error', e);
    res.status(500).json({ error: 'Failed to save event' });
  }
});

router.get('/calendar/events/:id', async (req, res) => {
  try {
    const customId = parseCustomEventId(req.params.id);
    if (!customId) return res.status(404).json({ error: 'Not found' });
    const event = await calendarEventModel.getById(customId);
    if (!event) return res.status(404).json({ error: 'Not found' });
    const attendeeIds = await calendarEventModel.getAttendeeIds(customId);
    const people = await loadCalendarPeople();
    const validAttendees = attendeeIds.filter((id) => people.byId.has(id));
    const classification = classifyAttendees(validAttendees, people);
    res.json({ ok: true, event: formatCustomEventResponse(event, { ...classification, people }) });
  } catch (e) {
    console.error('calendar event fetch error', e);
    res.status(500).json({ error: 'Failed to load event' });
  }
});

router.put('/calendar/events/:id', async (req, res) => {
  try {
    const customId = parseCustomEventId(req.params.id);
    if (!customId) return res.status(404).json({ error: 'Not found' });
    const existing = await calendarEventModel.getById(customId);
    if (!existing) return res.status(404).json({ error: 'Not found' });

    const payload = parseCalendarEventBody(req.body);
    if (payload.error) {
      return res.status(400).json({ error: payload.error });
    }

    const {
      adminIds = [],
      teacherIds = [],
      title,
      description,
      start,
      end,
      allDay
    } = payload;

    const previousIds = await calendarEventModel.getAttendeeIds(customId);
    const updated = await calendarEventModel.update(customId, { title, description, start, end, allDay });
    const attendeeSync = await syncCustomEventAttendees(customId, adminIds, teacherIds, previousIds);
    const eventResponse = formatCustomEventResponse(updated, attendeeSync.classification);
    res.json({ ok: true, event: eventResponse });

    const origin = `${req.protocol}://${req.get('host')}`;
    if (attendeeSync.addedUsers.length) {
      sendEventAssignmentEmails(eventResponse, attendeeSync.addedUsers, origin, req.session.user)
        .catch((err) => console.error('calendar event notification error', err));
    }
  } catch (e) {
    console.error('calendar event update error', e);
    res.status(500).json({ error: 'Failed to update event' });
  }
});

router.delete('/calendar/events/:id', async (req, res) => {
  try {
    const customId = parseCustomEventId(req.params.id);
    if (!customId) return res.status(404).json({ error: 'Not found' });
    const removed = await calendarEventModel.remove(customId);
    if (!removed) return res.status(404).json({ error: 'Not found' });
    res.json({ ok: true });
  } catch (e) {
    console.error('calendar event delete error', e);
    res.status(500).json({ error: 'Failed to delete event' });
  }
});

router.get('/calendar/ics', async (req, res) => {
  try {
    const now = new Date();
    const start = new Date(now.getFullYear() - 1, now.getMonth(), now.getDate());
    const end = new Date(now.getFullYear() + 1, now.getMonth(), now.getDate(), 23, 59, 59, 999);
    const items = await buildCalendarItems(start, end);

    const origin = `${req.protocol}://${req.get('host')}`;
    const lines = [
      'BEGIN:VCALENDAR',
      'VERSION:2.0',
      'PRODID:-//MD Technical School LMS//EN',
      'CALSCALE:GREGORIAN'
    ];

    const formatUtc = (value) => {
      const d = value instanceof Date ? value : new Date(value);
      if (Number.isNaN(d.getTime())) return null;
      return `${d.getUTCFullYear()}${pad2(d.getUTCMonth() + 1)}${pad2(d.getUTCDate())}T${pad2(d.getUTCHours())}${pad2(d.getUTCMinutes())}${pad2(d.getUTCSeconds())}Z`;
    };

    const formatDateOnly = (value) => {
      if (!value) return null;
      const d = value instanceof Date ? value : new Date(value);
      if (Number.isNaN(d.getTime())) return null;
      return `${d.getFullYear()}${pad2(d.getMonth() + 1)}${pad2(d.getDate())}`;
    };

    const parseDateKey = (key) => {
      if (!key) return null;
      if (key instanceof Date) return new Date(key);
      const normalized = typeof key === 'string' ? key : String(key);
      const parts = normalized.split('-').map((p) => Number(p));
      if (parts.length !== 3 || parts.some((n) => Number.isNaN(n))) return null;
      return new Date(parts[0], parts[1] - 1, parts[2]);
    };

    for (const item of items) {
      const uid = `${item.id || `anon-${Math.random().toString(36).slice(2)}`}@mdts-apps.com`;
      const summary = (item.title || 'Calendar Item').replace(/\r?\n/g, ' ');
      const description = (item.description || '').replace(/\r?\n/g, '\\n');
      const url = item.link ? `${origin}${item.link}` : `${origin}/admin`;

      lines.push('BEGIN:VEVENT');
      if (item.allDay) {
        const startDate = parseDateKey(item.startDate) || (item.start ? new Date(item.start) : null);
        const endDateInclusive = parseDateKey(item.endDate) || startDate;
        if (startDate) {
          lines.push(`DTSTART;VALUE=DATE:${formatDateOnly(startDate)}`);
        }
        if (endDateInclusive) {
          const exclusiveEnd = new Date(endDateInclusive);
          exclusiveEnd.setDate(exclusiveEnd.getDate() + 1);
          lines.push(`DTEND;VALUE=DATE:${formatDateOnly(exclusiveEnd)}`);
        }
      } else {
        const startUtc = formatUtc(item.start);
        const endUtc = formatUtc(item.end || item.start);
        if (startUtc) lines.push(`DTSTART:${startUtc}`);
        if (endUtc) lines.push(`DTEND:${endUtc}`);
      }
      lines.push(`UID:${uid}`);
      lines.push(`SUMMARY:${summary}`);
      if (description) lines.push(`DESCRIPTION:${description}`);
      if (url) lines.push(`URL:${url}`);
      lines.push('END:VEVENT');
    }

    lines.push('END:VCALENDAR');

    res.setHeader('Content-Type', 'text/calendar; charset=utf-8');
    res.setHeader('Content-Disposition', 'attachment; filename="mdts-admin-calendar.ics"');
    res.send(lines.join('\r\n'));
  } catch (e) {
    console.error('calendar ics export error', e);
    res.status(500).send('Failed to generate calendar');
  }
});

router.get('/notifications', async (req, res) => {
  try {
    const [admins, teachers, settings] = await Promise.all([
      userModel.listAdmins(),
      userModel.listTeachers(),
      notificationSettings.loadSettings()
    ]);
    const staffMembers = [...admins, ...teachers];
    res.render('admin_notifications', {
      user: req.session.user,
      admins: staffMembers,
      settings,
      events: notificationEvents,
      saved: req.query.saved === '1'
    });
  } catch (err) {
    console.error('notifications page load failed', err);
    res.status(500).render('admin_notifications', {
      user: req.session.user,
      admins: [],
      settings: notificationSettings.DEFAULTS,
      events: notificationEvents,
      saved: false,
      loadError: 'Unable to load notification preferences right now.'
    });
  }
});

router.post('/notifications', async (req, res) => {
  const keys = notificationEvents.map((evt) => evt.key);
  try {
    const payload = {};
    for (const key of keys) {
      const raw = req.body[key];
      const list = Array.isArray(raw) ? raw : (raw ? [raw] : []);
      payload[key] = list
        .map((value) => Number(value))
        .filter((id) => Number.isInteger(id) && id > 0);
    }
    await notificationSettings.saveSettings(payload);
    res.redirect('/admin/notifications?saved=1');
  } catch (err) {
    console.error('notifications update failed', err);
    const [admins, teachers, settings] = await Promise.all([
      userModel.listAdmins().catch(() => []),
      userModel.listTeachers().catch(() => []),
      notificationSettings.loadSettings().catch(() => notificationSettings.DEFAULTS)
    ]);
    const staffMembers = [...admins, ...teachers];
    res.status(500).render('admin_notifications', {
      user: req.session.user,
      admins: staffMembers,
      settings,
      events: notificationEvents,
      saved: false,
      loadError: 'Could not save your changes. Please try again.'
    });
  }
});

// Single lead profile page
router.get('/leads/:id', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const lead = await leadModel.getById(id);
    if (!lead) return res.status(404).send('Lead not found');
    const contacts = await leadModel.getContacts(id);
    
    // Fetch pre-registration data if email exists
    let preRegistration = null;
    let student = null;
    let adminNotes = [];
    if (lead.email) {
      preRegistration = await preRegModel.findByEmail(lead.email);
      // Check if there's a student account with this email
      student = await userModel.findByEmail(lead.email);
      // Get admin notes if student exists
      if (student && student.profile && student.profile.adminNotes) {
        adminNotes = student.profile.adminNotes;
      }
    }
    
    res.render('lead_profile', { user: req.session.user, lead, contacts, preRegistration, student, adminNotes, success: req.query.success });
  } catch (e) {
    console.error('Lead view error', e);
    res.status(500).send('Error loading lead');
  }
});

// Upload documents to lead's student account
const leadDocsUpload = multer({
  storage: multer.diskStorage({
    destination: (req, file, cb) => cb(null, path.join(__dirname, '../uploads')),
    filename: (req, file, cb) => {
      const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
      cb(null, 'lead-doc-' + uniqueSuffix + path.extname(file.originalname));
    }
  }),
  limits: { fileSize: 15 * 1024 * 1024 }, // 15MB limit
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png', '.doc', '.docx'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type. Only PDF, JPG, PNG, DOC, and DOCX files are allowed.'));
  }
}).array('documents', 10); // Allow up to 10 files

router.post('/leads/:id/upload-documents', (req, res) => {
  leadDocsUpload(req, res, async (err) => {
    const leadId = Number(req.params.id);
    
    if (err) {
      console.error('Document upload error:', err);
      return res.redirect(`/admin/leads/${leadId}?error=${encodeURIComponent(err.message)}`);
    }

    try {
      // Get the lead
      const lead = await leadModel.getById(leadId);
      if (!lead || !lead.email) {
        return res.redirect(`/admin/leads/${leadId}?error=${encodeURIComponent('Lead not found or has no email')}`);
      }

      // Find the student account by email
      const student = await userModel.findByEmail(lead.email);
      if (!student) {
        // Clean up uploaded files since no student account exists
        if (req.files && req.files.length > 0) {
          req.files.forEach(file => {
            try {
              const fs = require('fs');
              fs.unlinkSync(file.path);
            } catch (e) {
              console.error('Failed to delete file:', e);
            }
          });
        }
        return res.redirect(`/admin/leads/${leadId}?error=${encodeURIComponent('No student account found for this email address')}`);
      }

      // Process uploaded documents
      if (req.files && req.files.length > 0) {
        const label = req.body.label || 'Document';
        const uploads = req.files.map(file => ({
          originalName: file.originalname,
          mimeType: file.mimetype,
          size: file.size,
          url: `/uploads/${file.filename}`,
          label: label,
          uploadedBy: 'admin',
          uploadedAt: new Date().toISOString()
        }));

        // Add documents to student profile
        await userModel.addUploads(student.id, uploads);
        
        return res.redirect(`/admin/leads/${leadId}?success=1`);
      } else {
        return res.redirect(`/admin/leads/${leadId}?error=${encodeURIComponent('No files uploaded')}`);
      }
    } catch (e) {
      console.error('Failed to upload documents to student profile:', e);
      return res.redirect(`/admin/leads/${leadId}?error=${encodeURIComponent('Failed to upload documents')}`);
    }
  });
});


router.post('/announcements', async (req, res) => {
  const { message, audience } = req.body;
  if (message && audience) {
    await announcementModel.create({ authorId: req.session.user.id, audience, message });
  }
  res.redirect('/admin/announcements');
});

router.post('/announcements/:id/delete', async (req, res) => {
  const id = Number(req.params.id);
  if (!Number.isNaN(id)) {
    await announcementModel.remove(id);
  }
  res.redirect('/admin/announcements');
});

router.post('/classes/:id/tests/media', mediaUpload.single('media'), (req, res) => {
  const classId = Number(req.params.id);
  if (!req.file) return res.status(400).send('No file uploaded');
  res.redirect(`/admin/classes/${classId}#tests`);
});


router.post('/decline/:id', async (req, res) => {
  await userModel.setStatus(Number(req.params.id), 'declined');
  res.redirect('/admin/students/pending');
});

// Student profile - with automatic alumni status check
router.get('/students/:id', checkAlumniMiddleware, async (req, res) => {
  req.time && req.time('Start');
  
  const student = await userModel.findById(Number(req.params.id));
  if (!student) return res.status(404).send('Not found');
  req.time && req.time('findById');
  
  // Lead info for notes/contact metrics
  let lead = null, leadContacts = [];
  try {
    if (student.email) {
      lead = await require('../models/leadModel').getByEmail(student.email);
      if (lead) {
        leadContacts = await require('../models/leadModel').getContacts(lead.id);
      }
    }
  } catch (e) { console.error('Lead fetch failed', e); }
  req.time && req.time('Lead data');
  
  // Fetch signature documents for this student
  let signatureDocuments = [];
  try {
    const signatureDocumentModel = require('../models/signatureDocumentModel');
    signatureDocuments = await signatureDocumentModel.getDocumentsByStudent(student.id);
  } catch (e) { console.error('Signature docs fetch failed', e); }
  req.time && req.time('Signature docs');
  
  // Get all classes where this student is enrolled
  const allClasses = await classModel.getAllClasses();
  req.time && req.time('getAllClasses');
  
  const studentClasses = allClasses
    .filter(c => (c.studentIds || []).includes(student.id))
    .sort((a, b) => {
      const dateA = new Date(a.startDate);
      const dateB = new Date(b.startDate);
      return dateA - dateB; // Ascending order (oldest first)
    });
  
  // Get all available classes for enrollment dropdown
  const availableClasses = allClasses
    .filter(c => !(c.studentIds || []).includes(student.id))
    .sort((a, b) => {
      const getTime = (val) => {
        if (!val) return 0;
        const date = new Date(val);
        return Number.isNaN(date.getTime()) ? 0 : date.getTime();
      };
      return getTime(b.startDate) - getTime(a.startDate);
    });

  const todoTemplates = await todoModel.getAllTemplates();
  const studentTodoAssignments = await todoModel.getStudentAssignments(student.id);
  const teacherMap = Object.fromEntries((await userModel.getByRole('teacher')).map(t => [t.id, t.name]));
  req.time && req.time('Todos & teachers');
  
  // Get course financial information for enrolled classes
  const courseModel = require('../models/courseModel');
  const allCourses = await courseModel.getAllCourses();
  req.time && req.time('getAllCourses');
  
  // Create multiple lookup indexes for better matching
  const courseLookup = {};
  const courseByLowerName = {};
  allCourses.forEach(course => {
    const lowerName = course.program_name.toLowerCase().trim();
    courseLookup[course.program_name] = course;
    courseByLowerName[lowerName] = course;
    // Also index by words in the name for partial matching
    const words = lowerName.split(/[\s\/\-]+/);
    words.forEach(word => {
      if (word.length > 3 && !courseByLowerName[word]) {
        courseByLowerName[word] = course;
      }
    });
  });
  
  // Build tuition breakdown by matching class names to courses
  const tuitionBreakdown = [];
  let totalTuition = 0;
  
  studentClasses.forEach(klass => {
    let matchingCourse = null;
    
    // Try 1: Exact match by full name
    matchingCourse = courseLookup[klass.name];
    
    // Try 2: Case-insensitive match by full name
    if (!matchingCourse) {
      matchingCourse = courseByLowerName[klass.name.toLowerCase().trim()];
    }
    
    // Try 3: Match by shortName (badge like "HEL 101")
    if (!matchingCourse && klass.shortName) {
      matchingCourse = courseByLowerName[klass.shortName.toLowerCase().trim()];
    }
    
    // Try 4: Fuzzy match - check if course name contains class name or vice versa
    if (!matchingCourse) {
      const className = klass.name.toLowerCase().trim();
      const classShortName = (klass.shortName || '').toLowerCase().trim();
      
      for (const course of allCourses) {
        const courseName = course.program_name.toLowerCase().trim();
        
        // Check if course name contains class name (e.g., "Risk Management Framework" contains "Risk Management")
        if (courseName.includes(className) || className.includes(courseName)) {
          matchingCourse = course;
          break;
        }
        
        // Check shortName match
        if (classShortName && (courseName.includes(classShortName) || classShortName.includes(courseName))) {
          matchingCourse = course;
          break;
        }
        
        // Check if significant words match
        const classWords = className.split(/[\s\/\-]+/).filter(w => w.length > 3);
        const courseWords = courseName.split(/[\s\/\-]+/).filter(w => w.length > 3);
        const matchingWords = classWords.filter(cw => courseWords.includes(cw));
        
        if (matchingWords.length >= 2 || (matchingWords.length === 1 && classWords.length === 1)) {
          matchingCourse = course;
          break;
        }
      }
    }
    
    if (matchingCourse) {
      const cost = parseFloat(matchingCourse.program_cost) || 0;
      totalTuition += cost;
      tuitionBreakdown.push({
        courseName: matchingCourse.program_name,
        cost: cost,
        hours: matchingCourse.program_hours || 0,
        className: klass.name,
        classShortName: klass.shortName || '',
        classId: klass.id,
        isManual: false
      });
    }
  });
  
  req.time && req.time('Course matching');
  
  // Add manual tuition courses
  const manualTuitionCourses = student.profile?.manualTuitionCourses || [];
  manualTuitionCourses.forEach((manual, index) => {
    const cost = parseFloat(manual.cost) || 0;
    totalTuition += cost;
    tuitionBreakdown.push({
      courseName: manual.courseName,
      cost: cost,
      hours: manual.clockHours || 0,
      className: '',
      classShortName: '',
      classId: null,
      isManual: true,
      manualIndex: index,
      addedAt: manual.addedAt,
      addedBy: manual.addedBy
    });
  });
  
  const registrationFee = 75.00;
  const totalCost = totalTuition + registrationFee;
  
  req.time && req.time('Before render');
  
  res.render('student_profile', { 
    student, 
    role: 'admin', 
    reset: req.query.reset, 
    signatureDocsConfig, 
    lead, 
    leadContacts, 
    signatureDocuments,
    studentClasses,
    allClasses,
    availableClasses,
    todoTemplates,
    studentTodoAssignments,
    teacherMap,
    tuitionBreakdown,
    registrationFee,
    totalTuition,
    totalCost,
    flashMessage: req.query.message || null,
    flashError: req.query.error || null
  });
});

router.post('/students/:id/todo-assign', async (req, res) => {
  const studentId = Number(req.params.id);
  const templateId = Number(req.body.templateId || 0);
  if (!templateId) {
    return res.redirect(`/admin/students/${studentId}?error=Invalid%20template`);
  }
  try {
    await todoModel.assignTemplateToStudent(studentId, templateId);
    res.redirect(`/admin/students/${studentId}?message=Checklist%20assigned`);
  } catch (e) {
    console.error('Assign todo template error', e);
    res.redirect(`/admin/students/${studentId}?error=${encodeURIComponent(e.message || 'Failed to assign template')}`);
  }
});

router.delete('/students/:id/todo-assign/:assignmentId', async (req, res) => {
  const studentId = Number(req.params.id);
  const assignmentId = Number(req.params.assignmentId);
  try {
    await todoModel.deleteStudentAssignment(studentId, assignmentId);
    res.json({ ok: true });
  } catch (e) {
    console.error('Delete todo assignment error', e);
    res.status(400).json({ ok: false, error: e.message || 'Failed to delete checklist' });
  }
});

router.post('/students/:id/todo-items/:itemId/toggle', async (req, res) => {
  const studentId = Number(req.params.id);
  const itemId = Number(req.params.itemId);
  const completed = req.body.completed === '1';
  const wantsJSON = req.headers.accept && req.headers.accept.includes('application/json');
  try {
    const result = await todoModel.setTodoItemCompletion(studentId, itemId, completed);
    if (wantsJSON) {
      const assignments = await todoModel.getStudentAssignments(studentId);
      const assignment = assignments.find(a => a.id === result.assignmentId);
      return res.json({ ok: true, assignment, completedAt: result.completedAt, itemId });
    }
  } catch (e) {
    console.error('Toggle todo item error', e);
    if (wantsJSON) {
      return res.status(400).json({ ok: false, error: e.message || 'Failed to update checklist item' });
    }
  }
  if (!wantsJSON) res.redirect(`/admin/students/${studentId}`);
});

router.post('/students/:id/todo-items', async (req, res) => {
  const studentId = Number(req.params.id);
  const assignmentId = Number(req.body.assignmentId || 0);
  const parentItemId = req.body.parentItemId ? Number(req.body.parentItemId) : null;
  const { title } = req.body || {};
  const wantsJSON = req.headers.accept && req.headers.accept.includes('application/json');
  if (!assignmentId) {
    if (wantsJSON) return res.status(400).json({ ok: false, error: 'Invalid assignment' });
    return res.redirect(`/admin/students/${studentId}?error=Invalid%20assignment`);
  }
  try {
    const item = await todoModel.addStudentTodoItem(studentId, assignmentId, title, parentItemId);
    if (wantsJSON) {
      const assignments = await todoModel.getStudentAssignments(studentId);
      const assignment = assignments.find(a => a.id === assignmentId);
      return res.json({ ok: true, item, assignment });
    }
    res.redirect(`/admin/students/${studentId}?message=Checklist%20item%20added`);
  } catch (e) {
    console.error('Add student todo item error', e);
    if (wantsJSON) {
      return res.status(400).json({ ok: false, error: e.message || 'Failed to add item' });
    }
    res.redirect(`/admin/students/${studentId}?error=${encodeURIComponent(e.message || 'Failed to add item')}`);
  }
});

// Detailed Student Report for official records
router.get('/students/:id/report', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const student = await userModel.findById(studentId);
    if (!student) return res.status(404).send('Student not found');
    
    // Get all classes
    const allClasses = await classModel.getAllClasses();
    
    // Filter classes this student is enrolled in
    const classes = allClasses.filter(klass => 
      (klass.studentIds || []).includes(studentId)
    );
    
    // Gather all grades from all classes
    const grades = [];
    for (const klass of classes) {
      const classGrades = (klass.grades || []).filter(g => g.studentId === studentId);
      
      // Map grades with class and item names
      for (const grade of classGrades) {
        let itemName = 'Unknown';
        
        if (grade.testId) {
          const test = (klass.tests || []).find(t => t.id === grade.testId);
          itemName = test ? test.title : `Test #${grade.testId}`;
        } else if (grade.assignmentId) {
          const assignment = (klass.assignments || []).find(a => a.id === grade.assignmentId);
          itemName = assignment ? assignment.title : `Assignment #${grade.assignmentId}`;
        } else if (grade.labId) {
          itemName = `Lab #${grade.labId}`;
        }
        
        grades.push({
          className: klass.name,
          classShortName: klass.shortName,
          itemName: itemName,
          ...grade
        });
      }
    }
    
    // Sort grades by date (most recent first)
    grades.sort((a, b) => {
      const dateA = a.gradedAt ? new Date(a.gradedAt) : new Date(0);
      const dateB = b.gradedAt ? new Date(b.gradedAt) : new Date(0);
      return dateB - dateA;
    });
    
    // Calculate total clock hours from enrolled classes
    let totalClockHours = 0;
    let totalClockHoursCompleted = 0;
    for (const klass of classes) {
      if (klass.clockHours) {
        totalClockHours += klass.clockHours;
      }
    }
    
    // Get attendance records
    const attendance = student.profile?.attendance || [];
    const totalAttendanceDays = attendance.length;
    
    // Group attendance by class
    const attendanceByClass = {};
    attendance.forEach(record => {
      const className = record.className || 'Unknown';
      if (!attendanceByClass[className]) {
        attendanceByClass[className] = [];
      }
      attendanceByClass[className].push(record);
    });
    
    // Create attendance summary
    const attendanceSummary = Object.keys(attendanceByClass).map(className => ({
      className,
      daysPresent: attendanceByClass[className].length
    }));
    
    // Get certificates
    const certificates = student.profile?.certificates || [];
    
    // Get certification exam results (if they exist in profile)
    const certificationExams = student.profile?.certificationExams || [];
    
    // Fetch teacher information for each class
    const classesWithTeachers = await Promise.all(classes.map(async (klass) => {
      let teacherName = 'N/A';
      let assistantTeacherName = null;
      
      if (klass.teacherId) {
        const teacher = await userModel.findById(klass.teacherId);
        if (teacher) {
          teacherName = teacher.name || teacher.username;
        }
      }
      
      if (klass.assistantTeacherId) {
        const assistant = await userModel.findById(klass.assistantTeacherId);
        if (assistant) {
          assistantTeacherName = assistant.name || assistant.username;
        }
      }
      
      return {
        ...klass,
        teacherName,
        assistantTeacherName
      };
    }));
    
    res.render('student_report', { 
      student, 
      classes: classesWithTeachers, 
      grades,
      branding,
      studentClasses: classes,
      totalClockHours,
      totalClockHoursCompleted,
      attendanceByClass,
      totalAttendanceDays,
      attendanceSummary,
      certificates,
      certificationExams
    });
  } catch (e) {
    console.error('Student report error:', e);
    res.status(500).send('Error generating report');
  }
});

// Generate and assign a Student ID for a student missing one
router.post('/students/:id/generate-student-id', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    if (!student || student.role !== 'student') return res.status(404).send('Not found');
    const current = student.profile?.studentId;
    if (!current) {
      const sid = await userModel.generateStudentId();
      await userModel.updateProfile(id, { studentId: sid });
    }
    return res.redirect(`/admin/students/${id}`);
  } catch (e) {
    console.error('Generate student ID error', e);
    return res.redirect(`/admin/students/${req.params.id}`);
  }
});

router.post('/students/:id/reset-password', async (req, res) => {
  const id = Number(req.params.id);
  const user = await userModel.findById(id);
  if (!user) return res.status(404).send('Not found');
  const newPassword = crypto.randomBytes(4).toString('hex');
  await userModel.updatePassword(user.username, newPassword);
  if (user.email) {
    try {
      const { subject, html, text } = emailTemplates.render('passwordReset', {
        name: user.name || 'User',
        newPassword
      });
      await transporter.sendMail({
        from: 'no-reply@mdts-apps.com',
        to: user.email,
        subject,
        html,
        text
      });
      try { await userModel.setLastContacted(user.id, new Date()); } catch(_) {}
    } catch (e) {
      console.error('Error sending reset email', e);
    }
  }
  res.redirect(`/admin/students/${id}?reset=1`);});

// Update student demographics (gender, race)
router.post('/students/:id/demographics', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    if (!student || student.role !== 'student') return res.status(404).send('Not found');

    const allowedGender = ['female', 'male'];
    const allowedRace = [
      'american-indian-alaska-native',
      'asian',
      'black-african-american',
      'native-hawaiian-pacific-islander',
      'white',
      'two-or-more',
      'other'
    ];

    let gender = String(req.body.gender || '');
    let race = String(req.body.race || '');
    if (!allowedGender.includes(gender)) gender = '';
    if (!allowedRace.includes(race)) race = '';

    const grievanceAcknowledged = req.body.grievanceAcknowledged === 'true' || req.body.grievanceAcknowledged === true;

    await userModel.updateProfile(id, { gender, race, grievanceAcknowledged });
    return res.redirect(`/admin/students/${id}?saved=1`);
  } catch (e) {
    console.error('update demographics error', e);
    return res.status(500).send('Failed to update demographics');
  }
});

// Manual tuition course management
router.post('/students/:id/manual-tuition', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    if (!student || student.role !== 'student') return res.status(404).send('Not found');
    
    const profile = student.profile || {};
    const manualTuitionCourses = profile.manualTuitionCourses || [];
    
    const courseName = (req.body.courseName || '').trim();
    const clockHours = Number(req.body.clockHours) || 0;
    const cost = parseFloat(req.body.cost) || 0;
    
    if (!courseName) {
      return res.redirect(`/admin/students/${id}?error=${encodeURIComponent('Course name is required')}`);
    }
    
    // Add the manual course
    manualTuitionCourses.push({
      courseName,
      clockHours,
      cost,
      addedAt: new Date().toISOString(),
      addedBy: req.session.user?.name || 'Admin'
    });
    
    profile.manualTuitionCourses = manualTuitionCourses;
    await userModel.updateProfile(id, profile);
    
    return res.redirect(`/admin/students/${id}?message=${encodeURIComponent('Manual course added')}`);
  } catch (e) {
    console.error('add manual tuition error', e);
    return res.redirect(`/admin/students/${req.params.id}?error=${encodeURIComponent('Failed to add manual course')}`);
  }
});

// Delete manual tuition course
router.post('/students/:id/manual-tuition/:index/delete', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const courseIndex = Number(req.params.index);
    const student = await userModel.findById(id);
    if (!student || student.role !== 'student') return res.status(404).send('Not found');
    
    const profile = student.profile || {};
    const manualTuitionCourses = profile.manualTuitionCourses || [];
    
    if (courseIndex >= 0 && courseIndex < manualTuitionCourses.length) {
      manualTuitionCourses.splice(courseIndex, 1);
      profile.manualTuitionCourses = manualTuitionCourses;
      await userModel.updateProfile(id, profile);
    }
    
    return res.redirect(`/admin/students/${id}?message=${encodeURIComponent('Manual course removed')}`);
  } catch (e) {
    console.error('delete manual tuition error', e);
    return res.redirect(`/admin/students/${req.params.id}?error=${encodeURIComponent('Failed to delete manual course')}`);
  }
});

// Admin uploads: signed DocuSign documents and certifications
router.post('/students/:id/documents', (req, res) => {
  docsUpload(req, res, async (err) => {
    try {
      const id = Number(req.params.id);
      const student = await userModel.findById(id);
      if (!student || student.role !== 'student') return res.status(404).send('Not found');
      if (err) {
        console.error('Upload error', err);
        return res.status(400).send(err.message || 'Upload failed');
      }
      const profile = student.profile || {};
      const toMeta = (f) => ({ originalName: f.originalname, mimeType: f.mimetype, size: f.size, url: `/docs/stxd/${f.filename}` });
      const docFiles = Array.isArray(req.files?.docFiles) ? req.files.docFiles.map(toMeta) : [];
      const certFiles = Array.isArray(req.files?.certFiles) ? req.files.certFiles.map(toMeta) : [];
      const signedDocs = Array.isArray(profile.signedDocs) ? profile.signedDocs : [];
      const certifications = Array.isArray(profile.certifications) ? profile.certifications : [];
      if (docFiles.length) signedDocs.push(...docFiles);
      if (certFiles.length) certifications.push(...certFiles);
      await userModel.updateProfile(id, { signedDocs, certifications });
      return res.redirect(`/admin/students/${id}`);
    } catch (e) {
      console.error('documents upload error', e);
      return res.status(500).send('Failed to save documents');
    }
  });
});

// Admin uploads documents that the student must download, sign, and return
router.post('/students/:id/to-sign', (req, res) => {
  toSignUpload(req, res, async (err) => {
    try {
      const id = Number(req.params.id);
      const student = await userModel.findById(id);
      if (!student || student.role !== 'student') return res.status(404).send('Not found');
      if (err) {
        console.error('To-sign upload error', err);
        return res.status(400).send(err.message || 'Upload failed');
      }
      const label = (req.body.label || '').toString().trim();
      const notes = (req.body.notes || '').toString().trim();
      const nowIso = new Date().toISOString();
      const files = Array.isArray(req.files) ? req.files : [];
      if (!files.length) return res.redirect(`/admin/students/${id}`);
      const profile = student.profile || {};
      const toSignDocs = Array.isArray(profile.toSignDocs) ? profile.toSignDocs : [];
      const rng = () => Math.random().toString(36).slice(2, 8);
      files.forEach(f => {
        toSignDocs.push({
          id: `${Date.now()}-${rng()}`,
          originalName: f.originalname,
          mimeType: f.mimetype,
          size: f.size,
          url: `/docs/stxd/${f.filename}`,
          createdAt: nowIso,
          sentAt: nowIso,
          completedAt: null,
          studentFile: null,
          label: label || f.originalname,
          notes: notes || ''
        });
      });
      await userModel.updateProfile(id, { toSignDocs });
      // Notify student by email (best-effort)
      if (student.email) {
        try {
          await transporter.sendMail({
            from: 'noreply@mdts-apps.com',
            to: student.email,
            subject: 'New document(s) awaiting your signature',
            text: 'Please log in to your student portal to download, sign, and upload the required document(s).',
            html: '<p>Please log in to your student portal to download, sign, and upload the required document(s).</p>'
          });
          try { await userModel.setLastContacted(id, new Date()); } catch(_) {}
        } catch (e) { console.error('Notify student email failed', e); }
      }
      return res.redirect(`/admin/students/${id}`);
    } catch (e) {
      console.error('to-sign save error', e);
      return res.status(500).send('Failed to save to-sign documents');
    }
  });
});

// Upload general documents to student profile
router.post('/students/:id/upload-doc', (req, res) => {
  generalUpload(req, res, async (err) => {
    try {
      const id = Number(req.params.id);
      const student = await userModel.findById(id);
      if (!student || student.role !== 'student') return res.status(404).send('Not found');
      if (err) {
        console.error('Document upload error', err);
        return res.status(400).send(err.message || 'Upload failed');
      }
      const files = Array.isArray(req.files) ? req.files : [];
      if (!files.length) return res.redirect(`/admin/students/${id}`);
      
      const profile = student.profile || {};
      const uploads = Array.isArray(profile.uploads) ? profile.uploads : [];
      const nowIso = new Date().toISOString();
      
      files.forEach(f => {
        uploads.push({
          originalName: f.originalname,
          mimeType: f.mimetype,
          size: f.size,
          url: `/docs/stxd/${f.filename}`,
          uploadedAt: nowIso
        });
      });
      
      await userModel.updateProfile(id, { uploads });
      return res.redirect(`/admin/students/${id}`);
    } catch (e) {
      console.error('Document upload save error', e);
      return res.status(500).send('Failed to save uploaded documents');
    }
  });
});

// Delete uploaded document
router.delete('/students/:id/delete-doc/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const docIndex = Number(req.params.index);
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const profile = student.profile || {};
    const uploads = Array.isArray(profile.uploads) ? profile.uploads : [];
    
    if (docIndex < 0 || docIndex >= uploads.length) {
      return res.status(400).send('Invalid document index');
    }
    
    // Remove the document from the array
    const deletedDoc = uploads.splice(docIndex, 1)[0];
    
    // Update the profile with the new uploads array
    await userModel.updateProfile(studentId, { uploads });
    
    // Optional: Delete the physical file from the filesystem
    if (deletedDoc && deletedDoc.url) {
      const fs = require('fs');
      const path = require('path');
      const filePath = path.join(__dirname, '..', 'public', deletedDoc.url);
      
      fs.unlink(filePath, (err) => {
        if (err) {
          console.warn('Failed to delete physical file:', filePath, err);
          // Don't fail the request if file deletion fails
        } else {
          console.log('Deleted physical file:', filePath);
        }
      });
    }
    
    return res.status(200).json({ success: true, message: 'Document deleted successfully' });
  } catch (error) {
    console.error('Error deleting document:', error);
    return res.status(500).send('Failed to delete document');
  }
});

// Delete signed document
router.delete('/students/:id/delete-signed-doc/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const docIndex = Number(req.params.index);
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const profile = student.profile || {};
    const signedDocs = Array.isArray(profile.signedDocs) ? profile.signedDocs : [];
    
    if (docIndex < 0 || docIndex >= signedDocs.length) {
      return res.status(400).send('Invalid document index');
    }
    
    // Remove the document from the array
    const deletedDoc = signedDocs.splice(docIndex, 1)[0];
    
    // Update the profile with the new signedDocs array
    await userModel.updateProfile(studentId, { signedDocs });
    
    // Optional: Delete the physical file from the filesystem
    if (deletedDoc && deletedDoc.url) {
      const fs = require('fs');
      const path = require('path');
      const filePath = path.join(__dirname, '..', 'public', deletedDoc.url);
      
      fs.unlink(filePath, (err) => {
        if (err) {
          console.warn('Failed to delete physical file:', filePath, err);
        } else {
          console.log('Deleted physical file:', filePath);
        }
      });
    }
    
    return res.status(200).json({ success: true, message: 'Signed document deleted successfully' });
  } catch (error) {
    console.error('Error deleting signed document:', error);
    return res.status(500).send('Failed to delete signed document');
  }
});

// Delete certification
router.delete('/students/:id/delete-certification/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const certIndex = Number(req.params.index);
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const profile = student.profile || {};
    const certifications = Array.isArray(profile.certifications) ? profile.certifications : [];
    
    if (certIndex < 0 || certIndex >= certifications.length) {
      return res.status(400).send('Invalid certification index');
    }
    
    // Remove the certification from the array
    const deletedCert = certifications.splice(certIndex, 1)[0];
    
    // Update the profile with the new certifications array
    await userModel.updateProfile(studentId, { certifications });
    
    // Optional: Delete the physical file from the filesystem
    if (deletedCert && deletedCert.url) {
      const fs = require('fs');
      const path = require('path');
      const filePath = path.join(__dirname, '..', 'public', deletedCert.url);
      
      fs.unlink(filePath, (err) => {
        if (err) {
          console.warn('Failed to delete physical file:', filePath, err);
        } else {
          console.log('Deleted physical file:', filePath);
        }
      });
    }
    
    return res.status(200).json({ success: true, message: 'Certification deleted successfully' });
  } catch (error) {
    console.error('Error deleting certification:', error);
    return res.status(500).send('Failed to delete certification');
  }
});

// Upload required documents (ID and transcript) - Admin only
router.post('/students/:id/required-documents', (req, res) => {
  requiredDocsUpload(req, res, async (err) => {
    try {
      const id = Number(req.params.id);
      const student = await userModel.findById(id);
      if (!student || student.role !== 'student') return res.status(404).send('Student not found');
      if (err) {
        console.error('Upload error', err);
        return res.status(400).send(err.message || 'Upload failed');
      }

      const profile = student.profile || {};
      const updateData = {};

      // Process ID document
      if (req.files?.idDocument && req.files.idDocument.length > 0) {
        const idFile = req.files.idDocument[0];
        updateData.idDocument = `/docs/stxd/${idFile.filename}`;
        console.log('ID Document uploaded:', updateData.idDocument);
      }

      // Process transcript document
      if (req.files?.transcriptDocument && req.files.transcriptDocument.length > 0) {
        const transcriptFile = req.files.transcriptDocument[0];
        updateData.transcriptDocument = `/docs/stxd/${transcriptFile.filename}`;
        console.log('Transcript uploaded:', updateData.transcriptDocument);
      }

      // Only update if at least one file was uploaded
      if (Object.keys(updateData).length > 0) {
        await userModel.updateProfile(id, updateData);
        console.log('Required documents updated for student', id);
      }

      return res.redirect(`/admin/students/${id}`);
    } catch (e) {
      console.error('Required documents upload error', e);
      return res.status(500).send('Failed to save documents');
    }
  });
});

// Delete required document (ID or transcript) - Admin only
router.delete('/students/:id/delete-required-doc/:docType', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const docType = req.params.docType; // 'idDocument' or 'transcriptDocument'
    
    // Validate document type
    if (docType !== 'idDocument' && docType !== 'transcriptDocument') {
      return res.status(400).send('Invalid document type');
    }

    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const profile = student.profile || {};
    const documentUrl = profile[docType];
    
    if (!documentUrl) {
      return res.status(404).send('Document not found');
    }

    // Remove the document from profile
    const updateData = {};
    updateData[docType] = null;
    await userModel.updateProfile(studentId, updateData);
    
    // Delete the physical file from the filesystem
    const fs = require('fs');
    const path = require('path');
    const filePath = path.join(__dirname, '..', 'public', documentUrl);
    
    fs.unlink(filePath, (err) => {
      if (err) {
        console.warn('Failed to delete physical file:', filePath, err);
      } else {
        console.log('Deleted physical file:', filePath);
      }
    });
    
    return res.status(200).json({ success: true, message: 'Document deleted successfully' });
  } catch (error) {
    console.error('Error deleting required document:', error);
    return res.status(500).send('Failed to delete document');
  }
});

// Upload earned certificates - Admin only
router.post('/students/:id/certificates', (req, res) => {
  certificatesUpload(req, res, async (err) => {
    try {
      const id = Number(req.params.id);
      const student = await userModel.findById(id);
      if (!student || student.role !== 'student') return res.status(404).send('Student not found');
      if (err) {
        console.error('Certificate upload error', err);
        return res.status(400).send(err.message || 'Upload failed');
      }

      const profile = student.profile || {};
      const certificates = Array.isArray(profile.certificates) ? profile.certificates : [];
      
      // Process uploaded certificates
      if (req.files && req.files.length > 0) {
        const toMeta = (f) => ({
          originalName: f.originalname,
          mimeType: f.mimetype,
          size: f.size,
          url: `/docs/stxd/${f.filename}`,
          uploadedAt: new Date().toISOString()
        });
        
        const newCertificates = req.files.map(toMeta);
        certificates.push(...newCertificates);
        
        await userModel.updateProfile(id, { certificates });
        console.log(`Uploaded ${newCertificates.length} certificate(s) for student ${id}`);
      }

      return res.redirect(`/admin/students/${id}`);
    } catch (e) {
      console.error('Certificate upload error', e);
      return res.status(500).send('Failed to upload certificates');
    }
  });
});

// Delete earned certificate - Admin only
router.delete('/students/:id/delete-certificate/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const certIndex = Number(req.params.index);
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const profile = student.profile || {};
    const certificates = Array.isArray(profile.certificates) ? profile.certificates : [];
    
    if (certIndex < 0 || certIndex >= certificates.length) {
      return res.status(400).send('Invalid certificate index');
    }
    
    // Remove the certificate from the array
    const deletedCert = certificates.splice(certIndex, 1)[0];
    
    // Update the profile with the new certificates array
    await userModel.updateProfile(studentId, { certificates });
    
    // Delete the physical file from the filesystem
    if (deletedCert && deletedCert.url) {
      const fs = require('fs');
      const path = require('path');
      const filePath = path.join(__dirname, '..', 'public', deletedCert.url);
      
      fs.unlink(filePath, (err) => {
        if (err) {
          console.warn('Failed to delete physical file:', filePath, err);
        } else {
          console.log('Deleted physical file:', filePath);
        }
      });
    }
    
    return res.status(200).json({ success: true, message: 'Certificate deleted successfully' });
  } catch (error) {
    console.error('Error deleting certificate:', error);
    return res.status(500).send('Failed to delete certificate');
  }
});

// Send certificate via email
router.post('/students/:id/send-certificate', async (req, res) => {
  try {
    const { id } = req.params;
    const { classId, className, certIndex } = req.body;
    
    const student = await userModel.findById(Number(id));
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }

    if (!student.email) {
      return res.status(400).json({ error: 'Student has no email address on file' });
    }

    // Get the student's certificates
    const profile = student.profile || {};
    const certificates = Array.isArray(profile.certificates) ? profile.certificates : [];
    
    if (!certificates.length) {
      return res.status(404).json({ error: 'No certificates found for this student' });
    }

    // Use specific certificate if index provided, otherwise use most recent
    const targetCert = certIndex !== undefined && certificates[certIndex] 
      ? certificates[certIndex] 
      : certificates[certificates.length - 1];
    
    const certPath = path.join(__dirname, '..', 'public', targetCert.url);

    if (!fs.existsSync(certPath)) {
      return res.status(404).json({ error: 'Certificate file not found' });
    }

    // Send email with certificate attachment
    const emailSubject = `Course Completion Certificate - ${className || 'Your Course'}`;
    const emailBody = `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
        <h2 style="color: #333;">Congratulations, ${student.name}!</h2>
        <p>We are pleased to provide you with your course completion certificate for <strong>${className || 'your course'}</strong>.</p>
        <p>Please find your certificate attached to this email.</p>
        <p>This certificate recognizes your hard work and dedication in completing the program.</p>
        <hr style="border: none; border-top: 1px solid #eee; margin: 20px 0;">
        <p style="color: #666; font-size: 14px;">
          <strong>MD Technical School</strong><br>
          Excellence in Education
        </p>
      </div>
    `;

    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      subject: emailSubject,
      html: emailBody,
      attachments: [{
        filename: targetCert.originalName || 'certificate.pdf',
        path: certPath
      }]
    });

    console.log(`Certificate sent to ${student.email} for student ${id}`);
    res.json({ ok: true, message: 'Certificate sent successfully' });

  } catch (e) {
    console.error('Send certificate error:', e);
    res.status(500).json({ error: 'Failed to send certificate', details: e.message });
  }
});

// Helper function to convert data URL to buffer
function dataUrlToBuffer(url) {
  const match = /^data:.+;base64,(.+)$/.exec(url || '');
  return match ? Buffer.from(match[1], 'base64') : null;
}

// Helper function to build completion certificate PDF
function buildCompletionPdf({ studentName, courseName, dateStr, instructorSig, directorSig }) {
  return new Promise(resolve => {
    const doc = new PDFDocument({ size: 'LETTER', layout: 'landscape', margin: 50 });
    const buffers = [];
    doc.on('data', buffers.push.bind(buffers));
    doc.on('end', () => resolve(Buffer.concat(buffers)));

    const { width, height } = doc.page;

    // simple border background so text overlays an image-like backdrop
    doc.rect(0, 0, width, height).fill('#fff');
    doc.lineWidth(4).strokeColor('#0d274d').rect(20, 20, width - 40, height - 40).stroke();

    doc.fontSize(32).fillColor('#0d274d').text('CERTIFICATE OF COMPLETION', 0, 80, { align: 'center' });
    doc.moveDown(0.5);
    doc.fontSize(14).fillColor('black').text('This is to certify that', { align: 'center' });

    // name on a line
    const nameY = doc.y + 40;
    doc.moveTo(100, nameY).lineTo(width - 100, nameY).stroke('#0d274d');
    doc.fontSize(24).fillColor('black').text(studentName, 0, nameY - 22, { align: 'center' });

    doc.moveDown();
    doc.fontSize(14).text('has successfully completed', { align: 'center' });
    doc.fontSize(18).fillColor('#0d274d').text(courseName, { align: 'center' });
    doc.moveDown(0.5);
    doc.fontSize(14).fillColor('black').text(`on ${dateStr}`, { align: 'center' });

    const y = height - 150;
    if (instructorSig) {
      const buf = dataUrlToBuffer(instructorSig);
      if (buf) {
        try { doc.image(buf, 100, y, { width: 150 }); } catch (_) {}
      }
      doc.text('Instructor', 100, y + 70, { width: 150, align: 'center' });
    }
    if (directorSig) {
      const buf = dataUrlToBuffer(directorSig);
      if (buf) {
        try { doc.image(buf, width - 250, y, { width: 150 }); } catch (_) {}
      }
      doc.text('Director', width - 250, y + 70, { width: 150, align: 'center' });
    }

    doc.end();
  });
}

// Send course completion certificate to student
router.post('/classes/:classId/students/:studentId/send-completion-certificate', async (req, res) => {
  try {
    const classId = Number(req.params.classId);
    const studentId = Number(req.params.studentId);
    const student = await userModel.findById(studentId);
    if (!student || !student.email) return res.status(404).json({ error: 'Student not found' });

    const klass = await classModel.findClassById(classId);
    const teacher = klass ? await userModel.findById(klass.teacherId) : null;
    const dateStr = new Date().toLocaleDateString();
    const directorSig = req.app.locals.branding?.directorSignature;
    const instructorSig = teacher?.profile?.signature?.url || directorSig; // Use director signature as fallback
    const pdfBuffer = await buildCompletionPdf({
      studentName: student.name || `${student.firstName || ''} ${student.lastName || ''}`.trim(),
      courseName: klass ? klass.name : 'Course',
      dateStr,
      instructorSig,
      directorSig
    });

    const admins = await userModel.getByRole('admin');
    const cc = admins.map(a => a.email).filter(Boolean);
    const link = `${req.protocol}://${req.get('host')}/student/certificates/${classId}/completion`;

    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      cc,
      subject: 'Certificate of Completion',
      html: `Congratulations ${student.name || ''}! Your certificate of completion is attached or available <a href="${link}">here</a>.`,
      text: `Congratulations ${student.name || ''}! Your certificate of completion is attached or available at ${link}`,
      attachments: [{ filename: 'certificate.pdf', content: pdfBuffer }]
    });

    console.log(`Completion certificate sent to ${student.email} for class ${classId}`);
    res.json({ ok: true, message: 'Certificate sent successfully' });

  } catch (e) {
    console.error('Send completion certificate error:', e);
    res.status(500).json({ error: 'Failed to send certificate', details: e.message });
  }
});

// Historical Grades CRUD routes
router.post('/students/:id/historical-grades', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const courseData = req.body;
    const adminName = req.session.user.name || 'Admin';
    
    const student = await userModel.findById(studentId);
    const profile = student.profile || {};
    const historicalGrades = profile.historicalGrades || [];
    
    // Generate ID
    const newId = historicalGrades.length ? Math.max(...historicalGrades.map(c => c.id || 0)) + 1 : 1;
    
    historicalGrades.push({
      id: newId,
      ...courseData,
      addedBy: adminName,
      addedAt: new Date().toISOString()
    });
    
    await userModel.updateProfile(studentId, {
      ...profile,
      historicalGrades
    });
    
    res.json({ ok: true });
  } catch (e) {
    console.error('Add historical grade error:', e);
    res.status(500).json({ error: 'Failed to add historical course' });
  }
});

router.put('/students/:id/historical-grades/:courseId', async (req, res) => {
  try {
    const { id: studentId, courseId } = req.params;
    const courseData = req.body;
    
    const student = await userModel.findById(Number(studentId));
    const profile = student.profile || {};
    const historicalGrades = profile.historicalGrades || [];
    
    const courseIndex = historicalGrades.findIndex(c => c.id === Number(courseId));
    if (courseIndex === -1) {
      return res.status(404).json({ error: 'Course not found' });
    }
    
    historicalGrades[courseIndex] = {
      ...historicalGrades[courseIndex],
      ...courseData,
      editedBy: req.session.user.name || 'Admin',
      editedAt: new Date().toISOString()
    };
    
    await userModel.updateProfile(Number(studentId), {
      ...profile,
      historicalGrades
    });
    
    res.json({ ok: true });
  } catch (e) {
    console.error('Update historical grade error:', e);
    res.status(500).json({ error: 'Failed to update historical course' });
  }
});

router.delete('/students/:id/historical-grades/:courseId', async (req, res) => {
  try {
    const { id: studentId, courseId } = req.params;
    
    const student = await userModel.findById(Number(studentId));
    const profile = student.profile || {};
    const historicalGrades = (profile.historicalGrades || []).filter(c => c.id !== Number(courseId));
    
    await userModel.updateProfile(Number(studentId), {
      ...profile,
      historicalGrades
    });
    
    res.json({ ok: true });
  } catch (e) {
    console.error('Delete historical grade error:', e);
    res.status(500).json({ error: 'Failed to delete historical course' });
  }
});

router.post('/students/:id/sign-doc', async (req, res) => {
  const id = Number(req.params.id);
  const { docType, signatureDataUrl } = req.body;
  if (!docType || !signatureDataUrl) {
    if (req.xhr || (req.headers.accept && req.headers.accept.includes('application/json'))) {
      return res.status(400).json({ success: false });
    }
    return res.redirect(`/admin/students/${id}`);
  }
  try {
    await userModel.signDocument(id, docType, signatureDataUrl);
    const user = await userModel.findById(id);
    const docs = (user.profile && user.profile.documents) || [];
    const doc = docs.find(d => d.type === docType);
    const pending = docs.filter(d => d.requiredRole === 'admin' && !d.signatureDataUrl);
    if (!pending.length && user.email) {
      try {
        const { subject, html, text } = emailTemplates.render('registrationComplete', { name: user.name });
        await transporter.sendMail({
          from: 'no-reply@mdts-apps.com',
          to: user.email,
          subject,
          html,
          text
        });
        try { await userModel.setLastContacted(user.id, new Date()); } catch(_) {}
      } catch (e) {
        console.error('Error sending completion email', e);
      }
    }
    if (req.xhr || (req.headers.accept && req.headers.accept.includes('application/json'))) {
      return res.json({ success: true, signedAt: doc?.signedAt });
    }
  } catch (e) {
    console.error('Error signing document', e);
    if (req.xhr || (req.headers.accept && req.headers.accept.includes('application/json'))) {
      return res.status(500).json({ success: false });
    }
  }
  res.redirect(`/admin/students/${id}`);
});

router.post('/students/:id/step2', async (req, res) => {
  const id = Number(req.params.id);
  const { step2Programs } = req.body;
  
  try {
    // Handle step2Programs array
    let programsArray = [];
    if (step2Programs && typeof step2Programs === 'object') {
      // Convert step2Programs object to array
      programsArray = Object.values(step2Programs).map(prog => ({
        course: prog.course?.trim() || '',
        startDate: prog.startDate?.trim() || '',
        endDate: prog.endDate?.trim() || '',
        classTime: prog.classTime?.trim() || '',
        classDays: prog.classDays?.trim() || '',
        tuition: {
          tuition: prog.tuition?.tuition || '',
          registrationFee: prog.tuition?.registrationFee || '',
          books: prog.tuition?.books || '',
          equipment: prog.tuition?.equipment || '',
          miscFees: prog.tuition?.miscFees || '',
          totalCost: prog.tuition?.totalCost || ''
        }
      })).filter(prog => prog.course); // Only include entries with a course name
    }
    
    // Update profile with step2Programs array
    await userModel.updateProfile(id, {
      step2Programs: programsArray
    });
    
    const student = await userModel.findById(id);
    if (student && student.email) {
      const token = crypto.randomBytes(32).toString('hex');
      const expires = Date.now() + 1000 * 60 * 60 * 24;
      await userModel.setStep2Token(id, token, expires);
      const link = `${req.protocol}://${req.get('host')}/step2?token=${token}`;
      const { subject, html, text } = emailTemplates.render('enrollmentStep2', { name: student.name, link });
      await transporter.sendMail({
        from: 'no-reply@mdts-apps.com',
        to: student.email,
        subject,
        html,
        text
      });
    }
  } catch (e) {
    console.error('Error in step2', e);
  }
  res.redirect(`/admin/students/${id}`);
});

// Update student profile data
router.put('/students/:id', async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    console.log('=== PUT /students/:id called ===');
    console.log('Student ID:', id);
    console.log('Request body:', JSON.stringify(req.body, null, 2));
    
    const student = await userModel.findById(id);
    
    if (!student || student.role !== 'student') {
      console.log('Student not found or not a student');
      return res.status(404).json({ error: 'Student not found' });
    }

    const {
      // Personal information
      firstName, lastName, suffix, email, username,
      address_line1, address_city, address_state, address_zip,
      phone_home, phone_cell, phone_work,
      ssn,
      emergency_name, emergency_relation, emergency_phone,
      
      // Program information (for backwards compatibility)
      course, affiliateProgram, admissionDate, startDate, endDate, classTime, classDays,
      programs, // New array-based programs
      grievanceAcknowledged,
      
      // Demographics
      gender, race,
      
      // Tuition
      tuition_tuition, tuition_registrationFee, tuition_books, 
      tuition_equipment, tuition_miscFees, tuition_totalCost
    } = req.body;

    console.log('Extracted firstName:', firstName);
    console.log('Extracted lastName:', lastName);

    // Handle programs array
    let programsArray = [];
    if (programs && typeof programs === 'object') {
      // Convert programs object to array
      programsArray = Object.values(programs).map(prog => ({
        course: prog.course?.trim() || '',
        affiliateProgram: prog.affiliateProgram?.trim() || '',
        admissionDate: prog.admissionDate?.trim() || '',
        startDate: prog.startDate?.trim() || '',
        endDate: prog.endDate?.trim() || '',
        classTime: prog.classTime?.trim() || '',
        classDays: prog.classDays?.trim() || ''
      })).filter(prog => prog.course); // Only include entries with a course name
    }

    // Build the profile update object
    const profileUpdate = {
      firstName: firstName?.trim(),
      lastName: lastName?.trim(),
      suffix: suffix?.trim() || null,
      grievanceAcknowledged: grievanceAcknowledged === 'true' || grievanceAcknowledged === true,
      
      address: {
        line1: address_line1?.trim() || '',
        city: address_city?.trim() || '',
        state: address_state?.trim() || '',
        zip: address_zip?.trim() || ''
      },
      
      phones: {
        home: phone_home?.trim() || '',
        cell: phone_cell?.trim() || '',
        work: phone_work?.trim() || ''
      },
      
      ssn: ssn?.trim() || '',
      
      emergencyContact: {
        name: emergency_name?.trim() || '',
        relation: emergency_relation?.trim() || '',
        phone: emergency_phone?.trim() || ''
      },
      
      // Use programs array if available, otherwise keep old structure for backwards compatibility
      programs: programsArray.length > 0 ? programsArray : undefined,
      
      // Keep old fields for backwards compatibility (will be ignored if programs array exists)
      course: programsArray.length === 0 ? course?.trim() : undefined,
      affiliateProgram: programsArray.length === 0 ? affiliateProgram?.trim() : undefined,
      program: programsArray.length === 0 ? {
        admissionDate: admissionDate?.trim() || '',
        startDate: startDate?.trim() || '',
        endDate: endDate?.trim() || '',
        classTime: classTime?.trim() || '',
        classDays: classDays?.trim() || ''
      } : undefined,
      
      gender: gender?.trim() || '',
      race: race?.trim() || '',
      
      tuition: {
        tuition: tuition_tuition || '',
        registrationFee: tuition_registrationFee || '',
        books: tuition_books || '',
        equipment: tuition_equipment || '',
        miscFees: tuition_miscFees || '',
        totalCost: tuition_totalCost || ''
      }
    };

    // Remove undefined values
    Object.keys(profileUpdate).forEach(key => 
      profileUpdate[key] === undefined && delete profileUpdate[key]
    );

    console.log('Profile update object:', JSON.stringify(profileUpdate, null, 2));

    // Update basic user info
    const name = `${firstName?.trim() || ''} ${lastName?.trim() || ''}`.trim();
    console.log('Updating name to:', name);
    
    if (name) {
      await userModel.updateBasicInfo(id, {
        name,
        email: email?.trim(),
        username: username?.trim()
      });
      console.log('Basic info updated');
    }

    // Update profile
    await userModel.updateProfile(id, profileUpdate);
    console.log('Profile updated');

    // Get updated student data
    const updatedStudent = await userModel.findById(id);
    console.log('Updated student retrieved:', updatedStudent.name);
    
    res.json({ success: true, student: updatedStudent });
  } catch (error) {
    console.error('Error updating student:', error);
    res.status(500).json({ error: 'Failed to update student profile' });
  }
});

// Enroll student in a class
router.post('/students/:id/enroll', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const classId = Number(req.body.classId);
    
    if (!classId) {
      return res.status(400).send('Class ID is required');
    }
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    // Allow both approved and pending students to be enrolled
    if (student.status !== 'approved' && student.status !== 'pending') {
      return res.status(403).send('Only approved or pending students can be enrolled in classes.');
    }
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).send('Class not found');
    }
    
    // Add student to class
    await classModel.addStudent(classId, studentId);
    
    console.log(`Enrolled student ${studentId} (status: ${student.status}) in class ${classId}`);
    res.redirect(`/admin/students/${studentId}`);
  } catch (error) {
    console.error('Error enrolling student:', error);
    res.status(500).send('Failed to enroll student in class');
  }
});

// Unenroll student from a class
router.post('/students/:id/unenroll/:classId', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const classId = Number(req.params.classId);
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).send('Class not found');
    }
    
    // Remove student from class
    await classModel.removeStudent(classId, studentId);
    
    console.log(`Unenrolled student ${studentId} from class ${classId}`);
    res.redirect(`/admin/students/${studentId}`);
  } catch (error) {
    console.error('Error unenrolling student:', error);
    res.status(500).send('Failed to unenroll student from class');
  }
});

// Toggle class lock for student
router.post('/students/:id/toggle-lock/:classId', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const classId = Number(req.params.classId);
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).send('Class not found');
    }
    
    // Toggle lock status
    await classModel.toggleStudentLock(classId, studentId);
    
    console.log(`Toggled lock for student ${studentId} in class ${classId}`);
    res.redirect(`/admin/students/${studentId}`);
  } catch (error) {
    console.error('Error toggling class lock:', error);
    res.status(500).send('Failed to toggle class lock');
  }
});

// Delete student
router.delete('/students/:id', async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    const student = await userModel.findById(id);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }

    // Delete the student
    await userModel.deleteById(id);
    
    res.json({ success: true, message: 'Student deleted successfully' });
  } catch (error) {
    console.error('Error deleting student:', error);
    res.status(500).json({ error: 'Failed to delete student' });
  }
});

router.get('/denied', async (req, res) => {
  const users = await userModel.getAll();
  const denied = users.filter(u => u.role === 'student' && u.status === 'declined');
  res.render('admin_denied', { denied, user: { ...(req.session.user || {}), role: req.session.role } });
});


router.get('/students', async (req, res) => {
  const users = await userModel.getAll();
  const accepted = users
    .filter(u => u.role === 'student' && u.status === 'approved')
    .sort((a, b) => {
      // Sort by acceptedAt date, newest first (descending)
      const dateA = a.acceptedAt ? new Date(a.acceptedAt).getTime() : 0;
      const dateB = b.acceptedAt ? new Date(b.acceptedAt).getTime() : 0;
      return dateB - dateA;
    });
  
  // Enrich with lead information
  let leadsByEmail = new Map();
  let preRegsByEmail = new Map();
  try {
    const leads = await leadModel.getAll();
    leadsByEmail = new Map(leads.map(l => [l.email?.toLowerCase().trim(), l]));
  } catch (e) { console.error('Load leads for accepted failed', e); }
  
  try {
    const preregs = await preRegModel.getAll();
    preRegsByEmail = new Map(preregs.map(p => [p.email?.toLowerCase().trim(), p]));
  } catch (e) { console.error('Load preregs for accepted failed', e); }
  
  // Attach lead and pre-reg data to each accepted student
  const enrichedAccepted = accepted.map(student => {
    const email = student.email?.toLowerCase().trim();
    const lead = email ? leadsByEmail.get(email) : null;
    const preReg = email ? preRegsByEmail.get(email) : null;
    return {
      ...student,
      lead,
      preReg,
      leadId: lead?.id || null
    };
  });
  
  // Build classes per student
  const classes = await classModel.getAllClasses();
  const classMap = {};
  for (const k of classes) {
    const name = k.name || k.shortName || `Class #${k.id}`;
    (k.studentIds || []).forEach(sid => {
      const id = Number(sid);
      if (!classMap[id]) classMap[id] = [];
      classMap[id].push(name);
    });
  }
  res.render('admin_accepted', { students: enrichedAccepted, classesByStudent: classMap, user: { ...(req.session.user || {}), role: req.session.role } });
});

// Contact a single pending/accepted student (AJAX)
router.post('/students/:id/contact', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    if (!student || !student.email) return res.status(404).json({ error: 'Student not found' });
    const { subject, message } = req.body || {};
    if (!subject || !message) return res.status(400).json({ error: 'Missing subject or message' });
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      subject,
      html: message,
      text: message.replace(/<[^>]*>/g, '')
    });
    const now = new Date();
    try { await userModel.setLastContacted(id, now); } catch(_) {}
    try {
      const lead = await leadModel.upsertByEmail({ name: student.name || '', email: student.email, phone: student.profile?.phones?.cell || '', interestPercent: 0, source: 'admin-contact' });
      if (lead?.id) await leadModel.addContact(lead.id, { direction: 'outbound', method: 'email', note: (subject || '').slice(0,120) });
    } catch (e) { console.error('Lead log failed', e); }
    return res.json({ ok: true, lastContacted: now.toISOString() });
  } catch (e) {
    console.error('Student contact error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

// Send welcome email to student (AJAX)
router.post('/students/:id/send-welcome', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    if (!student || !student.email) return res.status(404).json({ error: 'Student not found' });
    
    const { classId, className } = req.body || {};
    
    // Use the studentWelcome email template
    const { subject, html, text } = emailTemplates.render('studentWelcome', {
      name: student.name || 'Student',
      className: className || 'your class',
      classId: classId || ''
    });
    
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      subject,
      html,
      text
    });
    
    const now = new Date();
    
    // Update last contacted
    try { await userModel.setLastContacted(id, now); } catch(_) {}
    
    // Log interaction in lead contacts
    try {
      const lead = await leadModel.upsertByEmail({ 
        name: student.name || '', 
        email: student.email, 
        phone: student.profile?.phones?.cell || '', 
        interestPercent: 0, 
        source: 'student-welcome' 
      });
      if (lead?.id) {
        await leadModel.addContact(lead.id, { 
          direction: 'outbound', 
          method: 'email', 
          note: `Welcome email sent for ${className || 'class'}` 
        });
      }
    } catch (e) { 
      console.error('Lead log failed', e); 
    }
    
    return res.json({ ok: true, sentAt: now.toISOString() });
  } catch (e) {
    console.error('Send welcome email error', e);
    return res.status(500).json({ error: 'Failed to send welcome email' });
  }
});

// Send onboarding email with temporary password
router.post('/students/:id/onboard', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    if (!student || !student.email) return res.status(404).json({ error: 'Student not found' });

    const tempPassword = crypto.randomBytes(4).toString('hex');
    await userModel.updatePassword(student.username, tempPassword);
    await userModel.updateProfile(id, {
      mustChangePassword: true,
      onboardedAt: new Date().toISOString()
    });

    const { subject, html, text } = emailTemplates.render('studentOnboard', {
      name: student.name || 'Student',
      username: student.username,
      tempPassword
    });

    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      subject,
      html,
      text
    });

    return res.json({ ok: true });
  } catch (e) {
    console.error('Student onboard email error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

// Send program details and tuition breakdown to student
router.post('/students/:id/send-program-details', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    if (!student || !student.email) return res.status(404).json({ error: 'Student not found' });

    const profile = student.profile || {};
    const program = profile.program || {};
    const tuition = profile.tuition || {};
    
    // Format dates
    const formatDate = (dateStr) => {
      if (!dateStr) return 'TBD';
      const date = new Date(dateStr);
      return date.toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' });
    };

    // Build profile link with token for secure access
    const profileLink = `${req.protocol}://${req.get('host')}/student/profile`;

    const { subject, html, text } = emailTemplates.render('programDetails', {
      name: student.name || 'Student',
      course: profile.course || 'Not specified',
      startDate: formatDate(program.startDate),
      endDate: formatDate(program.endDate),
      classTime: program.classTime || 'TBD',
      classDays: program.classDays || 'TBD',
      tuitionTotal: tuition.total || 'Contact us for details',
      registrationFee: tuition.registrationFee || '$0',
      booksFee: tuition.booksFee || '$0',
      additionalInfo: tuition.notes || '',
      profileLink
    });

    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      subject,
      html,
      text
    });

    const now = new Date();
    try { await userModel.setLastContacted(id, now); } catch(_) {}
    
    return res.json({ ok: true, lastContacted: now.toISOString() });
  } catch (e) {
    console.error('Send program details error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

// Send message from admin (used by highlights)
router.post('/messages/send', async (req, res) => {
  try {
    const { recipientId, recipientEmail, subject, body, source } = req.body;
    
    if (!subject || !body) {
      return res.status(400).json({ error: 'Subject and body are required' });
    }
    
    const senderId = req.session.user?.id || null;
    
    // If we have a recipientId, send via message system
    if (recipientId && !isNaN(parseInt(recipientId))) {
      const messageModel = require('../models/messageModel');
      await messageModel.sendMessage(senderId, parseInt(recipientId), subject, body);
    }
    
    // If we have an email, also send via email
    if (recipientEmail) {
      const nodemailer = require('nodemailer');
      const smtpSettings = {
        host: process.env.SMTP_HOST || 'smtp.gmail.com',
        port: parseInt(process.env.SMTP_PORT) || 587,
        secure: process.env.SMTP_SECURE === 'true',
        auth: {
          user: process.env.SMTP_USER || process.env.EMAIL_USER,
          pass: process.env.SMTP_PASS || process.env.EMAIL_PASS
        }
      };
      
      try {
        const transporter = nodemailer.createTransport(smtpSettings);
        await transporter.sendMail({
          from: process.env.SMTP_FROM || process.env.EMAIL_FROM || 'noreply@mdts-apps.com',
          to: recipientEmail,
          subject: subject,
          html: body.replace(/\n/g, '<br>'),
          text: body
        });
      } catch (emailErr) {
        console.error('Email send error (non-fatal):', emailErr.message);
        // Don't fail the request if email fails but message was saved
      }
    }
    
    res.json({ ok: true, success: true });
  } catch (err) {
    console.error('Send message error:', err);
    res.status(500).json({ error: 'Failed to send message' });
  }
});

// Admin Notes routes
router.post('/students/:id/notes', async (req, res) => {
  const { id } = req.params;
  const { content } = req.body;

  if (!content || !content.trim()) {
    return res.status(400).json({ error: 'Note content is required' });
  }

  try {
    const student = await userModel.findById(id);
    if (!student) {
      return res.status(404).json({ error: 'Student not found' });
    }

    const profile = student.profile || {};
    const adminNotes = profile.adminNotes || [];

    const newNote = {
      content: content.trim(),
      createdAt: new Date().toISOString(),
      createdBy: req.session.user?.name || 'Admin'
    };

    adminNotes.push(newNote);
    profile.adminNotes = adminNotes;

    await userModel.updateProfile(id, profile);
    res.json({ ok: true });
  } catch (e) {
    console.error('Add note error:', e);
    res.status(500).json({ error: 'Failed to add note' });
  }
});

router.put('/students/:id/notes/:index', async (req, res) => {
  const { id, index } = req.params;
  const { content } = req.body;

  if (!content || !content.trim()) {
    return res.status(400).json({ error: 'Note content is required' });
  }

  try {
    const student = await userModel.findById(id);
    if (!student) {
      return res.status(404).json({ error: 'Student not found' });
    }

    const profile = student.profile || {};
    const adminNotes = profile.adminNotes || [];
    const noteIndex = parseInt(index);

    if (noteIndex < 0 || noteIndex >= adminNotes.length) {
      return res.status(404).json({ error: 'Note not found' });
    }

    adminNotes[noteIndex].content = content.trim();
    adminNotes[noteIndex].editedAt = new Date().toISOString();
    adminNotes[noteIndex].editedBy = req.session.user?.name || 'Admin';

    profile.adminNotes = adminNotes;
    await userModel.updateProfile(id, profile);
    res.json({ ok: true });
  } catch (e) {
    console.error('Update note error:', e);
    res.status(500).json({ error: 'Failed to update note' });
  }
});

router.delete('/students/:id/notes/:index', async (req, res) => {
  const { id, index } = req.params;

  try {
    const student = await userModel.findById(id);
    if (!student) {
      return res.status(404).json({ error: 'Student not found' });
    }

    const profile = student.profile || {};
    const adminNotes = profile.adminNotes || [];
    const noteIndex = parseInt(index);

    if (noteIndex < 0 || noteIndex >= adminNotes.length) {
      return res.status(404).json({ error: 'Note not found' });
    }

    adminNotes.splice(noteIndex, 1);
    profile.adminNotes = adminNotes;

    await userModel.updateProfile(id, profile);
    res.json({ ok: true });
  } catch (e) {
    console.error('Delete note error:', e);
    res.status(500).json({ error: 'Failed to delete note' });
  }
});

// Marketing email constants
const marketingTypes = ['recruitment', 'retention', 'approval', 'events', 'information'];
const marketingPrefaces = {
  recruitment: '',
  retention: '',
  approval: '',
  events: '',
  information: ''
};

// Marketing email form
router.get('/marketing', async (req, res) => {
  const students = await userModel.getByRole('student');
  const preregs = await preRegModel.getAll();
  const rsvps = await rsvpModel.getAllRSVPs();
  const courseSet = new Set();
  const programSet = new Set();
  students.forEach(s => {
    const course = s.profile?.course;
    const program = s.profile?.affiliateProgram;
    if (course) courseSet.add(course);
    if (program) programSet.add(program);
  });
  preregs.forEach(p => {
    if (p.course) courseSet.add(p.course);
  });
  const courses = Array.from(courseSet).sort();
  const programs = Array.from(programSet).sort();
  res.render('admin_marketing', {
    students,
    preregs,
    rsvps,
    courses,
    programs,
    templates: marketingSubjects,
    user: req.session.user,
    sent: req.query.sent,
    error: null
  });
});

// Todo templates management
router.get('/todo-templates', async (req, res) => {
  const templates = await todoModel.getAllTemplates();
  res.render('admin_todo_templates', {
    user: req.session.user,
    templates,
    message: req.query.message || null,
    error: req.query.error || null
  });
});

router.post('/todo-templates', async (req, res) => {
  const { name, description } = req.body || {};
  try {
    await todoModel.createTemplate(name, description);
    res.redirect('/admin/todo-templates?message=Template%20created');
  } catch (e) {
    console.error('Create template error', e);
    res.redirect(`/admin/todo-templates?error=${encodeURIComponent(e.message || 'Failed to create template')}`);
  }
});

router.post('/todo-templates/:id/items', async (req, res) => {
  const templateId = Number(req.params.id);
  const { title } = req.body || {};
  try {
    await todoModel.addTemplateItem(templateId, title);
    res.redirect('/admin/todo-templates?message=Checklist%20item%20added');
  } catch (e) {
    console.error('Add template item error', e);
    res.redirect(`/admin/todo-templates?error=${encodeURIComponent(e.message || 'Failed to add item')}`);
  }
});

router.post('/todo-templates/items/:itemId/subitems', async (req, res) => {
  const itemId = Number(req.params.itemId);
  const { title } = req.body || {};
  try {
    await todoModel.addTemplateSubitem(itemId, title);
    res.redirect('/admin/todo-templates?message=Sub-item%20added');
  } catch (e) {
    console.error('Add template subitem error', e);
    res.redirect(`/admin/todo-templates?error=${encodeURIComponent(e.message || 'Failed to add sub-item')}`);
  }
});

// Preview marketing email with OpenAI
router.post('/marketing/preview', mediaUpload.single('image'), async (req, res) => {
  const { type, message, subject } = req.body;
  const imageUrl = req.file ? `${req.protocol}://${req.get('host')}/uploads/${req.file.filename}` : null;
  const apiKey = process.env.OPENAI_API_KEY;
  if (!apiKey) return res.status(500).json({ error: 'Missing OpenAI API key' });
  if (!marketingTypes.includes(type)) return res.status(400).json({ error: 'Invalid type' });
  try {
    const preface = marketingPrefaces[type] || '';
    const prompt = `${preface}\n\n${message || ''}`;
    const response = await fetch('https://api.openai.com/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${apiKey}`
      },
      body: JSON.stringify({
        model: 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: `Write a professional marketing email to students using the following context:\n${prompt}` }]
      })
    });
    const data = await response.json();
    const generated = data.choices?.[0]?.message?.content?.trim() || '';
    const tpl = marketingTemplates[type];
    const subj = subject && subject.trim() ? subject.trim() : tpl.subject;
    const html = tpl.html
      .replace(/{{imageTag}}/g, imageUrl ? `<img src="${imageUrl}" alt="" style="width:100%;height:auto;display:block;">` : '')
      .replace(/{{subject}}/g, subj)
      .replace(/{{message}}/g, generated)
      .replace(/{{year}}/g, new Date().getFullYear());
    res.json({ html });
  } catch (e) {
    console.error('OpenAI preview error', e);
    res.status(500).json({ error: 'Failed to generate preview' });
  }
});

// Send marketing email
router.post('/marketing', mediaUpload.single('image'), async (req, res) => {
  const students = await userModel.getByRole('student');
  const preregs = await preRegModel.getAll();
  const rsvps = await rsvpModel.getAllRSVPs();
  const courseSet = new Set();
  const programSet = new Set();
  students.forEach(s => {
    const course = s.profile?.course;
    const program = s.profile?.affiliateProgram;
    if (course) courseSet.add(course);
    if (program) programSet.add(program);
  });
  preregs.forEach(p => {
    if (p.course) courseSet.add(p.course);
  });
  const courses = Array.from(courseSet).sort();
  const programs = Array.from(programSet).sort();
  const { recipients, type, subject, message } = req.body;
  const imageUrl = req.file ? `${req.protocol}://${req.get('host')}/uploads/${req.file.filename}` : null;
  const attachments = req.file
    ? [{ filename: req.file.originalname, path: req.file.path }]
    : [];
  const ids = Array.isArray(recipients) ? recipients : [recipients].filter(Boolean);
  if (!ids.length || !marketingTypes.includes(type)) {
    return res.status(400).render('admin_marketing', {
      students,
      preregs,
      rsvps,
      courses,
      programs,
      templates: marketingSubjects,
      user: req.session.user,
      sent: null,
      error: 'Invalid form submission.'
    });
  }
  try {
    const tpl = marketingTemplates[type];
    const subj = (subject && subject.trim()) || tpl.subject;
    const preface = marketingPrefaces[type] || '';
    let bodyText = `${preface}\n\n${message || ''}`;
    const apiKey = process.env.OPENAI_API_KEY;
    if (apiKey) {
      try {
        const response = await fetch('https://api.openai.com/v1/chat/completions', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${apiKey}`
          },
          body: JSON.stringify({
            model: 'gpt-3.5-turbo',
            messages: [{ role: 'user', content: `Write a professional marketing email to students using the following context:\n${bodyText}` }]
          })
        });
        const data = await response.json();
        bodyText = data.choices?.[0]?.message?.content?.trim() || bodyText;
      } catch (e) {
        console.error('OpenAI error', e);
      }
    }

    const html = tpl.html
      .replace(/{{imageTag}}/g, imageUrl ? `<img src="${imageUrl}" alt="" style="width:100%;height:auto;display:block;">` : '')
      .replace(/{{subject}}/g, subj)
      .replace(/{{message}}/g, bodyText)
      .replace(/{{year}}/g, new Date().getFullYear());

    const now = new Date();
    for (const id of ids) {
      const [kind, actualId] = String(id).split('-');
      let recipient;
      if (kind === 'student') {
        recipient = await userModel.findById(Number(actualId));
      } else if (kind === 'pre') {
        recipient = preregs.find(p => p.id === Number(actualId));
      } else if (kind === 'rsvp') {
        recipient = rsvps.find(r => r.id === Number(actualId));
      }
      if (!recipient || !recipient.email) continue;
      await transporter.sendMail({
        from: 'no-reply@mdts-apps.com',
        to: recipient.email,
        subject: subj,
        html,
        text: bodyText,
        attachments
      });
      // Record outbound contact to lead tracker
      try {
        const lead = await leadModel.upsertByEmail({
          name: recipient.name || recipient.username || recipient.fullName || '',
          email: recipient.email,
          phone: recipient.phone || '',
          interestPercent: 0,
          source: 'admin-contact'
        });
        if (lead?.id) {
          await leadModel.addContact(lead.id, { direction: 'outbound', method: 'email', note: `Marketing: ${type}` });
        }
      } catch (e) { console.error('Lead contact log failed', e); }
      // Update LastContacted
      if (kind === 'student') {
        await userModel.setLastContacted(Number(actualId), now);
      } else if (kind === 'pre') {
        await preRegModel.setLastContacted(Number(actualId), now);
      }
    }

    res.redirect('/admin/marketing?sent=1');
  } catch (e) {
    console.error('Error sending marketing email', e);
    res.status(500).render('admin_marketing', {
      students,
      preregs,
      rsvps,
      courses,
      programs,
      templates: marketingSubjects,
      user: req.session.user,
      sent: null,
      error: 'Failed to send email.'
    });
  }
});

// Contact a single pre-registered user (AJAX from SweetAlert)
router.post('/pre-registrations/:id/contact', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const preregs = await preRegModel.getAll();
    const p = preregs.find(x => x.id === id);
    if (!p || !p.email) return res.status(404).json({ error: 'Not found' });
    const { subject, message } = req.body;
    const tpl = marketingTemplates['recruitment'];
    const subj = (subject && subject.trim()) || tpl.subject;
    const html = tpl.html
      .replace(/{{imageTag}}/g, '')
      .replace(/{{subject}}/g, subj)
      .replace(/{{message}}/g, message || '')
      .replace(/{{year}}/g, new Date().getFullYear());
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: p.email,
      subject: subj,
      html,
      text: message || ''
    });
    const now = new Date();
    await preRegModel.setLastContacted(id, now);
    // Record note in lead contacts
    try {
      const lead = await leadModel.upsertByEmail({ name: p.name || '', email: p.email, phone: p.phone || '', interestPercent: 0, source: 'pre-register' });
      if (lead?.id) {
        const snippet = (message || '').replace(/<[^>]*>/g, '').slice(0, 200);
        await leadModel.addContact(lead.id, { direction: 'outbound', method: 'email', note: `Subject: ${subj}${snippet ? `\n${snippet}` : ''}` });
      }
    } catch (e) { console.error('lead note add failed', e); }
    return res.json({ ok: true, lastContacted: now.toISOString() });
  } catch (e) {
    console.error('Contact prereg error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

// Add a manual note under a pre-registered user (stores in lead contacts)
router.post('/pre-registrations/:id/notes', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const { note, method } = req.body || {};
    if (!note || !note.trim()) return res.status(400).json({ error: 'empty' });
    const preregs = await preRegModel.getAll();
    const p = preregs.find(x => x.id === id);
    if (!p || !p.email) return res.status(404).json({ error: 'not found' });
    const lead = await leadModel.upsertByEmail({ name: p.name || '', email: p.email, phone: p.phone || '', interestPercent: 0, source: 'pre-register' });
    if (lead?.id) {
      await leadModel.addContact(lead.id, { direction: 'outbound', method: method || 'note', note });
      const contacts = await leadModel.getContacts(lead.id);
      return res.json({ ok: true, contacts });
    }
    return res.status(500).json({ error: 'lead failed' });
  } catch (e) {
    console.error('Add prereg note error', e);
    return res.status(500).json({ error: 'failed' });
  }
});

// Stop countdown: mark lastContacted now
router.post('/pre-registrations/:id/stop-timer', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const now = new Date();
    await preRegModel.setLastContacted(id, now);
    res.json({ ok: true, lastContacted: now.toISOString() });
  } catch (e) {
    console.error('stop timer error', e);
    res.status(500).json({ error: 'failed' });
  }
});

router.post('/pre-registrations/:id/signatures', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const collected = Boolean(req.body && (req.body.collected === true || req.body.collected === 'true' || req.body.collected === 1 || req.body.collected === '1'));
    await preRegModel.setSignatureCollected(id, collected);
    res.json({ ok: true, collected });
  } catch (e) {
    console.error('signature update error', e);
    res.status(500).json({ error: 'failed' });
  }
});

router.post('/pre-registrations/:id/schedule-interview', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const { date, time, adminId, durationMinutes = 30 } = req.body || {};
    if (!date || !time || !adminId) {
      return res.status(400).json({ error: 'Missing date, time, or admin' });
    }
    const start = new Date(`${date}T${time}`);
    if (Number.isNaN(start.getTime())) {
      return res.status(400).json({ error: 'Invalid date or time' });
    }
    const duration = Math.max(15, Number(durationMinutes) || 30);
    const preregs = await preRegModel.getAll();
    const prereg = preregs.find((p) => Number(p.id) === id);
    if (!prereg) {
      return res.status(404).json({ error: 'Pre-registration not found' });
    }

    const people = await loadCalendarPeople();
    if (!people.byId.has(Number(adminId))) {
      return res.status(400).json({ error: 'Invalid admin selection' });
    }

    const interview = await interviewModel.create({
      studentName: prereg.name || 'Prospective Student',
      studentEmail: prereg.email || '',
      studentPhone: prereg.phone || '',
      scheduledAt: start,
      durationMinutes: duration,
      notes: 'Scheduled from pre-registration actions',
      assignedAdminId: Number(adminId)
    });

    const end = new Date(start.getTime() + duration * 60000);
    const title = `Interview Call: ${prereg.name || 'Prospective Student'}`;
    const descriptionLines = [];
    if (prereg.phone) descriptionLines.push(`Phone: ${prereg.phone}`);
    if (prereg.email) descriptionLines.push(`Email: ${prereg.email}`);
    descriptionLines.push('Scheduled from pre-registration actions.');
    const description = descriptionLines.join('\n');

    const calendarEvent = await calendarEventModel.create({
      title,
      description,
      start,
      end,
      allDay: false,
      createdBy: req.session.user?.id || null
    });

    const attendeeSync = await syncCustomEventAttendees(calendarEvent.id, [Number(adminId)], [], []);
    const eventResponse = formatCustomEventResponse(calendarEvent, attendeeSync.classification);
    const origin = `${req.protocol}://${req.get('host')}`;
    if (attendeeSync.addedUsers.length) {
      sendEventAssignmentEmails(eventResponse, attendeeSync.addedUsers, origin, req.session.user)
        .catch((err) => console.error('calendar event notification error', err));
    }

    res.json({ ok: true, interviewId: interview?.id || null, event: eventResponse });
  } catch (e) {
    console.error('schedule interview error', e);
    res.status(500).json({ error: 'failed' });
  }
});

router.post('/students/:id/info-collected', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const collected = Boolean(req.body && (req.body.collected === true || req.body.collected === 'true' || req.body.collected === 1 || req.body.collected === '1'));
    console.log(`[INFO-COLLECTED] Student ${id}: Setting infoCollected = ${collected}`);
    const student = await userModel.findById(id);
    if (!student || student.role !== 'student') {
      console.error(`[INFO-COLLECTED] Student ${id} not found or not a student`);
      return res.status(404).json({ error: 'Student not found' });
    }
    const updatedProfile = await userModel.updateProfile(id, { infoCollected: collected });
    console.log(`[INFO-COLLECTED] Student ${id}: Profile updated successfully. infoCollected = ${updatedProfile?.infoCollected}`);
    
    // Verify the update by re-fetching the student
    const verifyStudent = await userModel.findById(id);
    const actualValue = verifyStudent?.profile?.infoCollected;
    console.log(`[INFO-COLLECTED] Student ${id}: Verification check - infoCollected = ${actualValue}`);
    
    res.json({ ok: true, collected: actualValue !== undefined ? actualValue : collected });
  } catch (e) {
    console.error('[INFO-COLLECTED] Update error:', e);
    res.status(500).json({ error: 'failed' });
  }
});

router.post('/students/:id/schedule-interview', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const { date, time, adminId, durationMinutes = 30 } = req.body || {};
    if (!date || !time || !adminId) {
      return res.status(400).json({ error: 'Missing date, time, or admin' });
    }
    const start = new Date(`${date}T${time}`);
    if (Number.isNaN(start.getTime())) {
      return res.status(400).json({ error: 'Invalid date or time' });
    }
    const duration = Math.max(15, Number(durationMinutes) || 30);
    const student = await userModel.findById(id);
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }

    const people = await loadCalendarPeople();
    if (!people.byId.has(Number(adminId))) {
      return res.status(400).json({ error: 'Invalid admin selection' });
    }

    const studentName = student.name || student.username || 'Student';
    const studentEmail = student.email || '';
    const studentPhone = (student.profile && student.profile.phones && (student.profile.phones.primary || student.profile.phones.cell || student.profile.phones.home || student.profile.phones.work)) || '';
    const interview = await interviewModel.create({
      studentName,
      studentEmail,
      studentPhone,
      scheduledAt: start,
      durationMinutes: duration,
      notes: 'Scheduled from pending student actions',
      assignedAdminId: Number(adminId)
    });

    const end = new Date(start.getTime() + duration * 60000);
    const title = `Interview Call: ${studentName}`;
    const descriptionLines = [];
    if (studentPhone) descriptionLines.push(`Phone: ${studentPhone}`);
    if (studentEmail) descriptionLines.push(`Email: ${studentEmail}`);
    if (student.profile && student.profile.course) descriptionLines.push(`Course: ${student.profile.course}`);
    descriptionLines.push('Scheduled from pending student actions.');
    const description = descriptionLines.join('\n');

    const calendarEvent = await calendarEventModel.create({
      title,
      description,
      start,
      end,
      allDay: false,
      createdBy: req.session.user?.id || null
    });

    const attendeeSync = await syncCustomEventAttendees(calendarEvent.id, [Number(adminId)], [], []);
    const eventResponse = formatCustomEventResponse(calendarEvent, attendeeSync.classification);
    const origin = `${req.protocol}://${req.get('host')}`;
    if (attendeeSync.addedUsers.length) {
      sendEventAssignmentEmails(eventResponse, attendeeSync.addedUsers, origin, req.session.user)
        .catch((err) => console.error('calendar event notification error', err));
    }

    res.json({ ok: true, interviewId: interview?.id || null, event: eventResponse });
  } catch (e) {
    console.error('student schedule interview error', e);
    res.status(500).json({ error: 'failed' });
  }
});

// Fetch notes for a prereg (via lead contacts)
router.get('/pre-registrations/:id/notes', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const preregs = await preRegModel.getAll();
    const p = preregs.find(x => x.id === id);
    if (!p || !p.email) return res.json({ contacts: [] });
    const lead = await leadModel.getByEmail(p.email);
    if (!lead) return res.json({ contacts: [] });
    const contacts = await leadModel.getContacts(lead.id);
    res.json({ contacts });
  } catch (e) {
    console.error('fetch notes error', e);
    res.status(500).json({ contacts: [] });
  }
});




// New class form
router.get('/classes/new', async (req, res) => {
  const users = await userModel.getAll();
  const teachers = users.filter(u => u.role === 'teacher');
  res.render('create_class', { teachers, error: null });
});
// GET form for creating teacher
router.get('/teachers/new', (req, res) => {
  res.render('create_teacher', { error: null });
});

// Counselors / Mentors page
router.get('/mentors', async (req, res) => {
  try {
    const mentors = await mentorModel.list();
    res.render('admin_mentors', { user: req.session.user, mentors, error: null });
  } catch (e) {
    console.error('mentors list error', e);
    res.render('admin_mentors', { user: req.session.user, mentors: [], error: 'Could not load mentors' });
  }
});

router.post('/mentors', async (req, res) => {
  try {
    const { role, name, phone, email, notes } = req.body || {};
    if (!name) return res.redirect('/admin/mentors');
    await mentorModel.create({ role, name: String(name).trim(), phone, email: (email||'').trim().toLowerCase(), notes });
    res.redirect('/admin/mentors');
  } catch (e) {
    console.error('mentor create error', e);
    res.redirect('/admin/mentors');
  }
});

router.post('/mentors/:id/delete', async (req, res) => {
  try { await mentorModel.remove(Number(req.params.id)); } catch (e) { console.error(e); }
  res.redirect('/admin/mentors');
});

router.post('/mentors/:id/notes', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const note = String(req.body.note || '').trim();
    if (note) await mentorModel.addNote(id, note);
    res.redirect('/admin/mentors');
  } catch (e) {
    console.error('mentor note add error', e);
    res.redirect('/admin/mentors');
  }
});

// Contact a mentor via email and record in interactions and notes
router.post('/mentors/:id/contact', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const mentors = await mentorModel.list();
    const m = mentors.find(x => x.id === id);
    if (!m || !m.email) return res.status(404).json({ error: 'not found' });
    const { subject, message } = req.body || {};
    if (!subject || !message) return res.status(400).json({ error: 'Missing subject or message' });
    await transporter.sendMail({ from: 'noreply@mdts-apps.com', to: m.email, subject, html: message, text: message.replace(/<[^>]*>/g, '') });
    // Interactions log as lead
    try {
      const lead = await leadModel.upsertByEmail({ name: m.name || '', email: m.email, phone: m.phone || '', interestPercent: 0, source: 'mentor' });
      if (lead?.id) await leadModel.addContact(lead.id, { direction: 'outbound', method: 'email', note: `Mentor: ${subject}` });
    } catch (e) { console.error('mentor lead log failed', e); }
    // Also keep a note under the mentor
    try { await mentorModel.addNote(id, `Contacted via email — Subject: ${subject}`); } catch {}
    return res.json({ ok: true });
  } catch (e) {
    console.error('mentor contact error', e);
    return res.status(500).json({ error: 'failed' });
  }
});

// --- Account Merging ---
// Search students by email/name/username
router.get('/merge/search', async (req, res) => {
  try {
    const q = String(req.query.q || '').trim();
    if (!q) return res.json({ items: [] });
    const like = `%${q}%`;
    const [rows] = await mergeDb.query(
      `SELECT id, username, email, name FROM mdtslms_users
       WHERE role='student' AND (username LIKE ? OR email LIKE ? OR name LIKE ?)
       ORDER BY id DESC LIMIT 20`,
      [like, like, like]
    );
    res.json({ items: rows });
  } catch (e) {
    console.error('merge search error', e);
    res.status(500).json({ items: [] });
  }
});

// Merging UI
router.get('/merge', async (req, res) => {
  const aid = Number(req.query.aid || 0);
  const bid = Number(req.query.bid || 0);
  let a = null, b = null, error = null;
  try {
    if (aid) a = await userModel.findById(aid);
    if (bid) b = await userModel.findById(bid);
    if (a && a.role !== 'student') a = null;
    if (b && b.role !== 'student') b = null;
  } catch (e) { error = 'Failed to load selected students.'; }
  res.render('admin_merge', { user: req.session.user, a, b, error });
});

// Execute merge
router.post('/merge', async (req, res) => {
  try {
    const aid = Number(req.body.aid);
    const bid = Number(req.body.bid);
    const target = Number(req.body.targetId);
    const source = target === aid ? bid : aid;
    if (!aid || !bid || !target || !source || aid === bid) return res.status(400).send('Invalid selection');
    const a = await userModel.findById(aid);
    const b = await userModel.findById(bid);
    if (!a || !b || a.role !== 'student' || b.role !== 'student') return res.status(400).send('Invalid users');
    const pick = req.body.pick || {};
    const from = (key) => (pick[key] === 'b' ? b : a);
    // Collect chosen top-level fields
    const name = from('name').name || '';
    const email = from('email').email || '';
    const username = from('username').username || '';
    // Check username uniqueness (allow if username belongs to either of the two)
    const existing = await userModel.findByUsername(username.trim());
    if (existing && existing.id !== target && existing.id !== source) {
      return res.status(400).render('admin_merge', { user: req.session.user, a, b, error: 'Username already exists. Choose a different username.' });
    }
    // Build profile updates
    const ap = a.profile || {}; const bp = b.profile || {};
    function pickVal(key, defA, defB){ return (pick[key] === 'b') ? defB : defA; }
    const addr = {
      line1: pickVal('address_line1', ap.address?.line1 || '', bp.address?.line1 || ''),
      city: pickVal('address_city', ap.address?.city || '', bp.address?.city || ''),
      state: pickVal('address_state', ap.address?.state || '', bp.address?.state || ''),
      zip: pickVal('address_zip', ap.address?.zip || '', bp.address?.zip || '')
    };
    const phones = {
      home: pickVal('phone_home', ap.phones?.home || '', bp.phones?.home || ''),
      cell: pickVal('phone_cell', ap.phones?.cell || '', bp.phones?.cell || ''),
      work: pickVal('phone_work', ap.phones?.work || '', bp.phones?.work || '')
    };
    const profileUpdates = {
      studentId: pickVal('studentId', ap.studentId || '', bp.studentId || ''),
      address: addr,
      phones,
      course: pickVal('course', ap.course || '', bp.course || ''),
      affiliateProgram: pickVal('affiliateProgram', ap.affiliateProgram || '', bp.affiliateProgram || ''),
      financialAidRequested: pickVal('financialAidRequested', !!ap.financialAidRequested, !!bp.financialAidRequested)
    };
    // Merge arrays if requested
    const mergeUploads = !!req.body.merge_uploads;
    const mergeDocs = !!req.body.merge_documents;
    const mergeLinks = !!req.body.merge_links;
    const upA = Array.isArray(ap.uploads) ? ap.uploads : []; const upB = Array.isArray(bp.uploads) ? bp.uploads : [];
    const docA = Array.isArray(ap.documents) ? ap.documents : []; const docB = Array.isArray(bp.documents) ? bp.documents : [];
    const lnA = Array.isArray(ap.links) ? ap.links : []; const lnB = Array.isArray(bp.links) ? bp.links : [];
    if (mergeUploads) profileUpdates.uploads = [...upA, ...upB];
    if (mergeDocs) profileUpdates.documents = [...docA, ...docB];
    if (mergeLinks) profileUpdates.links = [...lnA, ...lnB];

    // Apply updates
    await userModel.updateProfile(target, profileUpdates);
    await mergeDb.query('UPDATE mdtslms_users SET name=?, username=?, email=? WHERE id=?', [name, username, email, target]);
    // Mark merge info
    try { await userModel.updateProfile(target, { mergedFrom: (a.id === source ? [a.id] : []).concat(b.id === source ? [b.id] : []) }); } catch {}
    const action = String(req.body.sourceAction || 'deactivate');
    if (action === 'delete') {
      await userModel.deleteById(source);
    } else {
      await userModel.setActive(source, false);
      try { await userModel.updateProfile(source, { mergedInto: target }); } catch {}
    }
    return res.redirect(`/admin/students/${target}`);
  } catch (e) {
    console.error('merge error', e);
    return res.status(500).send('Merge failed');
  }
});

// Company Partners – list
router.get('/partners', async (req, res) => {
  try {
    const partners = await partnerModel.list();
    res.render('admin_partners', { user: req.session.user, partners, error: null });
  } catch (e) {
    console.error('partners list error', e);
    res.render('admin_partners', { user: req.session.user, partners: [], error: 'Could not load partners' });
  }
});

// Create partner
router.post('/partners', async (req, res) => {
  try {
    const name = String(req.body.name || '').trim();
    const notes = String(req.body.notes || '').trim();
    const website = String(req.body.website || '').trim();
    const phone = String(req.body.phone || '').trim();
    const address = String(req.body.address || '').trim();
    if (!name) return res.redirect('/admin/partners');
    await partnerModel.createPartner({ name, notes, website, phone, address });
    res.redirect('/admin/partners');
  } catch (e) {
    console.error('partner create error', e);
    res.redirect('/admin/partners');
  }
});

// Update partner details
router.post('/partners/:id/update', async (req, res) => {
  try {
    const id = Number(req.params.id);
    if (!id) return res.redirect('/admin/partners');
    const payload = {
      name: req.body.name !== undefined ? String(req.body.name || '').trim() : undefined,
      website: req.body.website !== undefined ? String(req.body.website || '').trim() : undefined,
      phone: req.body.phone !== undefined ? String(req.body.phone || '').trim() : undefined,
      address: req.body.address !== undefined ? String(req.body.address || '').trim() : undefined,
      notes: req.body.notes !== undefined ? String(req.body.notes || '').trim() : undefined
    };
    if (payload.name && !payload.name.length) delete payload.name;
    await partnerModel.updatePartner(id, payload);
  } catch (e) {
    console.error('partner update error', e);
  }
  res.redirect('/admin/partners');
});

// Delete partner
router.post('/partners/:id/delete', async (req, res) => {
  try { await partnerModel.deletePartner(Number(req.params.id)); } catch (e) { console.error(e); }
  res.redirect('/admin/partners');
});

// Fetch all contacts for a partner (JSON)
router.get('/partners/:id/contacts', async (req, res) => {
  try {
    const id = Number(req.params.id);
    if (!id) return res.status(400).json({ ok: false, error: 'Invalid partner' });
    const contacts = await partnerModel.listContactsByPartner(id);
    res.json({ ok: true, contacts });
  } catch (e) {
    console.error('partner contacts list error', e);
    res.status(500).json({ ok: false, error: 'Failed to load contacts' });
  }
});

// Add contact to a partner
router.post('/partners/:id/contacts', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const { role, name, phone, email, notes } = req.body;
    if (!name) return res.redirect('/admin/partners');
    await partnerModel.addContact(id, { role, name, phone, email, notes });
    if (req.headers.accept && req.headers.accept.includes('application/json')) {
      return res.json({ ok: true });
    }
    if ((req.headers['content-type'] || '').includes('application/json')) {
      return res.json({ ok: true });
    }
    res.redirect('/admin/partners');
  } catch (e) {
    console.error('partner contact add error', e);
    if ((req.headers['content-type'] || '').includes('application/json') || (req.headers.accept && req.headers.accept.includes('application/json'))) {
      return res.status(500).json({ ok: false, error: 'Failed to add contact' });
    }
    res.redirect('/admin/partners');
  }
});

// Delete a contact
router.post('/partners/contacts/:cid/delete', async (req, res) => {
  try {
    await partnerModel.deleteContact(Number(req.params.cid));
    if ((req.headers['content-type'] || '').includes('application/json') || (req.headers.accept && req.headers.accept.includes('application/json'))) {
      return res.json({ ok: true });
    }
  } catch (e) {
    console.error(e);
    if ((req.headers['content-type'] || '').includes('application/json') || (req.headers.accept && req.headers.accept.includes('application/json'))) {
      return res.status(500).json({ ok: false, error: 'Failed to remove contact' });
    }
  }
  res.redirect('/admin/partners');
});

// Contact a partner person (logs in Interactions)
router.post('/partners/contacts/:cid/contact', async (req, res) => {
  try {
    const cid = Number(req.params.cid);
    const contact = await partnerModel.getContactById(cid);
    if (!contact || !contact.email) return res.status(404).json({ error: 'Not found' });
    const { subject, message } = req.body || {};
    if (!subject || !message) return res.status(400).json({ error: 'Missing subject or message' });
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: contact.email,
      subject,
      html: message,
      text: message.replace(/<[^>]*>/g, '')
    });
    try {
      const lead = await leadModel.upsertByEmail({ name: contact.name || '', email: contact.email, phone: contact.phone || '', interestPercent: 0, source: 'partner' });
      if (lead?.id) await leadModel.addContact(lead.id, { direction: 'outbound', method: 'email', note: `Partner: ${subject}` });
    } catch (e) { console.error('partner contact lead log failed', e); }
    return res.json({ ok: true });
  } catch (e) {
    console.error('partner contact error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

router.post('/partners/:id/documents', partnerDocsUpload.single('document'), async (req, res) => {
  try {
    const partnerId = Number(req.params.id);
    const contactId = Number(req.body.contactId);
    if (!partnerId || !contactId) return res.redirect('/admin/partners');
    const contact = await partnerModel.getContactById(contactId);
    if (!contact || contact.partnerId !== partnerId) return res.redirect('/admin/partners');
    if (!req.file) return res.redirect('/admin/partners');
    const storedName = path.join('partner-docs', req.file.filename).replace(/\\/g, '/');
    await partnerModel.addDocument(contactId, { originalName: req.file.originalname, storedName });
  } catch (e) {
    console.error('partner document upload error', e);
  }
  res.redirect('/admin/partners');
});

router.post('/partners/contacts/:id/documents', partnerDocsUpload.single('document'), async (req, res) => {
  try {
    const contactId = Number(req.params.id);
    const contact = await partnerModel.getContactById(contactId);
    if (!contact) return res.redirect('/admin/partners');
    if (!req.file) return res.redirect('/admin/partners');
    const storedName = path.join('partner-docs', req.file.filename).replace(/\\/g, '/');
    await partnerModel.addDocument(contactId, { originalName: req.file.originalname, storedName });
  } catch (e) {
    console.error('partner document upload error', e);
  }
  res.redirect('/admin/partners');
});

router.post('/partners/documents/:id/delete', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const doc = await partnerModel.deleteDocument(id);
    if (doc && doc.storedName) {
      const filePath = path.join(__dirname, '..', 'uploads', doc.storedName);
      fs.promises.unlink(filePath).catch(() => {});
    }
  } catch (e) {
    console.error('partner document delete error', e);
  }
  res.redirect('/admin/partners');
});

router.post('/partners/contacts/:id/notes', async (req, res) => {
  try {
    const contactId = Number(req.params.id);
    const note = String(req.body.note || '').trim();
    if (!note) return res.redirect('/admin/partners');
    await partnerModel.addContactNote(contactId, note);
    if ((req.headers['content-type'] || '').includes('application/json') || (req.headers.accept && req.headers.accept.includes('application/json'))) {
      return res.json({ ok: true });
    }
  } catch (e) {
    console.error('partner note add error', e);
    if ((req.headers['content-type'] || '').includes('application/json') || (req.headers.accept && req.headers.accept.includes('application/json'))) {
      return res.status(500).json({ ok: false, error: 'Failed to add note' });
    }
  }
  res.redirect('/admin/partners');
});

router.post('/partners/notes/:id/delete', async (req, res) => {
  try {
    await partnerModel.deleteContactNote(Number(req.params.id));
    if ((req.headers['content-type'] || '').includes('application/json') || (req.headers.accept && req.headers.accept.includes('application/json'))) {
      return res.json({ ok: true });
    }
  } catch (e) {
    console.error('partner note delete error', e);
    if ((req.headers['content-type'] || '').includes('application/json') || (req.headers.accept && req.headers.accept.includes('application/json'))) {
      return res.status(500).json({ ok: false, error: 'Failed to delete note' });
    }
  }
  res.redirect('/admin/partners');
});

router.post('/partners/:id/company-notes', async (req, res) => {
  try {
    const partnerId = Number(req.params.id);
    const note = String(req.body.note || '').trim();
    if (!partnerId || !note) return res.redirect('/admin/partners');
    const createdBy = req.session?.user?.id || null;
    const createdByName = req.session?.user?.name || [req.session?.user?.firstName, req.session?.user?.lastName].filter(Boolean).join(' ') || null;
    await partnerModel.addPartnerNote(partnerId, { note, createdBy, createdByName });
  } catch (e) {
    console.error('partner company note add error', e);
  }
  res.redirect('/admin/partners');
});

router.post('/partners/company-notes/:id/delete', async (req, res) => {
  try {
    await partnerModel.deletePartnerNote(Number(req.params.id));
  } catch (e) {
    console.error('partner company note delete error', e);
  }
  res.redirect('/admin/partners');
});

router.post('/partners/:id/events', async (req, res) => {
  try {
    const partnerId = Number(req.params.id);
    if (!partnerId) return res.status(400).json({ ok: false, error: 'Invalid partner' });
    const partner = await partnerModel.getPartnerById(partnerId);
    if (!partner) return res.status(404).json({ ok: false, error: 'Partner not found' });
    const title = String(req.body.title || '').trim();
    if (!title) return res.status(400).json({ ok: false, error: 'Title required' });
    const description = String(req.body.description || '').trim();
    const startRaw = req.body.start;
    const endRaw = req.body.end;
    const allDay = req.body.allDay === true || req.body.allDay === 'true';
    const startDate = startRaw ? new Date(startRaw) : null;
    const endDate = endRaw ? new Date(endRaw) : null;
    if (!startDate || Number.isNaN(startDate.getTime())) {
      return res.status(400).json({ ok: false, error: 'Invalid start date' });
    }
    if (endDate && Number.isNaN(endDate.getTime())) {
      return res.status(400).json({ ok: false, error: 'Invalid end date' });
    }
    const defaultDescription = description || `Partner: ${partner.name}`;
    const event = await calendarEventModel.create({
      title,
      description: defaultDescription,
      start: startDate,
      end: endDate,
      allDay,
      createdBy: req.session?.user?.id || null
    });
    await partnerModel.linkCalendarEvent(partnerId, event.id);
    return res.json({
      ok: true,
      event: {
        id: event.id,
        title: event.title,
        start: event.start,
        end: event.end,
        allDay: event.allDay,
        description: event.description
      }
    });
  } catch (e) {
    console.error('partner calendar event create error', e);
    return res.status(500).json({ ok: false, error: 'Could not create event' });
  }
});


// Add admin user
router.get('/admins/new', (req, res) => {
  res.render('create_admin', { user: req.session.user, error: null, created: false });
});

router.post('/admins', async (req, res) => {
  try {
    const { name, username, email } = req.body;
    if (!name || !username || !email) {
      return res.status(400).render('create_admin', { user: req.session.user, error: 'All fields are required.', created: false });
    }
    // Uniqueness checks
    const existingUser = await userModel.findByUsername(username.trim());
    if (existingUser) return res.status(400).render('create_admin', { user: req.session.user, error: 'Username already taken.', created: false });
    const existingEmail = await userModel.findByEmail(email.trim().toLowerCase());
    if (existingEmail) return res.status(400).render('create_admin', { user: req.session.user, error: 'Email already in use.', created: false });

    // Generate password
    const crypto = require('crypto');
    const password = crypto.randomBytes(6).toString('base64');
    await userModel.createAdmin({ name: name.trim(), username: username.trim(), email: email.trim().toLowerCase(), password });

    // Send email
    try {
      await transporter.sendMail({
        from: 'noreply@mdts-apps.com',
        to: email.trim().toLowerCase(),
        subject: 'Your MDTS Admin Account',
        text: `Hello ${name},

Your admin account has been created.

Username: ${username}
Password: ${password}

Login: https://mdts-apps.com/admin

Please change your password after logging in.`,
        html: `<p>Hello ${name},</p><p>Your admin account has been created.</p><p><strong>Username:</strong> ${username}<br><strong>Password:</strong> ${password}</p><p><a href="https://lms.mdts-apps.com/admin">Login</a></p><p>Please change your password after logging in.</p>`
      });
    } catch (e) { console.error('Admin email send failed', e); }

    res.render('create_admin', { user: req.session.user, error: null, created: true });
  } catch (e) {
    console.error('Create admin error', e);
    res.status(500).render('create_admin', { user: req.session.user, error: 'Could not create admin.', created: false });
  }
});



// POST create teacher
router.post('/teachers', mediaUpload.fields([{ name: 'photo', maxCount: 1 }, { name: 'signature', maxCount: 1 }]), async (req, res) => {
  try {
    const { name, username, email, signatureDataUrl, sendWelcomeEmail } = req.body;
    if (!name || !username || !email) {
      return res.status(400).render('create_teacher', { error: 'Full name, username, and email are required.' });
    }
    
    // Generate a random password
    const password = Math.random().toString(36).slice(-12) + Math.random().toString(36).slice(-12).toUpperCase() + '123!';
    
    const t = await userModel.createTeacher({ name: name.trim(), username: username.trim(), email: email.trim(), password });
    
    // Optional photo
    if (req.files && Array.isArray(req.files.photo) && req.files.photo[0]) {
      const f = req.files.photo[0];
      await userModel.updateProfile(t.id, { photo: { url: `/uploads/${f.filename}`, originalName: f.originalname } });
    }
    
    // Optional drawn signature
    if (signatureDataUrl && /^data:image\//.test(String(signatureDataUrl))) {
      await userModel.updateProfile(t.id, { signature: { url: signatureDataUrl } });
    }
    
    // Optional links[]
    const links = Array.isArray(req.body['links[]']) ? req.body['links[]'] : (req.body.links || req.body.link || []);
    const urls = (Array.isArray(links) ? links : [links]).map(u => String(u||'').trim()).filter(Boolean);
    if (urls.length) {
      await userModel.addLinks(t.id, urls.map(url => ({ url })));
    }
    
    // Send welcome email if requested
    if (sendWelcomeEmail === 'on') {
      try {
        const emailTemplates = require('../utils/emailTemplates');
        const branding = require('../branding.json');
        
        const loginUrl = `${req.protocol}://${req.get('host')}/login`;
        const schoolName = branding?.schoolName || 'MD Technical School';
        
        const emailHtml = emailTemplates.wrapHtml(
          `Welcome to ${schoolName} - Teacher Account Created`,
          `
            <h2>Welcome to ${schoolName}!</h2>
            <p>Your teacher account has been created. Here are your login credentials:</p>
            <div style="background:#f8f9fa; padding:16px; border-radius:8px; margin:16px 0;">
              <p style="margin:0;"><strong>Username:</strong> ${username}</p>
              <p style="margin:8px 0 0;"><strong>Password:</strong> ${password}</p>
              <p style="margin:8px 0 0;"><strong>Login URL:</strong> <a href="${loginUrl}">${loginUrl}</a></p>
            </div>
            <p><strong>Important:</strong> Please log in and change your password after your first login for security.</p>
            <p>If you have any questions, please contact the administrator.</p>
          `
        );
        
        await transporter.sendMail({
          from: `"${schoolName}" <noreply@mdts-apps.com>`,
          to: email,
          subject: `Welcome to ${schoolName} - Teacher Account Created`,
          html: emailHtml
        });
        
        console.log(`Welcome email sent to teacher: ${email}`);
      } catch (emailError) {
        console.error('Failed to send welcome email:', emailError);
        // Don't fail the teacher creation if email fails
      }
    }
    
    res.redirect('/admin/teachers');
  } catch (e) {
    console.error(e);
    res.status(500).render('create_teacher', { error: 'Could not create teacher.' });
  }
});

// Create class handler
router.post('/classes', async (req, res) => {
  try {
     const schoolYear = (req.body.schoolYear || '').trim();
    const cohort = (req.body.cohort || '').trim();
    const name = (req.body.name || '').trim();
        const shortName = (req.body.shortName || '').trim();

    const description = (req.body.description || '').trim();
    const location = (req.body.location || '').trim();
    const teacherId = Number(req.body.teacherId || 0);
    const assistantTeacherId = req.body.assistantTeacherId ? Number(req.body.assistantTeacherId) : null;
    const weeks = Number(req.body.weeks || 0);
    const clockHours = Number(req.body.clockHours || 0);
    const startDate = (req.body.startDate || '').trim();
    const endDate = (req.body.endDate || '').trim();

    if (!schoolYear || !cohort || !name || !shortName || !teacherId || !startDate || !endDate) {
      const teachers = await userModel.getByRole('teacher');
      return res.status(400).render('create_class', { teachers, error: 'School Year, Cohort, Name, Short Name, Teacher and Dates are required.' });
    }

    // schedule arrays (day[], start[], end[])
    const days = Array.isArray(req.body.day) ? req.body.day : (req.body.day ? [req.body.day] : []);
    const starts = Array.isArray(req.body.start) ? req.body.start : (req.body.start ? [req.body.start] : []);
    const ends = Array.isArray(req.body.end) ? req.body.end : (req.body.end ? [req.body.end] : []);
    const schedule = [];
    for (let i = 0; i < Math.max(days.length, starts.length, ends.length); i++) {
      const d = (days[i] || '').trim();
      const st = (starts[i] || '').trim();
      const en = (ends[i] || '').trim();
      const h = req.body[`holiday${i}`] === 'on' || req.body[`holiday${i}`] === '1';
      if (d && st && en) schedule.push({ day: d, start: st, end: en, holiday: h });
    }

  const klass = await classModel.createClass({ schoolYear, cohort, name, weeks, shortName, description, location, teacherId, assistantTeacherId, schedule, startDate, endDate, clockHours });
    return res.redirect(`/admin/classes/${klass.id}`);
  } catch (e) {
    console.error(e);
     const teachers = await userModel.getByRole('teacher');

    return res.status(500).render('create_class', { teachers, error: 'Could not create class.' });
  }
});


router.get('/classes', async (_req, res) => {
  const classes = await classModel.getAllClasses();
  
  // Sort by start date with newest classes at the top
  classes.sort((a, b) => {
    const dateA = new Date(a.startDate);
    const dateB = new Date(b.startDate);
    return dateB - dateA; // Descending order (newest first)
  });
  
  const users = await userModel.getAll();
  const teachers = users.filter(u => u.role === 'teacher');
  const students = users.filter(u => u.role === 'student');

  const teacherMap = Object.fromEntries(teachers.map(u => [u.id, u.name]));
  const studentMap = new Map(
    students.map(s => [s.id, s])
  );

  const enrichedClasses = classes.map(klass => {
    const studentIds = Array.isArray(klass.studentIds) ? klass.studentIds : [];
    const studentDetails = studentIds
      .map(id => studentMap.get(Number(id)) || studentMap.get(id))
      .filter(Boolean)
      .map(student => {
        const profile = student.profile || {};
        const preferredName = student.name
          || [profile.firstName, profile.lastName].filter(Boolean).join(' ')
          || student.username
          || `Student #${student.id}`;
        return {
          id: student.id,
          name: preferredName,
          email: student.email || '',
          status: student.status || '',
          studentId: profile.studentId || '',
          program: profile.program || {},
          course: profile.course || ''
        };
      });

    return {
      ...klass,
      studentDetails
    };
  });

  res.render('class_list', { classes: enrichedClasses, teacherMap });
});

// Edit class form
router.get('/classes/:id/edit', async (req, res) => {
  const id = Number(req.params.id);
  const klass = await classModel.findClassById(id);
  if (!klass) return res.status(404).send('Not found');
  const teachers = await userModel.getByRole('teacher');
  const hydratedClass = {
    ...klass,
    startDateInput: toDateInput(klass.startDate),
    endDateInput: toDateInput(klass.endDate)
  };
  res.render('edit_class', { klass: hydratedClass, teachers, error: null });
});

// Update class handler
router.post('/classes/:id/edit', async (req, res) => {
  const id = Number(req.params.id);
  const klass = await classModel.findClassById(id);
  if (!klass) return res.status(404).send('Not found');
  try {
    const schoolYear = (req.body.schoolYear || '').trim();
    const cohort = (req.body.cohort || '').trim();
    const name = (req.body.name || '').trim();
    const shortName = (req.body.shortName || '').trim();
    const description = (req.body.description || '').trim();
    const location = (req.body.location || '').trim();
    const teacherId = Number(req.body.teacherId || 0);
    const assistantTeacherId = req.body.assistantTeacherId ? Number(req.body.assistantTeacherId) : null;
    const weeks = Number(req.body.weeks || 0);
    const clockHours = Number(req.body.clockHours || 0);
    const startDate = (req.body.startDate || '').trim();
    const endDate = (req.body.endDate || '').trim();

    if (!schoolYear || !cohort || !name || !shortName || !teacherId || !startDate || !endDate) {
      const teachers = await userModel.getByRole('teacher');
      return res.status(400).render('edit_class', {
        klass: {
          ...klass,
          schoolYear,
          cohort,
          name,
          shortName,
          description,
          location,
          teacherId,
          assistantTeacherId,
          weeks,
          clockHours,
          startDate,
          endDate,
          startDateInput: startDate,
          endDateInput: endDate
        },
        teachers,
        error: 'School Year, Cohort, Name, Short Name, Teacher and Dates are required.'
      });
    }

    // Build schedule from arrays (day[], start[], end[])
    const days = Array.isArray(req.body.day) ? req.body.day : (req.body.day ? [req.body.day] : []);
    const starts = Array.isArray(req.body.start) ? req.body.start : (req.body.start ? [req.body.start] : []);
    const ends = Array.isArray(req.body.end) ? req.body.end : (req.body.end ? [req.body.end] : []);
    const schedule = [];
    for (let i = 0; i < Math.max(days.length, starts.length, ends.length); i++) {
      const d = (days[i] || '').trim();
      const st = (starts[i] || '').trim();
      const en = (ends[i] || '').trim();
      const h = req.body[`holiday${i}`] === 'on' || req.body[`holiday${i}`] === '1';
      if (d && st && en) schedule.push({ day: d, start: st, end: en, holiday: h });
    }

    await classModel.updateClass(id, { schoolYear, cohort, name, shortName, description, location, teacherId, assistantTeacherId, weeks, startDate, endDate, schedule, clockHours });
    return res.redirect(`/admin/classes/${id}`);
  } catch (e) {
    console.error(e);
    const teachers = await userModel.getByRole('teacher');
    return res.status(500).render('edit_class', { klass, teachers, error: 'Could not update class.' });
  }
});

// Delete class
router.post('/classes/:id/delete', async (req, res) => {
  const id = Number(req.params.id);
  const klass = await classModel.findClassById(id);
  if (!klass) return res.status(404).send('Not found');
  await classModel.deleteClass(id);
  res.redirect('/admin/classes');
});

router.get('/chart/NewSignups', async (req, res) => {
      const users = await userModel.getAll();
  const classes = await classModel.getAllClasses();
  const teachers = users.filter(u => u.role === 'teacher');
  const students = users.filter(u => u.role === 'student');
  const pendingStudents = students.filter(u => u.status === 'pending' || !u.status);
  const approvedStudents = students.filter(u => u.status === 'approved');
  const announcements = await announcementModel.forAdmin();

  // ----- course counts -----
  const courseCountsMap = students.reduce((acc, s) => {
    const course = (s.profile && s.profile.course) || 'Unknown';
    acc[course] = (acc[course] || 0) + 1;
    return acc;
  }, {});
  const courseCounts = Object.entries(courseCountsMap).map(([course, count]) => ({ course, count }));

  // ----- state counts -----
  const stateCounts = {};
  students.forEach(s => {
    const state = s.profile && s.profile.address && s.profile.address.state;
    if (state) stateCounts[state] = (stateCounts[state] || 0) + 1;
  });
  const studentsByState = Object.entries(stateCounts).map(([state, count]) => ({ state, count }));

  // ----- affiliate counts -----
  const affiliateCounts = students.reduce((acc, s) => {
    const program = (s.profile && s.profile.affiliateProgram) || 'Unspecified';
    acc[program] = (acc[program] || 0) + 1;
    return acc;
  }, {});


const signupsLast7Days  = buildDailySeries(students, 7);    // day-by-day for past 7 days
const signupsLast30Days = buildDailySeries(students, 30);   // optional
const signupsLast365    = buildDailySeries(students, 365);  // optional

  // const sum = a => a.reduce((n, x) => n + x.count, 0);
  // const signupsTotals = {
  //   week:  sum(signupsDailyWeek),
  //   month: sum(signupsDailyMonth),
  //   year:  sum(signupsDailyYear)
  // };

  const classSizes = classes.map(c => ({ name: c.name, size: (c.studentIds || []).length }));

  // (optional) logging



res.json(signupsLast7Days);
  // res.render('admin_dashboard', {
  //   user: req.session.user,
  //   classes,
  //   teachers,
  //   students,
  //   announcements,
  //   pendingCount: pendingStudents.length,
  //   approvedCount: approvedStudents.length,
  //   classSizes,
  //   courseCounts,
  //   studentsByState,
  //   affiliateCounts,
  //   signupsLast7Days,
  //   signupsLast30Days,
  //   signupsLast365
  // });
});

router.get('/chart/CourseCounts', async (req, res) => {
      const users = await userModel.getAll();
  const classes = await classModel.getAllClasses();
  const teachers = users.filter(u => u.role === 'teacher');
  const students = users.filter(u => u.role === 'student');
  const pendingStudents = students.filter(u => u.status === 'pending' || !u.status);
  const approvedStudents = students.filter(u => u.status === 'approved');
  const announcements = await announcementModel.forAdmin();

  // ----- course counts -----
  const courseCountsMap = students.reduce((acc, s) => {
    const course = (s.profile && s.profile.course) || 'Unknown';
    acc[course] = (acc[course] || 0) + 1;
    return acc;
  }, {});
  const courseCounts = Object.entries(courseCountsMap).map(([course, count]) => ({ course, count }));

  // ----- state counts -----
  const stateCounts = {};
  students.forEach(s => {
    const state = s.profile && s.profile.address && s.profile.address.state;
    if (state) stateCounts[state] = (stateCounts[state] || 0) + 1;
  });
  const studentsByState = Object.entries(stateCounts).map(([state, count]) => ({ state, count }));

  // ----- affiliate counts -----
  const affiliateCounts = students.reduce((acc, s) => {
    const program = (s.profile && s.profile.affiliateProgram) || 'Unspecified';
    acc[program] = (acc[program] || 0) + 1;
    return acc;
  }, {});


const signupsLast7Days  = buildDailySeries(students, 7);    // day-by-day for past 7 days
const signupsLast30Days = buildDailySeries(students, 30);   // optional
const signupsLast365    = buildDailySeries(students, 365);  // optional

  // const sum = a => a.reduce((n, x) => n + x.count, 0);
  // const signupsTotals = {
  //   week:  sum(signupsDailyWeek),
  //   month: sum(signupsDailyMonth),
  //   year:  sum(signupsDailyYear)
  // };

  const classSizes = classes.map(c => ({ name: c.name, size: (c.studentIds || []).length }));

  // (optional) logging



res.json(courseCounts);
  // res.render('admin_dashboard', {
  //   user: req.session.user,
  //   classes,
  //   teachers,
  //   students,
  //   announcements,
  //   pendingCount: pendingStudents.length,
  //   approvedCount: approvedStudents.length,
  //   classSizes,
  //   courseCounts,
  //   studentsByState,
  //   affiliateCounts,
  //   signupsLast7Days,
  //   signupsLast30Days,
  //   signupsLast365
  // });
});

router.get('/chart/AffCounts', async (req, res) => {
      const users = await userModel.getAll();
  const classes = await classModel.getAllClasses();
  const teachers = users.filter(u => u.role === 'teacher');
  const students = users.filter(u => u.role === 'student');
  const pendingStudents = students.filter(u => u.status === 'pending' || !u.status);
  const approvedStudents = students.filter(u => u.status === 'approved');
  const announcements = await announcementModel.forAdmin();

  // ----- course counts -----
  const courseCountsMap = students.reduce((acc, s) => {
    const course = (s.profile && s.profile.course) || 'Unknown';
    acc[course] = (acc[course] || 0) + 1;
    return acc;
  }, {});
  const courseCounts = Object.entries(courseCountsMap).map(([course, count]) => ({ course, count }));

  // ----- state counts -----
  const stateCounts = {};
  students.forEach(s => {
    const state = s.profile && s.profile.address && s.profile.address.state;
    if (state) stateCounts[state] = (stateCounts[state] || 0) + 1;
  });
  const studentsByState = Object.entries(stateCounts).map(([state, count]) => ({ state, count }));

  // ----- affiliate counts -----
// ----- affiliate counts -----
const affiliateCountsMap = students.reduce((acc, s) => {
  const program = (s.profile && s.profile.affiliateProgram) || 'Unspecified';
  acc[program] = (acc[program] || 0) + 1;
  return acc;
}, {});

const affiliateCounts = Object.entries(affiliateCountsMap).map(([affiliate, count]) => ({
  affiliate,
  count
}));


const signupsLast7Days  = buildDailySeries(students, 7);    // day-by-day for past 7 days
const signupsLast30Days = buildDailySeries(students, 30);   // optional
const signupsLast365    = buildDailySeries(students, 365);  // optional

  // const sum = a => a.reduce((n, x) => n + x.count, 0);
  // const signupsTotals = {
  //   week:  sum(signupsDailyWeek),
  //   month: sum(signupsDailyMonth),
  //   year:  sum(signupsDailyYear)
  // };

  const classSizes = classes.map(c => ({ name: c.name, size: (c.studentIds || []).length }));

  // (optional) logging



res.json(affiliateCounts);
  // res.render('admin_dashboard', {
  //   user: req.session.user,
  //   classes,
  //   teachers,
  //   students,
  //   announcements,
  //   pendingCount: pendingStudents.length,
  //   approvedCount: approvedStudents.length,
  //   classSizes,
  //   courseCounts,
  //   studentsByState,
  //   affiliateCounts,
  //   signupsLast7Days,
  //   signupsLast30Days,
  //   signupsLast365
  // });
});

router.get('/chart/SignUp365', async (req, res) => {
  try {
    const users = await userModel.getAllpre();

    // Group by date (YYYY-MM-DD)
    const dailyCounts = users.reduce((acc, u) => {
      if (!u.createdAt) return acc;
      const date = new Date(u.createdAt).toISOString().split('T')[0]; // keep only YYYY-MM-DD
      acc[date] = (acc[date] || 0) + 1;
      return acc;
    }, {});

    // Convert to array for charting
    const result = Object.entries(dailyCounts).map(([date, count]) => ({
      date,
      count
    }));
    console.log("**************");
    console.log(result);

    // Sort chronologically
    result.sort((a, b) => new Date(a.date) - new Date(b.date));

    res.json(result);
  } catch (err) {
    console.error("Error fetching user signup counts:", err);
    res.status(500).json({ error: "Failed to fetch signup counts" });
  }
});

// router.get('/chart/SignUp365', async (req, res) => {
//       const users = await userModel.getAllpre();
//   const classes = await classModel.getAllClasses();
//   const teachers = users.filter(u => u.role === 'teacher');
//   const students = users.filter(u => u.role === 'student');
//   const pendingStudents = students.filter(u => u.status === 'pending' || !u.status);
//   const approvedStudents = students.filter(u => u.status === 'approved');
//   const announcements = await announcementModel.forAdmin();

//   // ----- course counts -----
//   const courseCountsMap = students.reduce((acc, s) => {
//     const course = (s.profile && s.profile.course) || 'Unknown';
//     acc[course] = (acc[course] || 0) + 1;
//     return acc;
//   }, {});
//   const courseCounts = Object.entries(courseCountsMap).map(([course, count]) => ({ course, count }));

//   // ----- state counts -----
//   const stateCounts = {};
//   students.forEach(s => {
//     const state = s.profile && s.profile.address && s.profile.address.state;
//     if (state) stateCounts[state] = (stateCounts[state] || 0) + 1;
//   });
//   const studentsByState = Object.entries(stateCounts).map(([state, count]) => ({ state, count }));

//   // ----- affiliate counts -----
//   const affiliateCounts = students.reduce((acc, s) => {
//     const program = (s.profile && s.profile.affiliateProgram) || 'Unspecified';
//     acc[program] = (acc[program] || 0) + 1;
//     return acc;
//   }, {});


// const signupsLast7Days  = buildDailySeries(students, 7);    // day-by-day for past 7 days
// const signupsLast30Days = buildDailySeries(students, 30);   // optional
// const signupsLast365    = buildDailySeries(students, 365);  // optional

//   // const sum = a => a.reduce((n, x) => n + x.count, 0);
//   // const signupsTotals = {
//   //   week:  sum(signupsDailyWeek),
//   //   month: sum(signupsDailyMonth),
//   //   year:  sum(signupsDailyYear)
//   // };

//   const classSizes = classes.map(c => ({ name: c.name, size: (c.studentIds || []).length }));

//   // (optional) logging
//   console.log("course counts:", JSON.stringify(courseCounts));
//   console.log("student by state:", JSON.stringify(studentsByState));
//   console.log("affiliate counts:", JSON.stringify(affiliateCounts));
//   console.log("signups totals:", JSON.stringify(signupsLast7Days));
//     console.log("signups totals:", JSON.stringify(signupsLast30Days));
//       console.log("signups totals:", JSON.stringify(signupsLast365));


// res.json(signupsLast365);
//   // res.render('admin_dashboard', {
//   //   user: req.session.user,
//   //   classes,
//   //   teachers,
//   //   students,
//   //   announcements,
//   //   pendingCount: pendingStudents.length,
//   //   approvedCount: approvedStudents.length,
//   //   classSizes,
//   //   courseCounts,
//   //   studentsByState,
//   //   affiliateCounts,
//   //   signupsLast7Days,
//   //   signupsLast30Days,
//   //   signupsLast365
//   // });
// });

router.get('/chart/signups', async (req, res) => {
  const range = req.query.range || 'week';
  const daysMap = { week: 7, month: 30, year: 365 };
  const days = daysMap[range] || 7;
  const start = new Date();
  start.setDate(start.getDate() - days + 1);
  const startDate = start.toISOString().slice(0, 10);
  try {
    const [rows] = await db.query(
      `SELECT DATE(appliedAt) AS date, COUNT(*) AS count
       FROM mdtslms_users
       WHERE role='student' AND DATE(appliedAt) >= ?
       GROUP BY DATE(appliedAt)
       ORDER BY DATE(appliedAt)`,
      [startDate]
    );
    const result = [];
    for (let i = 0; i < days; i++) {
      const d = new Date(start);
      d.setDate(start.getDate() + i);
      const ds = d.toISOString().slice(0, 10);
      const row = rows.find(r => r.date === ds);
      result.push({ date: ds, count: row ? row.count : 0 });
    }
    res.json(result);
  } catch (e) {
    res.status(500).json({ error: 'query failed' });
  }
});

router.get('/classes/:id', async (req, res) => {
  const id = Number(req.params.id);
  const klass = await classModel.findClassById(id);
  if (!klass) return res.status(404).send('Not found');
  
  // If requesting JSON (for grade management UI), return class data
  if (req.headers.accept && req.headers.accept.includes('application/json')) {
    return res.json(klass);
  }
  
  const users = await userModel.getAll();
  const students = users.filter(u => u.role === 'student');
  const teacher = await userModel.findById(klass.teacherId);
  const assistantTeacher = klass.assistantTeacherId ? await userModel.findById(klass.assistantTeacherId) : null;
  const discussions = await discussionModel.getByClass(id);

  // Include all enrolled students regardless of approval status
  const classStudents = students
    .filter(s => (klass.studentIds||[]).includes(s.id))
    .sort((a, b) => {
      // Sort alphabetically by name
      const nameA = (a.name || a.profile?.firstName || '').toLowerCase();
      const nameB = (b.name || b.profile?.firstName || '').toLowerCase();
      return nameA.localeCompare(nameB);
    });
  
  // Fetch welcome email status for each student
  const studentWelcomeStatus = {};
  for (const student of classStudents) {
    if (student.email) {
      try {
        const lead = await leadModel.getByEmail(student.email);
        if (lead) {
          const contacts = await leadModel.getContacts(lead.id);
          const welcomeContact = contacts.find(c => 
            c.method === 'email' && 
            c.note && 
            c.note.toLowerCase().includes('welcome')
          );
          if (welcomeContact) {
            studentWelcomeStatus[student.id] = {
              sent: true,
              date: welcomeContact.createdAt
            };
          }
        }
      } catch (e) {
        console.error(`Failed to fetch welcome status for student ${student.id}`, e);
      }
    }
  }
  
  // Load available simulations from sims folder
  const fs = require('fs');
  const path = require('path');
  const simsDir = path.join(__dirname, '..', 'sims');
  let availableSims = [];
  try {
    const entries = fs.readdirSync(simsDir, { withFileTypes: true });
    availableSims = entries
      .filter(d => d.isDirectory())
      .map(d => ({ name: d.name, indexPath: path.join(simsDir, d.name, 'index.html') }))
      .filter(e => fs.existsSync(e.indexPath))
      .map(e => ({ title: e.name, url: `/sims/${e.name}/index.html` }));
  } catch (_) {}
  const lib = resourceLibrary.loadLibrary();
  res.render('view_class', { 
    klass, 
    students, 
    classStudents, 
    studentView: false, 
    discussions, 
    teacher, 
    assistantTeacher, 
    availableSims, 
    availableLectures: lib.lectures, 
    availableAssignments: lib.assignments, 
    availableTests: (lib.tests || []).sort((a, b) => {
      const titleA = a.title || '';
      const titleB = b.title || '';
      return titleA.localeCompare(titleB);
    }),
    user: req.session,
    studentWelcomeStatus
  });
});

router.post('/classes/:id/lectures', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, date, dripDay } = req.body;
  if (title && url) {
    await classModel.addLecture(classId, {
      title: title.trim(),
      url: url.trim(),
      isPowerPoint: false,
      date,
      dripDay: dripDay ? Number(dripDay) : null
    });
    try { resourceLibrary.addLecture({ title: title.trim(), url: url.trim() }); } catch (_) {}
  }
  // Check if it's an AJAX request
  if (req.xhr || req.headers.accept?.indexOf('json') > -1) {
    return res.json({ success: true });
  }
  res.redirect(`/admin/classes/${classId}#lectures`);
});
router.post('/classes/:id/lectures/:lectureId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const lectureId = Number(req.params.lectureId);
  await classModel.removeLecture(classId, lectureId);
  res.redirect(`/admin/classes/${classId}#lectures`);
});
router.post('/classes/:id/lectures/:lectureId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const lectureId = Number(req.params.lectureId);
  const { title, date, dripDay } = req.body;
  await classModel.updateLecture(classId, lectureId, { 
    title: title && title.trim(), 
    date,
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/admin/classes/${classId}#lectures`);
});

// Syllabus upload route for admin
router.post('/classes/:id/syllabus', (req, res) => {
  syllabusUpload(req, res, async (err) => {
    if (err) {
      console.error('Multer error:', err);
      return res.status(400).json({ 
        success: false, 
        message: err.message || 'File upload failed' 
      });
    }
    
    try {
      const classId = Number(req.params.id);
      
      if (!req.file) {
        return res.status(400).json({ success: false, message: 'No file uploaded' });
      }

      console.log('Syllabus file uploaded:', req.file.filename);

      // Ensure syllabus columns exist
      await classModel.ensureSyllabusColumns();

      // Update the class with the syllabus information
      await classModel.updateSyllabus(
        classId,
        `/uploads/syllabi/${req.file.filename}`,
        req.file.originalname
      );

      console.log('Syllabus updated in database for class:', classId);

      res.json({ 
        success: true, 
        message: 'Syllabus uploaded successfully',
        fileName: req.file.originalname,
        filePath: `/uploads/syllabi/${req.file.filename}`
      });
    } catch (error) {
      console.error('Error uploading syllabus:', error);
      console.error('Error stack:', error.stack);
      // Delete the uploaded file if there was an error
      if (req.file) {
        const fs = require('fs');
        const filePath = path.join(__dirname, '../uploads/syllabi', req.file.filename);
        if (fs.existsSync(filePath)) {
          fs.unlinkSync(filePath);
        }
      }
      res.status(500).json({ 
        success: false, 
        message: error.message || 'Error uploading syllabus' 
      });
    }
  });
});

// Syllabus delete route for admin
router.delete('/classes/:id/syllabus', async (req, res) => {
  try {
    const classId = Number(req.params.id);
    
    // Get the current class data to find the file path
    const klass = await classModel.getById(classId);
    
    if (klass.syllabusPath) {
      // Delete the physical file
      const fs = require('fs');
      const filePath = path.join(__dirname, '..', klass.syllabusPath);
      if (fs.existsSync(filePath)) {
        fs.unlinkSync(filePath);
      }
    }

    // Remove syllabus from database
    await classModel.removeSyllabus(classId);

    res.json({ success: true, message: 'Syllabus deleted successfully' });
  } catch (error) {
    console.error('Error deleting syllabus:', error);
    res.status(500).json({ success: false, message: 'Error deleting syllabus' });
  }
});

router.post('/classes/:id/simulations', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, date, dripDay } = req.body;
  if (title && url) {
    await classModel.addSimulation(classId, { 
      title: title.trim(), 
      url: url.trim(), 
      date,
      dripDay: dripDay ? Number(dripDay) : null
    });
  }
  // Check if it's an AJAX request
  if (req.xhr || req.headers.accept?.indexOf('json') > -1) {
    return res.json({ success: true });
  }
  res.redirect(`/admin/classes/${classId}#simulations`);
});
router.post('/classes/:id/simulations/:simId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const simId = Number(req.params.simId);
  await classModel.removeSimulation(classId, simId);
  res.redirect(`/admin/classes/${classId}#simulations`);
});
router.post('/classes/:id/simulations/:simId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const simId = Number(req.params.simId);
  const { title, date, dripDay } = req.body;
  await classModel.updateSimulation(classId, simId, { 
    title: title && title.trim(), 
    date,
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/admin/classes/${classId}#simulations`);
});

router.post('/classes/:id/assignments', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, date, dripDay } = req.body;
  if (title && url) {
    await classModel.addAssignment(classId, { 
      title: title.trim(), 
      url: url.trim(), 
      date,
      dripDay: dripDay ? Number(dripDay) : null
    });
    try { resourceLibrary.addAssignment({ title: title.trim(), url: url.trim() }); } catch (_) {}
  }
  // Check if it's an AJAX request
  if (req.xhr || req.headers.accept?.indexOf('json') > -1) {
    return res.json({ success: true });
  }
  res.redirect(`/admin/classes/${classId}#assignments`);
});
router.post('/classes/:id/assignments/:assignmentId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const assignmentId = Number(req.params.assignmentId);
  await classModel.removeAssignment(classId, assignmentId);
  res.redirect(`/admin/classes/${classId}#assignments`);
});
router.post('/classes/:id/assignments/:assignmentId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const assignmentId = Number(req.params.assignmentId);
  const { title, date, dripDay } = req.body;
  await classModel.updateAssignment(classId, assignmentId, { 
    title: title && title.trim(), 
    date,
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/admin/classes/${classId}#assignments`);
});

router.post('/classes/:id/tests/library', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, timeLimit, dueDate } = req.body;
  if (title) {
    await classModel.addTest(classId, { title: title.trim(), timeLimit: Number(timeLimit) || 90, dueDate });
  }
  // Check if it's an AJAX request
  if (req.xhr || req.headers.accept?.indexOf('json') > -1) {
    return res.json({ success: true });
  }
  res.redirect(`/admin/classes/${classId}#tests`);
});

  router.post('/classes/:id/tests/upload', upload.single('csv'), async (req, res) => {
    const classId = Number(req.params.id);
    const title = (req.body.title || 'Uploaded Test').trim();
      const timeLimit = Number(req.body.timeLimit) || 90;

    const csv = req.file && req.file.buffer.toString('utf-8');
    if (csv) {
      const lines = csv.split(/\r?\n/).filter(l => l.trim());
      const [, ...rows] = lines;
      const questions = rows.map(line => {
        const cols = line.split(',');
        return {
          question: cols[0],
          answer: cols[1],
          explanation: cols[2],
          picture: cols[3],
          options: cols.slice(4, 11).filter(Boolean),
          test: cols[11],
          contentType: cols[12],
          title: cols[13],
          itemType: cols[14],
          path: cols[15]
        };
      });
      await testModel.replaceTestQuestions(title, questions);
      await classModel.addTest(classId, { title, timeLimit });
      try { resourceLibrary.addTest({ title, timeLimit }); } catch (_) {}
    }
    res.redirect(`/admin/classes/${classId}#tests`);
  });

router.post('/classes/:id/tests/generate', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, prompt, timeLimit } = req.body;
  const questionCount = Math.max(1, Math.min(200, Number(req.body.questionCount) || 10));
  const apiKey = process.env.OPENAI_API_KEY;
  if (!apiKey) return res.status(500).send('Missing OpenAI API key');
  try {
    const response = await fetch('https://api.openai.com/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${apiKey}`
      },
      body: JSON.stringify({
        model: 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: `Create a JSON array of exactly ${questionCount} questions for ${prompt}. Each item should contain fields: Question, Answer, Explanation, Picture, OptionA, OptionB, OptionC, OptionD, OptionE, OptionF, OptionG, Test, Content Type, Title, Item Type, Path. Respond with JSON only.` }]
      })
    });
    const data = await response.json();
    let items = [];
    try { items = JSON.parse(data.choices[0].message.content); } catch (_) { items = []; }
      const questions = items.slice(0, questionCount).map(q => {
        const answer = (q.Answer || '').toString().trim();
        let options = [q.OptionA, q.OptionB, q.OptionC, q.OptionD, q.OptionE, q.OptionF, q.OptionG]
          .map(v => (v || '').toString().trim())
          .filter(Boolean);
        // Ensure the answer is among the options
        const hasAnswer = options.some(o => o.toLowerCase() === answer.toLowerCase());
        if (answer && !hasAnswer) {
          if (options.length >= 7) options[options.length - 1] = answer; else options.push(answer);
        }
        return {
          question: q.Question,
          answer,
          explanation: q.Explanation,
          picture: q.Picture,
          options,
          test: q.Test,
          contentType: q['Content Type'],
          title: title,
          itemType: q['Item Type'],
          path: q.Path
        };
      });
      const testTitle = title || 'AI Generated Test';
  questions.forEach(q => { q.test = q.test || testTitle; });
      await testModel.insertQuestions(questions);
      await classModel.addTest(classId, { title: testTitle, timeLimit: Number(timeLimit) || 90 });
      try { resourceLibrary.addTest({ title: testTitle, timeLimit: Number(timeLimit) || 90 }); } catch (_) {}
    } catch (e) {
      console.error('OpenAI error', e);
    }
    res.redirect(`/admin/classes/${classId}#tests`);
  });

// Add external test link
router.post('/classes/:id/tests/external', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, dueDate, dripDay, official } = req.body;
  
  if (title && url) {
    await classModel.addTest(classId, {
      title: title.trim(),
      externalUrl: url.trim(),
      dueDate: dueDate || null,
      dripDay: dripDay ? Number(dripDay) : null,
      official: official === 'on' || official === true || official === 'true',
      isExternal: true
    });
  }
  res.redirect(`/admin/classes/${classId}#tests`);
});


router.get('/reports/pending-students', async (_req, res) => {
  const users = await userModel.getAll();
  const pending = users.filter(u => u.role === 'student' && (u.status === 'pending' || !u.status));
  res.render('pending_students_report', { pending });
});
router.post('/classes/:id/tests/:testId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  await classModel.removeTest(classId, testId);
  res.redirect(`/admin/classes/${classId}#tests`);
});
router.post('/classes/:id/tests/:testId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  const { title, dueDate, official, dripDay } = req.body;
  await classModel.updateTest(classId, testId, { 
    title: title && title.trim(), 
    dueDate,
    official: official === 'on' || official === true || official === 'true',
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/admin/classes/${classId}#tests`);
});

// Links management routes
router.post('/classes/:id/links', async (req, res) => {
  try {
    const classId = Number(req.params.id);
    const { title, url, description } = req.body;
    
    console.log('Adding link to class:', classId, { title, url, description });
    
    if (!title || !url) {
      console.error('Missing title or URL for link');
      return res.status(400).send('Title and URL are required');
    }
    
    const link = await classModel.addLink(classId, {
      title: title.trim(),
      url: url.trim(),
      description: description ? description.trim() : ''
    });
    
    console.log('Link added successfully:', link);
    res.redirect(`/admin/classes/${classId}#links`);
  } catch (error) {
    console.error('Error adding link:', error);
    console.error('Error stack:', error.stack);
    res.status(500).send('Error adding link: ' + error.message);
  }
});

router.post('/classes/:id/links/:linkId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const linkId = Number(req.params.linkId);
  await classModel.removeLink(classId, linkId);
  res.redirect(`/admin/classes/${classId}#links`);
});

router.post('/classes/:id/links/:linkId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const linkId = Number(req.params.linkId);
  const { title, url, description } = req.body;
  await classModel.updateLink(classId, linkId, { 
    title: title && title.trim(), 
    url: url && url.trim(),
    description: description ? description.trim() : ''
  });
  res.redirect(`/admin/classes/${classId}#links`);
});

router.get('/reports/class/:id', async (req, res) => {
  const id = Number(req.params.id);
  const klass = await classModel.findClassById(id);
  if (!klass) return res.status(404).send('Not found');
  const users = await userModel.getAll();
  const students = users.filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id));
  res.render('class_report', { klass, students, scope: 'admin' });
});

router.post('/classes/:id/add-student', async (req, res) => {
  const id = Number(req.params.id);
  const studentId = Number(req.body.studentId);
  const student = await userModel.findById(studentId);
  if (!student) {
    return res.status(400).send('Student not found');
  }
  // Allow both approved and pending students to be added to classes
  if (student.status !== 'approved' && student.status !== 'pending') {
    return res.status(400).send('Student must be approved or pending');
  }
  await classModel.addStudent(id, studentId);
  res.redirect(`/admin/classes/${id}`);
});

router.post('/classes/:id/remove-student', async (req, res) => {
  const id = Number(req.params.id);
  const studentId = Number(req.body.studentId);
  if (!Number.isInteger(id) || !Number.isInteger(studentId)) return res.status(400).send('Bad request');
  await classModel.removeStudent(id, studentId);
  res.redirect(`/admin/classes/${id}`);
});

// Manage grades for a specific student in a class
router.post('/classes/:classId/students/:studentId/grades', async (req, res) => {
  try {
    const classId = Number(req.params.classId);
    const studentId = Number(req.params.studentId);
    const { grades } = req.body;
    
    if (!classId || !studentId) {
      return res.status(400).json({ error: 'Invalid class or student ID' });
    }
    
    if (!Array.isArray(grades)) {
      return res.status(400).json({ error: 'Grades must be an array' });
    }
    
    // Get the class
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).json({ error: 'Class not found' });
    }
    
    // Get or initialize grades array
    let allGrades = klass.grades || [];
    
    // Process each grade
    for (const gradeData of grades) {
      const { type, itemId, score } = gradeData;
      
      if (score < 0 || score > 100) {
        return res.status(400).json({ error: 'Score must be between 0 and 100' });
      }
      
      // Find existing grade record or create new one
      let gradeRecord;
      if (type === 'test') {
        gradeRecord = allGrades.find(g => 
          g.studentId === studentId && g.testId === itemId
        );
      } else if (type === 'assignment') {
        gradeRecord = allGrades.find(g => 
          g.studentId === studentId && g.assignmentId === itemId
        );
      }
      
      if (gradeRecord) {
        // Update existing grade
        gradeRecord.score = score;
        gradeRecord.gradedAt = new Date().toISOString();
        gradeRecord.gradedBy = req.session.user.id;
      } else {
        // Create new grade record
        const newGrade = {
          studentId: studentId,
          score: score,
          gradedAt: new Date().toISOString(),
          gradedBy: req.session.user.id
        };
        
        if (type === 'test') {
          newGrade.testId = itemId;
        } else if (type === 'assignment') {
          newGrade.assignmentId = itemId;
        }
        
        allGrades.push(newGrade);
      }
    }
    
    // Update the class with new grades
    await classModel.updateGrades(classId, allGrades);
    
    // Handle overall grades if provided
    const { overallGrades } = req.body;
    if (overallGrades && (overallGrades.test !== null || overallGrades.assignment !== null || overallGrades.lab !== null)) {
      let studentEnrollments = klass.studentEnrollments || [];
      
      // Find or create enrollment record for this student
      let enrollment = studentEnrollments.find(e => Number(e.studentId) === Number(studentId));
      if (!enrollment) {
        enrollment = { studentId };
        studentEnrollments.push(enrollment);
      }
      
      // Update overall grades
      if (overallGrades.test !== null) enrollment.overallTestGrade = overallGrades.test;
      if (overallGrades.assignment !== null) enrollment.overallAssignmentGrade = overallGrades.assignment;
      if (overallGrades.lab !== null) enrollment.overallLabGrade = overallGrades.lab;
      
      // Save updated enrollments
      await classModel.updateStudentEnrollments(classId, studentEnrollments);
    }
    
    console.log(`Admin ${req.session.user.id} updated ${grades.length} grades for student ${studentId} in class ${classId}`);
    res.json({ ok: true, message: 'Grades updated successfully' });
    
  } catch (error) {
    console.error('Update grades error:', error);
    res.status(500).json({ error: 'Failed to update grades' });
  }
});

// Set overall final grade for a student in a class (overrides calculated grade)
router.post('/classes/:classId/students/:studentId/overall-grade', async (req, res) => {
  try {
    const classId = Number(req.params.classId);
    const studentId = Number(req.params.studentId);
    const { finalGrade } = req.body;
    
    if (!classId || !studentId) {
      return res.status(400).json({ error: 'Invalid class or student ID' });
    }
    
    if (finalGrade < 0 || finalGrade > 100) {
      return res.status(400).json({ error: 'Grade must be between 0 and 100' });
    }
    
    // Get the class
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).json({ error: 'Class not found' });
    }
    
    // Get or create student enrollments array
    let studentEnrollments = klass.studentEnrollments || [];
    let enrollment = studentEnrollments.find(e => e.studentId === studentId);
    
    if (!enrollment) {
      // Create new enrollment record
      enrollment = {
        studentId: studentId,
        enrolledAt: new Date().toISOString()
      };
      studentEnrollments.push(enrollment);
    }
    
    // Set the final grade override
    enrollment.finalGradeOverride = finalGrade;
    enrollment.finalGradeSetAt = new Date().toISOString();
    enrollment.finalGradeSetBy = req.session.user.id;
    
    // Update the class
    await classModel.updateStudentEnrollments(classId, studentEnrollments);
    
    console.log(`Admin ${req.session.user.id} set overall final grade (${finalGrade}) for student ${studentId} in class ${classId}`);
    res.json({ ok: true, message: 'Overall final grade set successfully' });
    
  } catch (error) {
    console.error('Set overall final grade error:', error);
    res.status(500).json({ error: 'Failed to set overall final grade' });
  }
});

// Teacher list
router.get('/teachers', async (req, res) => {
  const users = await userModel.getAll();
  const teachers = users.filter(u => u.role === 'teacher');
  res.render('teacher_list', { teachers });
});

router.post('/classes/:id/duplicate', async (req, res) => {
  const id = Number(req.params.id);
  const copy = await classModel.duplicateClass(id);
  if (!copy) return res.status(404).send('Not found');
  res.redirect(`/admin/classes/${copy.id}`);
});

router.post('/classes/:id/rename', async (req, res) => {
  const id = Number(req.params.id);
  const { name } = req.body;
  if (name && name.trim()) {
    await classModel.renameClass(id, name.trim());
  }
  res.redirect(`/admin/classes/${id}`);
});

// Detailed attendance submission (new format with time in/out, hours, absent/tardy)
router.post('/classes/:id/attendance/detailed', async (req, res) => {
  const classId = Number(req.params.id);
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Class not found');
  
  const { date, scheduledHours, timeIn, timeOut, instructorSignature } = req.body;
  
  // Collect student entries from form
  const entries = [];
  let idx = 0;
  
  while (req.body[`studentId_${idx}`] !== undefined) {
    let studentIdRaw = req.body[`studentId_${idx}`];
    // Handle array case (if form has duplicate fields)
    if (Array.isArray(studentIdRaw)) {
      studentIdRaw = studentIdRaw[0];
    }
    const studentId = Number(studentIdRaw);
    
    // Skip if studentId is not a valid number
    if (isNaN(studentId) || !studentId) {
      idx++;
      continue;
    }
    
    let hoursRaw = req.body[`hoursCompleted_${idx}`];
    if (Array.isArray(hoursRaw)) hoursRaw = hoursRaw[0];
    const hoursCompleted = parseFloat(hoursRaw) || 0;
    
    const present = req.body[`present_${idx}`] === 'on';
    const tardy = req.body[`tardy_${idx}`] === 'on';
    
    let reasonRaw = req.body[`reason_${idx}`];
    if (Array.isArray(reasonRaw)) reasonRaw = reasonRaw[0];
    const reason = reasonRaw || '';
    
    entries.push({
      studentId,
      hoursCompleted: present ? hoursCompleted : 0,
      present,
      tardy,
      reason
    });
    idx++;
  }
  
  // Build attendance data
  const currentUser = req.user || req.session?.user || {};
  const attendanceData = {
    date: date || new Date().toISOString().slice(0, 10),
    scheduledHours: parseFloat(scheduledHours) || 0,
    timeIn: timeIn || '',
    timeOut: timeOut || '',
    entries,
    instructorSignature: instructorSignature || null,
    signedAt: instructorSignature ? new Date().toISOString() : null,
    takenBy: currentUser.name || currentUser.email || 'Admin',
    takenById: currentUser.id || null
  };
  
  // Record detailed attendance
  await classModel.recordDetailedAttendance(classId, attendanceData);
  
  // Update each student's profile with attendance record
  for (const entry of entries) {
    if (!entry.absent && entry.studentId && !isNaN(entry.studentId)) {
      const student = await userModel.findById(entry.studentId);
      if (student) {
        const profile = student.profile || {};
        profile.attendance = profile.attendance || [];
        
        // Check if attendance for this date and class already exists
        const existing = profile.attendance.find(a => a.date === attendanceData.date && a.classId === classId);
        if (!existing) {
          profile.attendance.push({
            date: attendanceData.date,
            classId: classId,
            className: klass.name,
            studentName: student.name,
            hoursCompleted: entry.hoursCompleted,
            tardy: entry.tardy
          });
          await userModel.updateProfile(entry.studentId, { attendance: profile.attendance });
        }
      }
    }
  }
  
  res.redirect(`/admin/classes/${classId}`);
});

// Individual student attendance record view (admin)
router.get('/classes/:classId/students/:studentId/attendance-record', async (req, res) => {
  try {
    const classId = Number(req.params.classId);
    const studentId = Number(req.params.studentId);
    
    console.log('Admin attendance record request:', { classId, studentId });
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      console.log('Class not found:', classId);
      return res.status(404).send('Class not found');
    }
    
    const student = await userModel.findById(studentId);
    if (!student) {
      console.log('Student not found:', studentId);
      return res.status(404).send('Student not found');
    }
    
    // Get attendance record using the new function
    const attendanceRecord = await classModel.getStudentAttendanceRecord(classId, studentId);
    console.log('Attendance record retrieved:', attendanceRecord ? 'success' : 'null');
    
    // Get teacher info
    const teacher = await userModel.findById(klass.teacherId);
    
    res.render('student_attendance_record', {
      klass,
      student,
      teacher,
      attendanceRecord: attendanceRecord || { records: [], summary: { totalScheduledHours: 0, totalCompletedHours: 0, percentageCompleted: 0, meetsRequirement: false } },
      user: req.session.user
    });
  } catch (error) {
    console.error('Error loading attendance record:', error);
    res.status(500).send('Error loading attendance record: ' + error.message);
  }
});

// Delete a specific attendance entry for a student (admin)
router.delete('/classes/:classId/students/:studentId/attendance/:date', async (req, res) => {
  try {
    const classId = Number(req.params.classId);
    const studentId = Number(req.params.studentId);
    const date = req.params.date;
    
    console.log('Admin delete attendance request:', { classId, studentId, date });
    
    const result = await classModel.deleteStudentAttendanceEntry(classId, studentId, date);
    
    if (result.success) {
      res.status(200).json({ success: true, message: 'Attendance record deleted successfully' });
    } else {
      res.status(400).send(result.error || 'Failed to delete attendance record');
    }
  } catch (error) {
    console.error('Error deleting attendance record:', error);
    res.status(500).send('Error deleting attendance record: ' + error.message);
  }
});


// Attendance submission for admin (legacy format)
router.post('/classes/:id/attendance', async (req, res) => {
  const classId = Number(req.params.id);
  const date = req.body.date || new Date().toISOString().slice(0,10);
  const presentIds = Object.keys(req.body)
    .filter(k => k.startsWith('present_'))
    .map(k => Number(k.replace('present_', '')));
  
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Class not found');
  
  // Record attendance in class model
  await classModel.recordAttendance(classId, date, presentIds);
  
  // Update each student's profile with attendance record
  for (const studentId of presentIds) {
    const student = await userModel.findById(studentId);
    if (student) {
      const profile = student.profile || {};
      profile.attendance = profile.attendance || [];
      
      // Check if attendance for this date and class already exists
      const existing = profile.attendance.find(a => a.date === date && a.classId === classId);
      if (!existing) {
        profile.attendance.push({
          date: date,
          classId: classId,
          className: klass.name,
          studentName: student.name
        });
        await userModel.updateProfile(studentId, { attendance: profile.attendance });
      }
    }
  }
  
  res.redirect(`/admin/classes/${classId}`);
});

// Edit attendance for a specific date
router.post('/classes/:id/attendance/edit', async (req, res) => {
  const classId = Number(req.params.id);
  const date = req.body.date;
  const presentArray = req.body.present || [];
  const presentIds = Array.isArray(presentArray) 
    ? presentArray.map(id => Number(id))
    : [Number(presentArray)];
  
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Class not found');
  
  // Update attendance record
  const attendance = klass.attendance || [];
  const recordIndex = attendance.findIndex(a => a.date === date);
  
  if (recordIndex >= 0) {
    // Get old present IDs to remove from student profiles
    const oldPresentIds = attendance[recordIndex].present || [];
    
    // Update the attendance record
    attendance[recordIndex].present = presentIds;
    await classModel.updateClass(classId, { attendance });
    
    // Update student profiles
    // Remove attendance from students who are no longer marked present
    for (const studentId of oldPresentIds) {
      if (!presentIds.includes(studentId)) {
        const student = await userModel.findById(studentId);
        if (student && student.profile && student.profile.attendance) {
          const updatedAttendance = student.profile.attendance.filter(
            a => !(a.date === date && a.classId === classId)
          );
          await userModel.updateProfile(studentId, { attendance: updatedAttendance });
        }
      }
    }
    
    // Add attendance to students who are newly marked present
    for (const studentId of presentIds) {
      if (!oldPresentIds.includes(studentId)) {
        const student = await userModel.findById(studentId);
        if (student) {
          const profile = student.profile || {};
          profile.attendance = profile.attendance || [];
          
          const existing = profile.attendance.find(a => a.date === date && a.classId === classId);
          if (!existing) {
            profile.attendance.push({
              date: date,
              classId: classId,
              className: klass.name,
              studentName: student.name
            });
            await userModel.updateProfile(studentId, { attendance: profile.attendance });
          }
        }
      }
    }
  }
  
  res.redirect(`/admin/classes/${classId}`);
});

// Delete attendance for a specific date
router.post('/classes/:id/attendance/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const date = req.body.date;
  
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Class not found');
  
  // Remove attendance record from class
  const attendance = (klass.attendance || []).filter(a => a.date !== date);
  await classModel.updateClass(classId, { attendance });
  
  // Remove from all student profiles
  const studentIds = klass.studentIds || [];
  for (const studentId of studentIds) {
    const student = await userModel.findById(studentId);
    if (student && student.profile && student.profile.attendance) {
      const updatedAttendance = student.profile.attendance.filter(
        a => !(a.date === date && a.classId === classId)
      );
      await userModel.updateProfile(studentId, { attendance: updatedAttendance });
    }
  }
  
  res.redirect(`/admin/classes/${classId}`);
});

// Attendance report for a class (admin)
router.get('/classes/:id/attendance', async (req, res) => {
  const klass = await classModel.findClassById(Number(req.params.id));
  if (!klass) return res.status(404).send('Not found');
  const users = await userModel.getAll();
  const students = users
    .filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id))
    .sort((a, b) => {
      // Sort alphabetically by name
      const nameA = (a.name || a.profile?.firstName || '').toLowerCase();
      const nameB = (b.name || b.profile?.firstName || '').toLowerCase();
      return nameA.localeCompare(nameB);
    });
  
  // Enhance students with their attendance dates
  const attendanceRecords = klass.attendance || [];
  const studentsWithAttendance = students.map(student => {
    // Get all dates where this student was present
    const presentDates = attendanceRecords
      .filter(record => (record.present || []).includes(student.id))
      .map(record => record.date)
      .sort(); // Sort dates chronologically
    
    return {
      ...student,
      studentId: student.profile?.studentId || student.id,
      presentDates: presentDates,
      totalPresent: presentDates.length
    };
  });
  
  res.render('attendance_report', { klass, students: studentsWithAttendance, user: req.session.user });
});

// Overall Attendance Report - Day by Day
router.get('/classes/:id/attendance/overall', async (req, res) => {
  const klass = await classModel.findClassById(Number(req.params.id));
  if (!klass) return res.status(404).send('Not found');
  const users = await userModel.getAll();
  const students = users
    .filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id))
    .sort((a, b) => {
      const nameA = (a.name || a.profile?.firstName || '').toLowerCase();
      const nameB = (b.name || b.profile?.firstName || '').toLowerCase();
      return nameA.localeCompare(nameB);
    });
  
  // Get all attendance records sorted by date
  const attendanceRecords = (klass.attendance || []).sort((a, b) => a.date.localeCompare(b.date));
  
  res.render('overall_attendance_report', { klass, students, attendanceRecords, user: req.session.user });
});

// Individual student grade report (admin)
router.get('/classes/:classId/students/:studentId/grade-report', async (req, res) => {
  const classId = Number(req.params.classId);
  const studentId = Number(req.params.studentId);
  
  const klass = await classModel.findClassById(classId);
  if (!klass) {
    return res.status(404).send('Class not found');
  }
  
  const users = await userModel.getAll();
  const student = users.find(u => u.id === studentId);
  
  if (!student || student.role !== 'student' || !(klass.studentIds || []).includes(studentId)) {
    return res.status(404).send('Student not found or not enrolled in this class');
  }
  
  res.render('grade_report', { klass, student });
});

// Print gradebook for entire class (admin)
router.get('/classes/:id/gradebook/print', async (req, res) => {
  const classId = Number(req.params.id);
  const klass = await classModel.findClassById(classId);
  
  if (!klass) {
    return res.status(404).send('Class not found');
  }
  
  const users = await userModel.getAll();
  const students = users.filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id));
  
  res.render('gradebook_print', { klass, students });
});

router.post('/users/:id/deactivate', async (req, res) => {
  await userModel.setActive(Number(req.params.id), false);
  res.redirect('back');
});

router.post('/users/:id/activate', async (req, res) => {
  await userModel.setActive(Number(req.params.id), true);
  res.redirect('back');
});

router.post('/teachers/:id/resend', async (req, res) => {
  const id = Number(req.params.id);
  if (!Number.isInteger(id)) {
    return res.status(400).json({ ok: false, error: 'Invalid teacher id.' });
  }

  try {
    const teacher = await userModel.findById(id);
    if (!teacher || teacher.role !== 'teacher') {
      return res.status(404).json({ ok: false, error: 'Teacher not found.' });
    }
    if (!teacher.email) {
      return res.status(400).json({ ok: false, error: 'Teacher email address is missing.' });
    }

    const previousSalt = teacher.salt;
    const previousHash = teacher.hash;
    const password = generateTeacherTempPassword();
    const schoolName = branding?.schoolName || 'MD Technical School';
    const loginUrl = `${req.protocol}://${req.get('host')}/login`;

    try {
      await userModel.updatePassword(teacher.username, password);
    } catch (updateErr) {
      console.error('Teacher password update failed during resend', updateErr);
      return res.status(500).json({ ok: false, error: 'Unable to update teacher password.' });
    }

    try {
      await transporter.sendMail({
        from: `"${schoolName}" <noreply@mdts-apps.com>`,
        to: teacher.email,
        subject: `${schoolName} Teacher Login Information`,
        text: `Hello ${teacher.name || teacher.username},

Your ${schoolName} teacher login credentials have been reset.

Username: ${teacher.username}
Temporary Password: ${password}
Login: ${loginUrl}

Please log in and update your password after signing in.`,
        html: `<p>Hello ${teacher.name || teacher.username},</p>
<p>Your ${schoolName} teacher login credentials have been reset.</p>
<div style="background:#f8f9fa;padding:16px;border-radius:8px;margin:16px 0;">
  <p style="margin:0;"><strong>Username:</strong> ${teacher.username}</p>
  <p style="margin:8px 0 0;"><strong>Temporary Password:</strong> ${password}</p>
  <p style="margin:8px 0 0;"><strong>Login:</strong> <a href="${loginUrl}">${loginUrl}</a></p>
</div>
<p>Please log in and update your password after signing in.</p>`
      });
    } catch (sendErr) {
      console.error('Teacher login email resend failed', sendErr);
      if (previousSalt && previousHash) {
        try {
          await userModel.setPasswordHash(id, previousSalt, previousHash);
        } catch (restoreErr) {
          console.error('Failed to restore teacher password after email failure', restoreErr);
        }
      }
      return res.status(502).json({ ok: false, error: 'Email could not be sent. Password unchanged.' });
    }

    console.log(`Teacher login resent for ${teacher.email} (id: ${teacher.id})`);
    return res.json({ ok: true, message: `Login email sent to ${teacher.email}` });
  } catch (err) {
    console.error('Teacher login resend error', err);
    return res.status(500).json({ ok: false, error: 'Unable to resend login information.' });
  }
});

// Delete teacher
router.post('/teachers/:id/delete', async (req, res) => {
  const id = Number(req.params.id);
  if (!Number.isInteger(id)) {
    return res.status(400).send('Invalid teacher id');
  }
  try {
    await userModel.deleteById(id);
  } catch (err) {
    console.error('Teacher delete error', err);
    return res.status(500).send('Could not delete teacher');
  }
  res.redirect('/admin/teachers');
});

// Edit teacher (placeholder)
// Edit teacher (placeholder)
router.get('/teachers/:id/edit', async (req, res) => {
  const teacher = await userModel.findById(Number(req.params.id));
  if (!teacher || teacher.role !== 'teacher') return res.status(404).send('Not found');
  res.render('teacher_profile', { teacher, role: 'admin', saved: req.query.saved });
});

router.post('/teachers/:id/edit', mediaUpload.single('photo'), async (req, res) => {
  const id = Number(req.params.id);
  const teacher = await userModel.findById(id);
  if (!teacher || teacher.role !== 'teacher') return res.status(404).send('Not found');
  if (req.file) {
    await userModel.updateProfile(id, {
      photo: {
        url: `/uploads/${req.file.filename}`,
        originalName: req.file.originalname
      }
    });
  }
  // Optional drawn signature from edit form
  if (req.body && req.body.signatureDataUrl && req.body.signatureDataUrl.startsWith('data:image/')) {
    await userModel.updateProfile(id, { signature: { url: req.body.signatureDataUrl } });
  }
  res.redirect(`/admin/teachers/${id}/edit?saved=1`);
});

router.post('/teachers/:id/link', mediaUpload.single('image'), async (req, res) => {
  const id = Number(req.params.id);
  const teacher = await userModel.findById(id);
  if (!teacher || teacher.role !== 'teacher') return res.status(404).send('Not found');
  const { url } = req.body;
  if (url) {
    const link = { url };
    if (req.file) {
      link.image = { url: `/uploads/${req.file.filename}`, originalName: req.file.originalname };
    }
    try { await userModel.addLinks(id, [link]); }
    catch (e) { console.error('add link', e); }
  }
  // Optional drawn signature from edit form
  if (req.body && req.body.signatureDataUrl && req.body.signatureDataUrl.startsWith('data:image/')) {
    await userModel.updateProfile(id, { signature: { url: req.body.signatureDataUrl } });
  }
  res.redirect(`/admin/teachers/${id}/edit?saved=1`);
});


router.get('/reports', async (_req, res) => {
  const classes = await classModel.getAllClasses();
  const users = await userModel.getAll();
  const studentMap = Object.fromEntries(users.filter(u => u.role === 'student').map(u => [u.id, u]));
  const teacherMap = Object.fromEntries(users.filter(u => u.role === 'teacher').map(u => [u.id, u]));

  const report = classes.map(k => {
    const studentsArr = (k.studentIds || [])
      .map(id => studentMap[id])
      .filter(Boolean)
      .map(u => ({ id: u.id, name: u.name }));
    return {
      classId: k.id,
      className: k.name,
      teacher: (teacherMap[k.teacherId] || {}).name || 'Unknown',
      studentsList: studentsArr,
      studentCount: studentsArr.length,
      tests: (k.tests || []).length,
      grades: (k.grades || []).length
    };
  });
  res.render('reports', { report, scope: 'admin' });
});

router.get('/reports/audit', async (_req, res) => {
  const [users, classes] = await Promise.all([
    userModel.getAll(),
    classModel.getAllClasses()
  ]);

  const testsTaken = {};
  const assignmentsDone = {};
  const testsCreated = {};
  const lecturesCreated = {};

  for (const k of classes) {
    for (const g of k.grades || []) {
      if (g.studentId && g.testId) {
        testsTaken[g.studentId] = (testsTaken[g.studentId] || 0) + 1;
      }
    }
    for (const s of k.submissions || []) {
      if (s.studentId) {
        assignmentsDone[s.studentId] = (assignmentsDone[s.studentId] || 0) + 1;
      }
    }
    if (k.teacherId) {
      testsCreated[k.teacherId] = (testsCreated[k.teacherId] || 0) + (k.tests?.length || 0);
      lecturesCreated[k.teacherId] = (lecturesCreated[k.teacherId] || 0) + (k.lectures?.length || 0);
    }
  }

  const report = users.map(u => ({
    id: u.id,
    name: u.name,
    role: u.role,
    lastLogin: u.profile?.lastLogin || '',
    timeSpent: u.profile?.timeSpent || 0,
    pagesVisited: Array.isArray(u.profile?.pagesVisited) ? u.profile.pagesVisited.length : 0,
    testsTaken: testsTaken[u.id] || 0,
    simsTaken: 0,
    assignmentsCompleted: assignmentsDone[u.id] || 0,
    testsCreated: testsCreated[u.id] || 0,
    lecturesTaken: 0,
    lecturesCreated: lecturesCreated[u.id] || 0,
    affiliateProgram: u.profile?.affiliateProgram || '',
    fundingSource: u.profile?.tuition?.fundingSource || ''
  }));

  res.render('audit_report', { report, user: _req.session.user });
});

// Events dashboard for analytics
router.get('/events', async (req, res) => {
  // Admin calendar view
  res.render('events_calendar', { user: req.session.user });
});

// Calendar feed for admin (RSVP events only for now)
router.get('/events/feed', async (_req, res) => {
  try {
    const events = await eventModel.getAllEvents();
    const out = events.map(e => ({
      id: e.id,
      title: e.name,
      date: e.eventDate,
      description: e.description || '',
      attachment: e.attachment || '',
      notes: e.notes || '',
      checklist: Array.isArray(e.checklist) ? e.checklist : [],
      type: 'event'
    }));
    
    // Add class schedules to calendar - generate dates for next 3 months
    const classes = await classModel.getAllClasses();
    const today = new Date();
    const threeMonthsLater = new Date(today);
    threeMonthsLater.setMonth(threeMonthsLater.getMonth() + 3);
    
    const dayMap = { 'Sunday': 0, 'Monday': 1, 'Tuesday': 2, 'Wednesday': 3, 'Thursday': 4, 'Friday': 5, 'Saturday': 6 };
    
    classes.forEach(klass => {
      if (klass.schedule && Array.isArray(klass.schedule)) {
        klass.schedule.forEach(scheduleItem => {
          const dayOfWeek = dayMap[scheduleItem.day];
          if (dayOfWeek !== undefined) {
            // Generate dates for this day of week within the date range
            let currentDate = new Date(today);
            currentDate.setDate(currentDate.getDate() - currentDate.getDay() + dayOfWeek); // Move to next occurrence of this day
            if (currentDate < today) currentDate.setDate(currentDate.getDate() + 7); // If in past, move to next week
            
            while (currentDate <= threeMonthsLater) {
              // Check if within class start/end dates if they exist
              if (klass.startDate) {
                const startDate = new Date(klass.startDate);
                if (currentDate < startDate) {
                  currentDate.setDate(currentDate.getDate() + 7);
                  continue;
                }
              }
              if (klass.endDate) {
                const endDate = new Date(klass.endDate);
                if (currentDate > endDate) break;
              }
              
              const dateStr = currentDate.toISOString().slice(0, 10);
              out.push({
                id: `class-${klass.id}-${dateStr}`,
                title: `${klass.name}${klass.cohort ? ' - ' + klass.cohort : ''}`,
                date: dateStr,
                description: `Class: ${klass.name}\nCohort: ${klass.cohort || 'N/A'}\nTime: ${scheduleItem.start || ''} - ${scheduleItem.end || ''}`,
                type: 'class',
                classId: klass.id,
                className: klass.name,
                cohort: klass.cohort || ''
              });
              
              currentDate.setDate(currentDate.getDate() + 7); // Move to next week
            }
          }
        });
      }
    });
    
    res.json({ events: out });
  } catch (e) {
    console.error('events feed', e);
    res.status(500).json({ events: [] });
  }
});

// Update notes
router.post('/events/:id/notes', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const { notes } = req.body || {};
    await db.query('UPDATE mdtslms_events SET notes=? WHERE id=?', [notes || '', id]);
    res.json({ ok: true });
  } catch (e) {
    console.error('update notes', e);
    res.status(500).json({ error: 'Failed' });
  }
});

// Update checklist
router.post('/events/:id/checklist', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const list = Array.isArray(req.body.items) ? req.body.items : [];
    await db.query('UPDATE mdtslms_events SET checklist=? WHERE id=?', [JSON.stringify(list), id]);
    res.json({ ok: true });
  } catch (e) {
    console.error('update checklist', e);
    res.status(500).json({ error: 'Failed' });
  }
});

// Custom report builder routes
router.get('/reports/custom', async (_req, res) => {
  const [tables] = await db.query('SHOW TABLES');
  const tableNames = tables.map(t => Object.values(t)[0]);
  res.render('custom_report', { tables: tableNames, columns: [], results: null, selected: {}, zeroAdmin: !!_req.session?.zeroAdmin, pkCol: null });
});

router.get('/reports/custom/columns', async (req, res) => {
  const table = req.query.table;
  if (!table) return res.json([]);
  const [cols] = await db.query('SHOW COLUMNS FROM ??', [table]);
  res.json(cols.map(c => ({ name: c.Field, type: c.Type })));
});

router.post('/reports/custom', async (req, res) => {
  const { table, joinTable, joinBaseCol, joinJoinCol, columns = [], filterCol, operator, value, jsonPath } = req.body;
  const [tables] = await db.query('SHOW TABLES');
  const tableNames = tables.map(t => Object.values(t)[0]);
  if (!table || !tableNames.includes(table)) {
    return res.render('custom_report', { tables: tableNames, columns: [], results: [], selected: {}, error: 'Invalid table selected.', zeroAdmin: !!req.session?.zeroAdmin, pkCol: null });
  }

  const [baseCols] = await db.query('SHOW COLUMNS FROM ??', [table]);
  const baseColNames = baseCols.map(c => c.Field);
  let joinCols = [];
  let joinColNames = [];
  if (joinTable && tableNames.includes(joinTable)) {
    const [jc] = await db.query('SHOW COLUMNS FROM ??', [joinTable]);
    joinCols = jc;
    joinColNames = jc.map(c => c.Field);
  }

  // Determine primary key column (single-column only)
  let pkCol = null;
  try {
    const [keys] = await db.query('SHOW KEYS FROM ?? WHERE Key_name = "PRIMARY"', [table]);
    if (keys && keys.length === 1) pkCol = keys[0].Column_name;
  } catch (_) {}

  const colTypes = {};
  baseCols.forEach(c => colTypes[`${table}.${c.Field}`] = c.Type);
  joinCols.forEach(c => colTypes[`${joinTable}.${c.Field}`] = c.Type);

  const allCols = Object.keys(colTypes);
  let selectedCols = Array.isArray(columns) ? columns : [columns];
  selectedCols = selectedCols.filter(c => allCols.includes(c));
  if (!selectedCols.length) {
    selectedCols = baseColNames.map(c => `${table}.${c}`);
  }

  const allowedOps = ['=', '!=', '>', '<', 'LIKE'];
  let sql = `SELECT ${selectedCols.map(c => `${db.escapeId(c.split('.')[0])}.${db.escapeId(c.split('.')[1])}`).join(', ')} FROM ${db.escapeId(table)}`;
  const params = [];
  if (joinTable && tableNames.includes(joinTable) && joinBaseCol && joinJoinCol && baseColNames.includes(joinBaseCol) && joinColNames.includes(joinJoinCol)) {
    sql += ` LEFT JOIN ${db.escapeId(joinTable)} ON ${db.escapeId(table)}.${db.escapeId(joinBaseCol)} = ${db.escapeId(joinTable)}.${db.escapeId(joinJoinCol)}`;
  }

  if (filterCol && value && allowedOps.includes(operator) && allCols.includes(filterCol)) {
    const [fTable, fCol] = filterCol.split('.');
    const identifier = `${db.escapeId(fTable)}.${db.escapeId(fCol)}`;
    if (colTypes[filterCol]?.toLowerCase().includes('json') && jsonPath) {
      sql += ` WHERE JSON_EXTRACT(${identifier}, ?) ${operator} ?`;
      params.push(jsonPath, operator === 'LIKE' ? `%${value}%` : value);
    } else {
      sql += ` WHERE ${identifier} ${operator} ?`;
      params.push(operator === 'LIKE' ? `%${value}%` : value);
    }
  }

  const [rows] = await db.query(sql, params);
  const colObjects = [];
  baseCols.forEach(c => colObjects.push({ table, name: c.Field, type: c.Type }));
  if (joinTable && joinCols.length) joinCols.forEach(c => colObjects.push({ table: joinTable, name: c.Field, type: c.Type }));

  res.render('custom_report', {
    tables: tableNames,
    columns: colObjects,
    results: rows,
    selected: { table, joinTable, joinBaseCol, joinJoinCol, columns: selectedCols, filterCol, operator, value, jsonPath },
    zeroAdmin: !!req.session?.zeroAdmin,
    pkCol
  });
});

// Zero Admin Access: elevate session for destructive schema ops
router.post('/reports/custom/zero', (req, res) => {
  const { password } = req.body || {};
  if (password === 'TheMadden04!') {
    if (req.session) req.session.zeroAdmin = true;
    return res.json({ ok: true });
  }
  return res.status(401).json({ error: 'Invalid password' });
});

// Update row values (Zero Admin)
router.post('/reports/custom/rows/update', async (req, res) => {
  try {
    if (!req.session?.zeroAdmin) return res.status(403).json({ error: 'Unauthorized' });
    const { table, keyCol, keyVal, updates } = req.body || {};
    if (!table || !keyCol || keyVal === undefined || typeof updates !== 'object' || updates === null) {
      return res.status(400).json({ error: 'Missing parameters' });
    }
    // basic guard against suspicious identifier input
    if (!/^\w+$/.test(table) || !/^\w+$/.test(keyCol)) {
      return res.status(400).json({ error: 'Invalid identifiers' });
    }
    const [tables] = await db.query('SHOW TABLES');
    const tableNames = tables.map(t => Object.values(t)[0]);
    if (!tableNames.includes(table)) return res.status(400).json({ error: 'Invalid table' });
    const [cols] = await db.query('SHOW COLUMNS FROM ??', [table]);
    const colMap = new Map(cols.map(c => [c.Field, c]));
    if (!colMap.has(keyCol)) return res.status(400).json({ error: 'Invalid key' });
    const entries = Object.entries(updates).filter(([c, _]) => colMap.has(c) && c !== keyCol);
    if (!entries.length) return res.json({ ok: true, updated: 0 });

    // Build parameter list for UPDATE using identifier/value pairs
    const setClause = entries.map(() => '?? = ?').join(', ');
    const params = [table];
    entries.forEach(([c, v]) => params.push(c, v));
    params.push(keyCol, keyVal);
    const sql = `UPDATE ?? SET ${setClause} WHERE ?? = ? LIMIT 1`;
    const [result] = await db.query(sql, params);
    if (!result.affectedRows) return res.status(404).json({ error: 'Row not found' });

    // Fetch updated row (limited to originally editable columns + key)
    const retCols = [keyCol, ...entries.map(e => e[0])];
    const selectSql = `SELECT ${retCols.map(() => '??').join(', ')} FROM ?? WHERE ?? = ? LIMIT 1`;
    const selectParams = [...retCols, table, keyCol, keyVal];
    const [rows] = await db.query(selectSql, selectParams);
    return res.json({ ok: true, updated: result.affectedRows || 0, row: rows[0] || null });
  } catch (e) {
    console.error('Row update failed', e);
    return res.status(500).json({ error: 'Update failed' });
  }
});


// Preview test routes
router.get('/classes/:id/tests/:testId/preview', async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  console.log('Admin preview test', { classId, testId, userId: req.session.user.id });
  const klass = await classModel.findClassById(classId);
  if (!klass) {
    console.log('Class not found', classId);
    return res.status(404).send('Not found');
  }
  const test = (klass.tests || []).find(t => t.id === testId);
  if (!test) {
    console.log('Test not found', { classId, testId });
    return res.status(404).send('Test not found');
  }
  test.questions = await testModel.getQuestionsByTest(test.title);
  console.log(test.questions);
  res.render('take_test', {
    klass,
    test,
    attempts: 0,
    user: req.session.user,
    action: `/admin/classes/${classId}/tests/${testId}/preview`
  });
});

router.post('/classes/:id/tests/:testId/preview', async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  console.log('Admin submit preview', { classId, testId, userId: req.session.user.id });
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Not found');
  const test = (klass.tests || []).find(t => t.id === testId);
  if (!test) return res.status(404).send('Test not found');
  test.questions = await testModel.getQuestionsByTest(test.title);
  let score = 0;
  test.questions.forEach((q, i) => {
    const chosen = Number(req.body[`q_${i}`]);
    const correct = Number(q.answer);
    if (!Number.isNaN(chosen) && chosen === correct) score++;
  });
  const pct = Math.round((score / test.questions.length) * 100);
  console.log('Admin preview score', { classId, testId, userId: req.session.user.id, score: pct });
  res.render('test_result', { klass, test, score: pct, student: req.session.user, user: req.session.user });
});

// Upload media for a specific question while previewing (admin)
router.post('/classes/:id/tests/:testId/questions/:qIndex/media', mediaUpload.single('media'), async (req, res) => {
  try {
    const classId = Number(req.params.id);
    const testId = Number(req.params.testId);
    const qIndex = Number(req.params.qIndex);
    const klass = await classModel.findClassById(classId);
    if (!klass) return res.status(404).send('Not found');
    const test = (klass.tests || []).find(t => t.id === testId);
    if (!test) return res.status(404).send('Test not found');
    // Load question text by index for robustness
    let question = '';
    try {
      const questions = await testModel.getQuestionsByTest(test.title);
      if (Array.isArray(questions) && questions[qIndex]) question = questions[qIndex].question || '';
    } catch (_) {}
    if (!req.file) return res.redirect(`/admin/classes/${classId}/tests/${testId}/preview`);
    const url = `/uploads/${req.file.filename}`;
    const isVideo = (req.file.mimetype || '').startsWith('video/');
    try {
      await testModel.updateQuestionMediaByText(test.title, question, url, isVideo);
    } catch (e) {
      console.error('update media failed', e);
    }
    return res.redirect(`/admin/classes/${classId}/tests/${testId}/preview#q${qIndex}`);
  } catch (e) {
    console.error('Admin question media upload error', e);
    return res.redirect(`/admin/classes/${req.params.id}/tests/${req.params.testId}/preview`);
  }
});

// Test library management
router.get('/other/tests', (req, res) => {
  const lib = resourceLibrary.loadLibrary();
  res.render('admin_test_library', { tests: lib.tests || [] });
});

router.post('/other/tests/:title/update', async (req, res) => {
  const oldTitle = req.params.title;
  const { title, timeLimit } = req.body;
  if (title) {
    await testModel.renameTest(oldTitle, title.trim());
    resourceLibrary.updateTest(oldTitle, { title: title.trim(), timeLimit: Number(timeLimit) || 90 });
  }
  res.redirect('/admin/other/tests');
});

router.post('/other/tests/:title/delete', async (req, res) => {
  const title = req.params.title;
  await testModel.deleteTest(title);
  resourceLibrary.removeTest(title);
  res.redirect('/admin/other/tests');
});

// Manual trigger for set_user_status_inactive
router.post('/set-inactive-users', async (req, res) => {
  try {
    await db.query('SELECT set_user_status_inactive()');
    res.json({ success: true, message: 'Successfully executed set_user_status_inactive()' });
  } catch (err) {
    console.error('Error executing set_user_status_inactive():', err);
    res.status(500).json({ success: false, message: 'Error: ' + err.message });
  }
});

// Inactive users management page
router.get('/manage-inactive-users', async (req, res) => {
  try {
    const users = await userModel.getAll();
    const students = users.filter(u => u.role === 'student' && (u.status === 'approved' || u.status === 'inactive'));
    
    res.render('manage_inactive_users', { 
      students,
      user: req.session.user 
    });
  } catch (err) {
    console.error('Error loading inactive users management:', err);
    res.status(500).send('Error loading page');
  }
});

// Mark student as inactive
router.post('/students/:id/mark-inactive', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    await userModel.setStatus(studentId, 'inactive');
    
    console.log(`Student ${studentId} marked as inactive by admin ${req.session.user.id}`);
    res.json({ ok: true, message: 'Student marked as inactive' });
    
  } catch (e) {
    console.error('Mark inactive error:', e);
    res.status(500).json({ error: 'Failed to mark student as inactive' });
  }
});

// Mark student as active (and update start date)
router.post('/students/:id/mark-active', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    // Update status to approved (active)
    await userModel.setStatus(studentId, 'approved');
    
    // Update start date to current date
    const profile = student.profile || {};
    profile.startDate = new Date().toISOString();
    await userModel.updateProfile(studentId, profile);
    
    console.log(`Student ${studentId} marked as active with new start date by admin ${req.session.user.id}`);
    res.json({ ok: true, message: 'Student marked as active', startDate: profile.startDate });
    
  } catch (e) {
    console.error('Mark active error:', e);
    res.status(500).json({ error: 'Failed to mark student as active' });
  }
});

// Auto-mark students inactive (3+ weeks old start date)
router.post('/auto-mark-inactive', async (req, res) => {
  try {
    const users = await userModel.getAll();
    const students = users.filter(u => u.role === 'student' && u.status === 'approved');
    
    const threeWeeksAgo = new Date();
    threeWeeksAgo.setDate(threeWeeksAgo.getDate() - 21); // 3 weeks = 21 days
    
    let markedInactive = 0;
    
    for (const student of students) {
      const startDate = student.profile?.startDate;
      if (startDate) {
        const studentStartDate = new Date(startDate);
        if (studentStartDate < threeWeeksAgo) {
          await userModel.setStatus(student.id, 'inactive');
          markedInactive++;
          console.log(`Auto-marked student ${student.id} as inactive (start date: ${startDate})`);
        }
      }
    }
    
    console.log(`Auto-marked ${markedInactive} students as inactive by admin ${req.session.user.id}`);
    res.json({ ok: true, message: `${markedInactive} student(s) marked as inactive`, count: markedInactive });
    
  } catch (e) {
    console.error('Auto mark inactive error:', e);
    res.status(500).json({ error: 'Failed to auto-mark students as inactive' });
  }
});

// Manual trigger for alumni transition
router.post('/transition-to-alumni', async (req, res) => {
  try {
    const alumniTransition = require('../middleware/alumniAutoTransition');
    const result = await alumniTransition.checkAllStudentsAlumniStatus();
    
    // Save the timestamp of this transition
    const fs = require('fs');
    const path = require('path');
    const timestamp = {
      lastTransitionDate: new Date().toISOString(),
      checked: result.checked,
      transitioned: result.transitioned
    };
    const timestampPath = path.join(__dirname, '../data/alumniTransitionTimestamp.json');
    fs.writeFileSync(timestampPath, JSON.stringify(timestamp, null, 2));
    
    res.json({ 
      success: true, 
      message: `Checked ${result.checked} students, transitioned ${result.transitioned} to alumni status`,
      checked: result.checked,
      transitioned: result.transitioned
    });
  } catch (err) {
    console.error('Error executing alumni transition:', err);
    res.status(500).json({ success: false, message: 'Error: ' + err.message });
  }
});

// ===== Career Advisor Interviews =====

// GET: View all career advisor interviews
router.get('/career-advisor-interviews', async (req, res) => {
  try {
    const interviews = careerAdvisorInterviewModel.getAll();
    const upcoming = careerAdvisorInterviewModel.getUpcoming();
    const pending = interviews.filter(i => i.status === 'pending');
    const confirmed = interviews.filter(i => i.status === 'confirmed');
    const settings = careerAdvisorSettings.getSettings();
    
    res.render('admin_career_interviews', {
      user: { ...(req.session.user || {}), role: req.session.role },
      interviews,
      upcoming,
      pending,
      confirmed,
      settings,
      saved: req.query.saved,
      settingsSaved: req.query.settingsSaved,
      branding: req.app.locals.branding
    });
  } catch (err) {
    console.error('Career advisor interviews page error:', err);
    res.status(500).send('Error loading career advisor interviews');
  }
});

// POST: Send confirmation email for interview
router.post('/career-advisor-interviews/:id/confirm', async (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    const { advisorName, meetingLink } = req.body;
    
    if (!advisorName) {
      return res.redirect('/admin/career-advisor-interviews?saved=0');
    }
    
    const interview = careerAdvisorInterviewModel.getById(id);
    if (!interview) {
      return res.redirect('/admin/career-advisor-interviews?saved=0');
    }
    
    // Update interview with advisor details
    careerAdvisorInterviewModel.markConfirmationSent(id, advisorName, meetingLink || null);
    
    // Send confirmation email
    const dateObj = new Date(interview.date + 'T12:00:00');
    const formattedDate = dateObj.toLocaleDateString('en-US', { 
      weekday: 'long', 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    });
    
    const meetingLinkHtml = meetingLink 
      ? `<p><strong>Meeting Link:</strong> <a href="${meetingLink}">${meetingLink}</a></p>`
      : '<p>A meeting link will be provided shortly.</p>';
    
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: interview.studentEmail,
      subject: 'Career Advisor Interview Confirmed',
      html: `
        <h2>Your Career Advisor Interview is Confirmed</h2>
        <p>Dear ${interview.studentName},</p>
        <p>Your career advisor interview has been confirmed with the following details:</p>
        <ul>
          <li><strong>Date:</strong> ${formattedDate}</li>
          <li><strong>Time:</strong> ${interview.time}</li>
          <li><strong>Career Advisor:</strong> ${advisorName}</li>
          <li><strong>Topic:</strong> ${interview.interestArea}</li>
        </ul>
        ${meetingLinkHtml}
        <p>Please be prepared to discuss your career goals and have any questions ready.</p>
        <p>If you need to reschedule, please contact us as soon as possible.</p>
        <br>
        <p>Best regards,<br>Career Services Team</p>
      `,
      text: `Career Advisor Interview Confirmed\n\nDear ${interview.studentName},\n\nYour interview is confirmed:\n\nDate: ${formattedDate}\nTime: ${interview.time}\nAdvisor: ${advisorName}\nTopic: ${interview.interestArea}\n\n${meetingLink ? 'Meeting Link: ' + meetingLink : 'A meeting link will be provided shortly.'}\n\nBest regards,\nCareer Services Team`
    });
    
    // Add to student's interactions
    try {
      const lead = await leadModel.getByEmail(interview.studentEmail);
      if (lead) {
        await leadModel.addContact(lead.id, {
          type: 'career-advisor-confirmed',
          notes: `Career advisor interview confirmed with ${advisorName} for ${formattedDate} at ${interview.time}`,
          scheduledFor: new Date(`${interview.date}T${convertTo24HourAdmin(interview.time)}`).toISOString()
        });
      }
    } catch (err) {
      console.error('Failed to add confirmation to interactions:', err);
    }
    
    res.redirect('/admin/career-advisor-interviews?saved=1');
  } catch (err) {
    console.error('Error sending confirmation:', err);
    res.redirect('/admin/career-advisor-interviews?saved=0');
  }
});

// POST: Update career advisor availability settings
router.post('/career-advisor-interviews/settings', async (req, res) => {
  try {
    const { datePattern, availableDays, availableTimes, daysAhead, specificDates, excludedDates } = req.body;
    
    // Parse available days
    const parsedDays = Array.isArray(availableDays) 
      ? availableDays.map(d => parseInt(d)).filter(d => !isNaN(d))
      : (availableDays ? [parseInt(availableDays)].filter(d => !isNaN(d)) : [1, 2, 3, 4, 5]);
    
    // Parse available times (split by newlines)
    const parsedTimes = availableTimes 
      ? availableTimes.split('\n').map(t => t.trim()).filter(t => t.length > 0)
      : careerAdvisorSettings.DEFAULT_SETTINGS.availableTimes;
    
    // Parse specific dates
    const parsedSpecificDates = specificDates
      ? specificDates.split('\n').map(d => d.trim()).filter(d => d.match(/^\d{4}-\d{2}-\d{2}$/))
      : [];
    
    // Parse excluded dates
    const parsedExcludedDates = excludedDates
      ? excludedDates.split('\n').map(d => d.trim()).filter(d => d.match(/^\d{4}-\d{2}-\d{2}$/))
      : [];
    
    const settings = {
      datePattern: datePattern || 'biweekly',
      availableDays: parsedDays,
      availableTimes: parsedTimes,
      daysAhead: parseInt(daysAhead) || 40,
      specificDates: parsedSpecificDates,
      excludedDates: parsedExcludedDates
    };
    
    careerAdvisorSettings.updateSettings(settings);
    res.redirect('/admin/career-advisor-interviews?settingsSaved=1');
  } catch (err) {
    console.error('Error saving career advisor settings:', err);
    res.redirect('/admin/career-advisor-interviews?settingsSaved=0');
  }
});

// POST: Add meeting link to existing interview
router.post('/career-advisor-interviews/:id/add-link', async (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    const { meetingLink } = req.body;
    
    if (!meetingLink) {
      return res.redirect('/admin/career-advisor-interviews?saved=0');
    }
    
    const interview = careerAdvisorInterviewModel.getById(id);
    if (!interview) {
      return res.redirect('/admin/career-advisor-interviews?saved=0');
    }
    
    // Update interview with meeting link
    careerAdvisorInterviewModel.update(id, { meetingLink });
    
    // Send email with meeting link
    const dateObj = new Date(interview.date + 'T12:00:00');
    const formattedDate = dateObj.toLocaleDateString('en-US', { 
      weekday: 'long', 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    });
    
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: interview.studentEmail,
      subject: 'Meeting Link for Your Career Advisor Interview',
      html: `
        <h2>Meeting Link Ready</h2>
        <p>Dear ${interview.studentName},</p>
        <p>Here is the meeting link for your upcoming career advisor interview:</p>
        <p><strong>Meeting Link:</strong> <a href="${meetingLink}">${meetingLink}</a></p>
        <p><strong>Date:</strong> ${formattedDate}<br>
        <strong>Time:</strong> ${interview.time}</p>
        <p>See you soon!</p>
        <br>
        <p>Best regards,<br>Career Services Team</p>
      `,
      text: `Meeting Link Ready\n\nDear ${interview.studentName},\n\nMeeting Link: ${meetingLink}\nDate: ${formattedDate}\nTime: ${interview.time}\n\nSee you soon!\n\nBest regards,\nCareer Services Team`
    });
    
    res.redirect('/admin/career-advisor-interviews?saved=1');
  } catch (err) {
    console.error('Error adding meeting link:', err);
    res.redirect('/admin/career-advisor-interviews?saved=0');
  }
});

// POST: Update interview status
router.post('/career-advisor-interviews/:id/status', async (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    const { status } = req.body;
    
    if (!['pending', 'confirmed', 'completed', 'cancelled'].includes(status)) {
      return res.status(400).json({ ok: false, error: 'Invalid status' });
    }
    
    const interview = careerAdvisorInterviewModel.getById(id);
    if (!interview) {
      return res.status(404).json({ ok: false, error: 'Interview not found' });
    }
    
    careerAdvisorInterviewModel.update(id, { status });
    
    // Add to student's interactions
    try {
      const lead = await leadModel.getByEmail(interview.studentEmail);
      if (lead) {
        await leadModel.addContact(lead.id, {
          type: 'career-advisor-status',
          notes: `Career advisor interview status updated to: ${status}`
        });
      }
    } catch (err) {
      console.error('Failed to add status update to interactions:', err);
    }
    
    res.json({ ok: true });
  } catch (err) {
    console.error('Error updating interview status:', err);
    res.status(500).json({ ok: false, error: 'Server error' });
  }
});

// POST: Add notes to interview
router.post('/career-advisor-interviews/:id/notes', async (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    const { notes } = req.body;
    
    const interview = careerAdvisorInterviewModel.getById(id);
    if (!interview) {
      return res.redirect('/admin/career-advisor-interviews?saved=0');
    }
    
    careerAdvisorInterviewModel.update(id, { notes: notes || '' });
    
    res.redirect('/admin/career-advisor-interviews?saved=1');
  } catch (err) {
    console.error('Error adding notes:', err);
    res.redirect('/admin/career-advisor-interviews?saved=0');
  }
});

// POST: Send custom email to student
router.post('/career-advisor-interviews/:id/send-email', async (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    const { subject, body } = req.body;
    
    const interview = careerAdvisorInterviewModel.getById(id);
    if (!interview) {
      return res.redirect('/admin/career-advisor-interviews?saved=0');
    }
    
    // Send email to student
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: interview.studentEmail,
      subject: subject,
      html: body.replace(/\n/g, '<br>'),
      text: body
    });
    
    res.redirect('/admin/career-advisor-interviews?saved=1');
  } catch (err) {
    console.error('Error sending email:', err);
    res.redirect('/admin/career-advisor-interviews?saved=0');
  }
});

// Helper function to convert 12-hour time to 24-hour format
function convertTo24HourAdmin(time12h) {
  const [time, modifier] = time12h.split(' ');
  let [hours, minutes] = time.split(':');
  
  if (hours === '12') {
    hours = '00';
  }
  
  if (modifier === 'PM') {
    hours = parseInt(hours, 10) + 12;
  }
  
  return `${hours}:${minutes || '00'}:00`;
}

// ===== Student Career Services (Profile Section) =====

// POST: Add interview to student's career services
router.post('/students/:id/career-services/interview', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const { date, time, type, status, notes } = req.body;
    const profile = student.profile || {};
    const careerServices = profile.careerServices || {};
    const interviews = careerServices.interviews || [];
    
    interviews.push({
      date: new Date(date).toISOString(),
      time: time || '',
      type: type || 'General',
      status: status || 'scheduled',
      notes: notes || '',
      addedBy: req.session.user.name || 'Admin',
      addedAt: new Date().toISOString()
    });
    
    careerServices.interviews = interviews;
    await userModel.updateProfile(studentId, { careerServices });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error adding career interview:', error);
    res.status(500).json({ error: 'Failed to add interview' });
  }
});

// PUT: Update interview in student's career services
router.put('/students/:id/career-services/interview/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const index = Number(req.params.index);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const { date, time, type, status, notes } = req.body;
    const profile = student.profile || {};
    const careerServices = profile.careerServices || {};
    const interviews = careerServices.interviews || [];
    
    if (index < 0 || index >= interviews.length) {
      return res.status(400).json({ error: 'Invalid interview index' });
    }
    
    interviews[index] = {
      ...interviews[index],
      date: new Date(date).toISOString(),
      time: time || '',
      type: type || 'General',
      status: status || 'scheduled',
      notes: notes || '',
      updatedBy: req.session.user.name || 'Admin',
      updatedAt: new Date().toISOString()
    };
    
    careerServices.interviews = interviews;
    await userModel.updateProfile(studentId, { careerServices });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error updating career interview:', error);
    res.status(500).json({ error: 'Failed to update interview' });
  }
});

// DELETE: Remove interview from student's career services
router.delete('/students/:id/career-services/interview/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const index = Number(req.params.index);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const profile = student.profile || {};
    const careerServices = profile.careerServices || {};
    const interviews = careerServices.interviews || [];
    
    if (index < 0 || index >= interviews.length) {
      return res.status(400).json({ error: 'Invalid interview index' });
    }
    
    interviews.splice(index, 1);
    careerServices.interviews = interviews;
    await userModel.updateProfile(studentId, { careerServices });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error deleting career interview:', error);
    res.status(500).json({ error: 'Failed to delete interview' });
  }
});

// POST: Add note to student's career services
router.post('/students/:id/career-services/note', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const { title, content } = req.body;
    const profile = student.profile || {};
    const careerServices = profile.careerServices || {};
    const notes = careerServices.notes || [];
    
    notes.push({
      title: title || 'Career Note',
      content: content || '',
      addedBy: req.session.user.name || 'Admin',
      date: new Date().toISOString()
    });
    
    careerServices.notes = notes;
    await userModel.updateProfile(studentId, { careerServices });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error adding career note:', error);
    res.status(500).json({ error: 'Failed to add note' });
  }
});

// PUT: Update note in student's career services
router.put('/students/:id/career-services/note/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const index = Number(req.params.index);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const { title, content } = req.body;
    const profile = student.profile || {};
    const careerServices = profile.careerServices || {};
    const notes = careerServices.notes || [];
    
    if (index < 0 || index >= notes.length) {
      return res.status(400).json({ error: 'Invalid note index' });
    }
    
    notes[index] = {
      ...notes[index],
      title: title || 'Career Note',
      content: content || '',
      updatedBy: req.session.user.name || 'Admin',
      updatedAt: new Date().toISOString()
    };
    
    careerServices.notes = notes;
    await userModel.updateProfile(studentId, { careerServices });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error updating career note:', error);
    res.status(500).json({ error: 'Failed to update note' });
  }
});

// DELETE: Remove note from student's career services
router.delete('/students/:id/career-services/note/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const index = Number(req.params.index);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const profile = student.profile || {};
    const careerServices = profile.careerServices || {};
    const notes = careerServices.notes || [];
    
    if (index < 0 || index >= notes.length) {
      return res.status(400).json({ error: 'Invalid note index' });
    }
    
    notes.splice(index, 1);
    careerServices.notes = notes;
    await userModel.updateProfile(studentId, { careerServices });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error deleting career note:', error);
    res.status(500).json({ error: 'Failed to delete note' });
  }
});

// POST: Upload document to student's career services
router.post('/students/:id/career-services/document', (req, res) => {
  generalDocUpload(req, res, async (err) => {
    try {
      const studentId = Number(req.params.id);
      const student = await userModel.findById(studentId);
      
      if (!student || student.role !== 'student') {
        return res.status(404).json({ error: 'Student not found' });
      }
      
      if (err) {
        console.error('Career document upload error:', err);
        return res.status(400).json({ error: err.message || 'Upload failed' });
      }
      
      if (!req.file) {
        return res.status(400).json({ error: 'No file uploaded' });
      }
      
      const profile = student.profile || {};
      const careerServices = profile.careerServices || {};
      const documents = careerServices.documents || [];
      
      documents.push({
        originalName: req.file.originalname,
        mimeType: req.file.mimetype,
        size: req.file.size,
        url: `/docs/stxd/${req.file.filename}`,
        uploadedAt: new Date().toISOString(),
        uploadedBy: req.session.user.name || 'Admin'
      });
      
      careerServices.documents = documents;
      await userModel.updateProfile(studentId, { careerServices });
      
      res.json({ success: true });
    } catch (error) {
      console.error('Error uploading career document:', error);
      res.status(500).json({ error: 'Failed to upload document' });
    }
  });
});

// DELETE: Remove document from student's career services
router.delete('/students/:id/career-services/document/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const index = Number(req.params.index);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const profile = student.profile || {};
    const careerServices = profile.careerServices || {};
    const documents = careerServices.documents || [];
    
    if (index < 0 || index >= documents.length) {
      return res.status(400).json({ error: 'Invalid document index' });
    }
    
    const deletedDoc = documents[index];
    documents.splice(index, 1);
    careerServices.documents = documents;
    await userModel.updateProfile(studentId, { careerServices });
    
    // Delete physical file
    if (deletedDoc && deletedDoc.url) {
      const fs = require('fs');
      const path = require('path');
      const filePath = path.join(__dirname, '..', 'public', deletedDoc.url);
      
      fs.unlink(filePath, (err) => {
        if (err) console.warn('Failed to delete physical file:', filePath, err);
        else console.log('Deleted physical file:', filePath);
      });
    }
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error deleting career document:', error);
    res.status(500).json({ error: 'Failed to delete document' });
  }
});

// POST: Add/Update employment survey for student
router.post('/students/:id/career-services/employment-survey', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const { employmentStatus, jobTitle, employer, startDate, salaryRange, relatedToTraining, surveyDate, notes } = req.body;
    const profile = student.profile || {};
    const careerServices = profile.careerServices || {};
    
    careerServices.employmentSurvey = {
      employmentStatus: employmentStatus || '',
      jobTitle: jobTitle || '',
      employer: employer || '',
      startDate: startDate || null,
      salaryRange: salaryRange || '',
      relatedToTraining: relatedToTraining === true || relatedToTraining === 'true',
      surveyDate: surveyDate || new Date().toISOString(),
      notes: notes || '',
      recordedBy: req.session.user.name || 'Admin',
      recordedAt: new Date().toISOString()
    };
    
    await userModel.updateProfile(studentId, { careerServices });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error recording employment survey:', error);
    res.status(500).json({ error: 'Failed to record survey' });
  }
});

// PUT: Update employment survey for student
router.put('/students/:id/career-services/employment-survey', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const { employmentStatus, jobTitle, employer, startDate, salaryRange, relatedToTraining, surveyDate, notes } = req.body;
    const profile = student.profile || {};
    const careerServices = profile.careerServices || {};
    
    careerServices.employmentSurvey = {
      ...(careerServices.employmentSurvey || {}),
      employmentStatus: employmentStatus || '',
      jobTitle: jobTitle || '',
      employer: employer || '',
      startDate: startDate || null,
      salaryRange: salaryRange || '',
      relatedToTraining: relatedToTraining === true || relatedToTraining === 'true',
      surveyDate: surveyDate || new Date().toISOString(),
      notes: notes || '',
      updatedBy: req.session.user.name || 'Admin',
      updatedAt: new Date().toISOString()
    };
    
    await userModel.updateProfile(studentId, { careerServices });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error updating employment survey:', error);
    res.status(500).json({ error: 'Failed to update survey' });
  }
});

// ===== Certification Exam Results Management =====

// POST: Add certification exam result for a student in a specific class
router.post('/students/:id/cert-exam/:classId', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const classId = Number(req.params.classId);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const { provider, examName, result, score, examDate, notes } = req.body;
    
    if (!provider || !examName || !result || !examDate) {
      return res.status(400).json({ error: 'Missing required fields' });
    }
    
    const profile = student.profile || {};
    const certExamResults = profile.certExamResults || {};
    
    // Initialize array for this class if it doesn't exist
    if (!certExamResults[classId]) {
      certExamResults[classId] = [];
    }
    
    // Add the new exam result
    certExamResults[classId].push({
      provider,
      examName,
      result,
      score: score || '',
      examDate: new Date(examDate).toISOString(),
      notes: notes || '',
      recordedBy: req.session.user.name || 'Admin',
      recordedAt: new Date().toISOString()
    });
    
    await userModel.updateProfile(studentId, { certExamResults });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error adding cert exam result:', error);
    res.status(500).json({ error: 'Failed to add exam result' });
  }
});

// GET: Get a specific certification exam result
router.get('/students/:id/cert-exam/:classId/:examIndex', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const classId = Number(req.params.classId);
    const examIndex = Number(req.params.examIndex);
    
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const profile = student.profile || {};
    const certExamResults = profile.certExamResults || {};
    const classExams = certExamResults[classId] || [];
    
    if (examIndex < 0 || examIndex >= classExams.length) {
      return res.status(404).json({ error: 'Exam result not found' });
    }
    
    res.json(classExams[examIndex]);
  } catch (error) {
    console.error('Error fetching cert exam result:', error);
    res.status(500).json({ error: 'Failed to fetch exam result' });
  }
});

// PUT: Update certification exam result
router.put('/students/:id/cert-exam/:classId/:examIndex', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const classId = Number(req.params.classId);
    const examIndex = Number(req.params.examIndex);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const { provider, examName, result, score, examDate, notes } = req.body;
    
    const profile = student.profile || {};
    const certExamResults = profile.certExamResults || {};
    const classExams = certExamResults[classId] || [];
    
    if (examIndex < 0 || examIndex >= classExams.length) {
      return res.status(404).json({ error: 'Exam result not found' });
    }
    
    // Update the exam result
    classExams[examIndex] = {
      ...classExams[examIndex],
      provider,
      examName,
      result,
      score: score || '',
      examDate: new Date(examDate).toISOString(),
      notes: notes || '',
      updatedBy: req.session.user.name || 'Admin',
      updatedAt: new Date().toISOString()
    };
    
    certExamResults[classId] = classExams;
    await userModel.updateProfile(studentId, { certExamResults });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error updating cert exam result:', error);
    res.status(500).json({ error: 'Failed to update exam result' });
  }
});

// DELETE: Remove certification exam result
router.delete('/students/:id/cert-exam/:classId/:examIndex', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const classId = Number(req.params.classId);
    const examIndex = Number(req.params.examIndex);
    const student = await userModel.findById(studentId);
    
    if (!student || student.role !== 'student') {
      return res.status(404).json({ error: 'Student not found' });
    }
    
    const profile = student.profile || {};
    const certExamResults = profile.certExamResults || {};
    const classExams = certExamResults[classId] || [];
    
    if (examIndex < 0 || examIndex >= classExams.length) {
      return res.status(404).json({ error: 'Exam result not found' });
    }
    
    // Remove the exam result
    classExams.splice(examIndex, 1);
    certExamResults[classId] = classExams;
    
    await userModel.updateProfile(studentId, { certExamResults });
    
    res.json({ success: true });
  } catch (error) {
    console.error('Error deleting cert exam result:', error);
    res.status(500).json({ error: 'Failed to delete exam result' });
  }
});

// ============ SCHEDULE BUILDER ROUTES ============

// Schedule Builder main page
router.get('/schedule-builder', async (req, res) => {
  const users = await userModel.getAll();
  const teachers = users.filter(u => u.role === 'teacher' && u.status === 'approved');
  const classes = await classModel.getAllClasses();
  const sessions = await scheduleModel.getAllScheduledSessions();
  const programTracks = await scheduleModel.getAllProgramTracks();
  const locations = await scheduleModel.getAllLocations();
  const cohorts = await scheduleModel.getUniqueCohorts();
  const courses = await scheduleModel.getUniqueCourses();
  
  // Create teacher map for easy lookup
  const teacherMap = {};
  teachers.forEach(t => { teacherMap[t.id] = t.name; });
  
  res.render('admin_schedule_builder', {
    user: req.session.user,
    branding,
    teachers,
    teacherMap,
    classes,
    sessions,
    programTracks,
    locations,
    cohorts,
    courses
  });
});

// Teacher Availability page
router.get('/schedule-builder/teacher-availability', async (req, res) => {
  const users = await userModel.getAll();
  const teachers = users.filter(u => u.role === 'teacher' && u.status === 'approved');
  const allAvailability = await scheduleModel.getAllTeacherAvailability();
  
  // Group availability by teacher
  const availabilityByTeacher = {};
  teachers.forEach(t => { availabilityByTeacher[t.id] = []; });
  allAvailability.forEach(a => {
    if (availabilityByTeacher[a.teacherId]) {
      availabilityByTeacher[a.teacherId].push(a);
    }
  });
  
  res.render('admin_teacher_availability', {
    user: req.session.user,
    branding,
    teachers,
    availabilityByTeacher
  });
});

// Get availability for a specific teacher (API)
router.get('/schedule-builder/teacher-availability/:teacherId', async (req, res) => {
  const teacherId = Number(req.params.teacherId);
  const availability = await scheduleModel.getTeacherAvailability(teacherId);
  res.json(availability);
});

// Set teacher availability (API)
router.post('/schedule-builder/teacher-availability/:teacherId', async (req, res) => {
  const teacherId = Number(req.params.teacherId);
  const { slots } = req.body;
  try {
    const availability = await scheduleModel.setTeacherAvailability(teacherId, slots || []);
    res.json({ success: true, availability });
  } catch (e) {
    console.error('Error setting teacher availability:', e);
    res.status(500).json({ error: 'Failed to save availability' });
  }
});

// Add single availability slot (API)
router.post('/schedule-builder/availability-slot', async (req, res) => {
  try {
    const id = await scheduleModel.addTeacherAvailabilitySlot(req.body.teacherId, req.body);
    res.json({ success: true, id });
  } catch (e) {
    console.error('Error adding availability slot:', e);
    res.status(500).json({ error: 'Failed to add slot' });
  }
});

// Delete availability slot (API)
router.delete('/schedule-builder/availability-slot/:id', async (req, res) => {
  try {
    await scheduleModel.deleteTeacherAvailabilitySlot(Number(req.params.id));
    res.json({ success: true });
  } catch (e) {
    console.error('Error deleting availability slot:', e);
    res.status(500).json({ error: 'Failed to delete slot' });
  }
});

// Get all scheduled sessions (API)
router.get('/schedule-builder/sessions', async (req, res) => {
  const { start, end, teacherId, cohort, courseId } = req.query;
  let sessions;
  
  if (teacherId) {
    sessions = await scheduleModel.getScheduledSessionsByTeacher(Number(teacherId), start, end);
  } else if (cohort) {
    sessions = await scheduleModel.getScheduledSessionsByCohort(cohort, start, end);
  } else if (courseId) {
    sessions = await scheduleModel.getScheduledSessionsByCourse(Number(courseId), start, end);
  } else if (start && end) {
    sessions = await scheduleModel.getScheduledSessionsInRange(start, end);
  } else {
    sessions = await scheduleModel.getAllScheduledSessions();
  }
  
  res.json(sessions);
});

// Create scheduled session (API)
router.post('/schedule-builder/sessions', async (req, res) => {
  try {
    // Check for conflicts
    const conflicts = await scheduleModel.checkTeacherConflicts(
      req.body.teacherId,
      req.body.sessionDate,
      req.body.startTime,
      req.body.endTime
    );
    
    if (conflicts.length > 0) {
      return res.status(400).json({ 
        error: 'Teacher conflict detected', 
        conflicts 
      });
    }
    
    const session = { ...req.body, createdBy: req.session.user?.id };
    const id = await scheduleModel.createScheduledSession(session);
    res.json({ success: true, id });
  } catch (e) {
    console.error('Error creating session:', e);
    res.status(500).json({ error: 'Failed to create session' });
  }
});

// Bulk create sessions (API)
router.post('/schedule-builder/sessions/bulk', async (req, res) => {
  try {
    const { sessions } = req.body;
    const results = { created: [], conflicts: [] };
    
    for (const session of sessions) {
      const conflicts = await scheduleModel.checkTeacherConflicts(
        session.teacherId,
        session.sessionDate,
        session.startTime,
        session.endTime
      );
      
      if (conflicts.length > 0) {
        results.conflicts.push({ session, conflicts });
      } else {
        const id = await scheduleModel.createScheduledSession({
          ...session,
          createdBy: req.session.user?.id
        });
        results.created.push({ ...session, id });
      }
    }
    
    res.json(results);
  } catch (e) {
    console.error('Error bulk creating sessions:', e);
    res.status(500).json({ error: 'Failed to create sessions' });
  }
});

// Bulk update sessions (API) - Must be before :id route
router.put('/schedule-builder/sessions/bulk-update', async (req, res) => {
  try {
    const { scope, courseName, cohortName, updates } = req.body;
    
    console.log('Bulk update request:', { scope, courseName, cohortName, updates });
    
    if (!scope || !updates || Object.keys(updates).length === 0) {
      return res.status(400).json({ error: 'Invalid bulk update request - missing scope or updates' });
    }
    
    let whereClause = '';
    const params = [];
    
    if (scope === 'course') {
      whereClause = 'WHERE courseName = ?';
      params.push(courseName);
    } else if (scope === 'cohort') {
      whereClause = 'WHERE cohortName = ?';
      params.push(cohortName);
    } else if (scope === 'course_cohort') {
      whereClause = 'WHERE courseName = ? AND cohortName = ?';
      params.push(courseName, cohortName);
    } else {
      return res.status(400).json({ error: 'Invalid scope: ' + scope });
    }
    
    // Build SET clause from updates
    const setClause = [];
    const setParams = [];
    
    if (updates.startTime !== undefined) {
      setClause.push('startTime = ?');
      setParams.push(updates.startTime);
    }
    if (updates.endTime !== undefined) {
      setClause.push('endTime = ?');
      setParams.push(updates.endTime);
    }
    if (updates.teacherId !== undefined) {
      setClause.push('teacherId = ?');
      setParams.push(updates.teacherId);
    }
    if (updates.locationOnline !== undefined) {
      setClause.push('locationOnline = ?');
      setParams.push(updates.locationOnline);
    }
    if (updates.locationPhysical !== undefined) {
      setClause.push('locationPhysical = ?');
      setParams.push(updates.locationPhysical || null);
    }
    if (updates.colorCode !== undefined) {
      setClause.push('colorCode = ?');
      setParams.push(updates.colorCode);
    }
    if (updates.programTrack !== undefined) {
      setClause.push('programTrack = ?');
      setParams.push(updates.programTrack || null);
    }
    
    if (setClause.length === 0) {
      return res.status(400).json({ error: 'No valid fields to update' });
    }
    
    const query = `UPDATE mdtslms_scheduled_sessions SET ${setClause.join(', ')} ${whereClause}`;
    console.log('Bulk update query:', query, [...setParams, ...params]);
    const [result] = await db.query(query, [...setParams, ...params]);
    
    res.json({ success: true, updated: result.affectedRows });
  } catch (e) {
    console.error('Error bulk updating sessions:', e);
    res.status(500).json({ error: 'Failed to bulk update sessions: ' + e.message });
  }
});

// Update scheduled session (API)
router.put('/schedule-builder/sessions/:id', async (req, res) => {
  try {
    const id = Number(req.params.id);
    
    if (isNaN(id) || id <= 0) {
      return res.status(400).json({ error: 'Invalid session ID' });
    }
    
    // Validate required fields
    if (!req.body.sessionDate || !req.body.startTime || !req.body.endTime) {
      return res.status(400).json({ error: 'Missing required fields (date, start time, or end time)' });
    }
    
    // Check for conflicts (exclude current session)
    const conflicts = await scheduleModel.checkTeacherConflicts(
      req.body.teacherId,
      req.body.sessionDate,
      req.body.startTime,
      req.body.endTime,
      id
    );
    
    if (conflicts.length > 0) {
      return res.status(400).json({ 
        error: 'Teacher conflict detected', 
        conflicts 
      });
    }
    
    await scheduleModel.updateScheduledSession(id, req.body);
    res.json({ success: true });
  } catch (e) {
    console.error('Error updating session:', e);
    res.status(500).json({ error: 'Failed to update session' });
  }
});

// Bulk delete sessions (API) - Must be before :id route
router.delete('/schedule-builder/sessions/bulk-delete', async (req, res) => {
  try {
    const { scope, courseName, cohortName } = req.body;
    
    if (!scope) {
      return res.status(400).json({ error: 'Invalid bulk delete request' });
    }
    
    let whereClause = '';
    const params = [];
    
    if (scope === 'course') {
      whereClause = 'WHERE courseName = ?';
      params.push(courseName);
    } else if (scope === 'cohort') {
      whereClause = 'WHERE cohortName = ?';
      params.push(cohortName);
    } else if (scope === 'course_cohort') {
      whereClause = 'WHERE courseName = ? AND cohortName = ?';
      params.push(courseName, cohortName);
    } else {
      return res.status(400).json({ error: 'Invalid scope' });
    }
    
    const query = `DELETE FROM mdtslms_scheduled_sessions ${whereClause}`;
    const [result] = await db.query(query, params);
    
    res.json({ success: true, deleted: result.affectedRows });
  } catch (e) {
    console.error('Error bulk deleting sessions:', e);
    res.status(500).json({ error: 'Failed to bulk delete sessions' });
  }
});

// Clear all sessions (use with caution!)
router.delete('/schedule-builder/sessions/all', async (req, res) => {
  try {
    const count = await scheduleModel.clearAllScheduledSessions();
    res.json({ success: true, deleted: count });
  } catch (e) {
    console.error('Error clearing sessions:', e);
    res.status(500).json({ error: 'Failed to clear sessions' });
  }
});

// Delete scheduled session (API)
router.delete('/schedule-builder/sessions/:id', async (req, res) => {
  try {
    await scheduleModel.deleteScheduledSession(Number(req.params.id));
    res.json({ success: true });
  } catch (e) {
    console.error('Error deleting session:', e);
    res.status(500).json({ error: 'Failed to delete session' });
  }
});

// Check for conflicts (API)
router.post('/schedule-builder/check-conflicts', async (req, res) => {
  const { teacherId, sessionDate, startTime, endTime, excludeSessionId } = req.body;
  const conflicts = await scheduleModel.checkTeacherConflicts(
    teacherId, sessionDate, startTime, endTime, excludeSessionId
  );
  res.json({ hasConflicts: conflicts.length > 0, conflicts });
});

// Import CSV schedule (API)
router.post('/schedule-builder/import', async (req, res) => {
  try {
    const { parsedData, teacherMapping, courseConfig, defaultTrack, year, clearExisting } = req.body;
    
    if (!parsedData || !Array.isArray(parsedData)) {
      return res.status(400).json({ error: 'Invalid CSV data' });
    }
    
    // Optionally clear existing sessions first
    if (clearExisting) {
      await scheduleModel.clearAllScheduledSessions();
    }
    
    const results = await scheduleModel.importScheduleFromCSV(parsedData, {
      teacherMapping: teacherMapping || {},
      courseConfig: courseConfig || {},
      defaultTrack: defaultTrack || 'Cyber II Track',
      createdBy: req.session.user?.id,
      skipConflicts: !clearExisting,
      year: year || new Date().getFullYear()
    });
    
    res.json(results);
  } catch (e) {
    console.error('Error importing CSV schedule:', e);
    res.status(500).json({ error: 'Failed to import schedule: ' + e.message });
  }
});

// Get teachers for CSV import mapping
router.get('/schedule-builder/import/teachers', async (req, res) => {
  try {
    const [rows] = await db.query(
      `SELECT id, CONCAT(first_name, ' ', last_name) as name 
       FROM mdtslms_students WHERE role IN ('teacher', 'admin') 
       ORDER BY first_name, last_name`
    );
    res.json(rows);
  } catch (e) {
    console.error('Error fetching teachers:', e);
    res.status(500).json({ error: 'Failed to fetch teachers' });
  }
});

// Clear all sessions (use with caution!)
router.delete('/schedule-builder/sessions/all', async (req, res) => {
  try {
    const count = await scheduleModel.clearAllScheduledSessions();
    res.json({ success: true, deleted: count });
  } catch (e) {
    console.error('Error clearing sessions:', e);
    res.status(500).json({ error: 'Failed to clear sessions' });
  }
});

// Program Tracks CRUD
router.get('/schedule-builder/program-tracks', async (req, res) => {
  const tracks = await scheduleModel.getAllProgramTracks();
  res.json(tracks);
});

router.post('/schedule-builder/program-tracks', async (req, res) => {
  try {
    const id = await scheduleModel.createProgramTrack(req.body);
    res.json({ success: true, id });
  } catch (e) {
    console.error('Error creating program track:', e);
    res.status(500).json({ error: 'Failed to create program track' });
  }
});

router.put('/schedule-builder/program-tracks/:id', async (req, res) => {
  try {
    await scheduleModel.updateProgramTrack(Number(req.params.id), req.body);
    res.json({ success: true });
  } catch (e) {
    console.error('Error updating program track:', e);
    res.status(500).json({ error: 'Failed to update program track' });
  }
});

router.delete('/schedule-builder/program-tracks/:id', async (req, res) => {
  try {
    await scheduleModel.deleteProgramTrack(Number(req.params.id));
    res.json({ success: true });
  } catch (e) {
    console.error('Error deleting program track:', e);
    res.status(500).json({ error: 'Failed to delete program track' });
  }
});

// Locations
router.get('/schedule-builder/locations', async (req, res) => {
  const locations = await scheduleModel.getAllLocations();
  res.json(locations);
});

router.post('/schedule-builder/locations', async (req, res) => {
  try {
    const id = await scheduleModel.createLocation(req.body);
    res.json({ success: true, id });
  } catch (e) {
    console.error('Error creating location:', e);
    res.status(500).json({ error: 'Failed to create location' });
  }
});

// Export schedule to Excel
router.get('/schedule-builder/export', async (req, res) => {
  const { type, id, start, end, format } = req.query;
  let sessions;
  let filename = 'schedule';
  
  if (type === 'teacher' && id) {
    sessions = await scheduleModel.getScheduledSessionsByTeacher(Number(id), start, end);
    const teacher = await userModel.findById(Number(id));
    filename = `schedule_teacher_${teacher?.name?.replace(/\s+/g, '_') || id}`;
  } else if (type === 'cohort' && id) {
    sessions = await scheduleModel.getScheduledSessionsByCohort(id, start, end);
    filename = `schedule_cohort_${id.replace(/\s+/g, '_')}`;
  } else if (type === 'course' && id) {
    sessions = await scheduleModel.getScheduledSessionsByCourse(Number(id), start, end);
    filename = `schedule_course_${id}`;
  } else {
    sessions = start && end 
      ? await scheduleModel.getScheduledSessionsInRange(start, end)
      : await scheduleModel.getAllScheduledSessions();
    filename = 'schedule_all';
  }
  
  // Get teacher names
  const users = await userModel.getAll();
  const teacherMap = {};
  users.filter(u => u.role === 'teacher').forEach(t => { teacherMap[t.id] = t.name; });
  
  if (format === 'csv') {
    // Generate CSV
    const headers = ['Cohort', 'Course', 'Date', 'Start Time', 'End Time', 'Teacher', 'Location (Online)', 'Location (Physical)', 'Program Track', 'Notes'];
    const rows = sessions.map(s => [
      s.cohortName,
      s.courseName,
      s.sessionDate,
      s.startTime,
      s.endTime,
      teacherMap[s.teacherId] || `Teacher #${s.teacherId}`,
      s.locationOnline ? 'Yes' : 'No',
      s.locationPhysical || '',
      s.programTrack || '',
      s.notes || ''
    ]);
    
    const csv = [headers.join(','), ...rows.map(r => r.map(c => `"${String(c).replace(/"/g, '""')}"`).join(','))].join('\n');
    
    res.setHeader('Content-Type', 'text/csv');
    res.setHeader('Content-Disposition', `attachment; filename="${filename}.csv"`);
    res.send(csv);
  } else {
    // Return JSON (can be used by frontend to generate Excel)
    res.json({
      filename,
      headers: ['Cohort', 'Course', 'Date', 'Start Time', 'End Time', 'Teacher', 'Location (Online)', 'Location (Physical)', 'Program Track', 'Notes'],
      data: sessions.map(s => ({
        cohort: s.cohortName,
        course: s.courseName,
        date: s.sessionDate,
        startTime: s.startTime,
        endTime: s.endTime,
        teacher: teacherMap[s.teacherId] || `Teacher #${s.teacherId}`,
        locationOnline: s.locationOnline ? 'Yes' : 'No',
        locationPhysical: s.locationPhysical || '',
        programTrack: s.programTrack || '',
        notes: s.notes || '',
        colorCode: s.colorCode
      }))
    });
  }
});

// ========== TEACHER TIME-OFF / VACATION MANAGEMENT ==========

// Get all time-off entries (or filter by teacher)
router.get('/schedule-builder/time-off', async (req, res) => {
  try {
    const { teacherId } = req.query;
    if (teacherId) {
      const timeOff = await scheduleModel.getTeacherTimeOff(Number(teacherId));
      res.json(timeOff);
    } else {
      const timeOff = await scheduleModel.getAllTimeOff();
      res.json(timeOff);
    }
  } catch (e) {
    console.error('Error getting time-off:', e);
    res.status(500).json({ error: 'Failed to get time-off entries' });
  }
});

// Add teacher time-off
router.post('/schedule-builder/time-off', async (req, res) => {
  try {
    const { teacherId, startDate, endDate, reason } = req.body;
    if (!teacherId || !startDate || !endDate) {
      return res.status(400).json({ error: 'Teacher, start date, and end date are required' });
    }
    const id = await scheduleModel.addTimeOff({
      teacherId: Number(teacherId),
      startDate,
      endDate,
      title: reason || 'Time Off',
      timeOffType: 'vacation'
    });
    res.json({ success: true, id });
  } catch (e) {
    console.error('Error adding time-off:', e);
    res.status(500).json({ error: 'Failed to add time-off' });
  }
});

// Delete teacher time-off
router.delete('/schedule-builder/time-off/:id', async (req, res) => {
  try {
    await scheduleModel.deleteTimeOff(Number(req.params.id));
    res.json({ success: true });
  } catch (e) {
    console.error('Error deleting time-off:', e);
    res.status(500).json({ error: 'Failed to delete time-off' });
  }
});

// Update teacher time-off
router.put('/schedule-builder/time-off/:id', async (req, res) => {
  try {
    const { teacherId, startDate, endDate, reason } = req.body;
    if (!teacherId || !startDate || !endDate) {
      return res.status(400).json({ error: 'Teacher, start date, and end date are required' });
    }
    await scheduleModel.updateTimeOff(Number(req.params.id), {
      teacherId: Number(teacherId),
      startDate,
      endDate,
      title: reason || 'Time Off',
      timeOffType: 'vacation'
    });
    res.json({ success: true });
  } catch (e) {
    console.error('Error updating time-off:', e);
    res.status(500).json({ error: 'Failed to update time-off' });
  }
});

// ========== HOLIDAYS MANAGEMENT ==========

// Get all holidays (with optional date range)
router.get('/schedule-builder/holidays', async (req, res) => {
  try {
    const { year } = req.query;
    const holidays = await scheduleModel.getSchoolHolidays(year ? parseInt(year) : null);
    res.json(holidays);
  } catch (e) {
    console.error('Error getting holidays:', e);
    res.status(500).json({ error: 'Failed to get holidays' });
  }
});

// Add holiday
router.post('/schedule-builder/holidays', async (req, res) => {
  try {
    const { name, date, recurring } = req.body;
    if (!name || !date) {
      return res.status(400).json({ error: 'Name and date are required' });
    }
    const id = await scheduleModel.addSchoolHoliday({
      name,
      date,
      isRecurringYearly: recurring === true || recurring === 'true' || recurring === 1
    });
    res.json({ success: true, id });
  } catch (e) {
    console.error('Error adding holiday:', e);
    res.status(500).json({ error: 'Failed to add holiday' });
  }
});

// Delete holiday
router.delete('/schedule-builder/holidays/:id', async (req, res) => {
  try {
    await scheduleModel.deleteSchoolHoliday(Number(req.params.id));
    res.json({ success: true });
  } catch (e) {
    console.error('Error deleting holiday:', e);
    res.status(500).json({ error: 'Failed to delete holiday' });
  }
});

// Update holiday
router.put('/schedule-builder/holidays/:id', async (req, res) => {
  try {
    const { name, date, recurring } = req.body;
    if (!name || !date) {
      return res.status(400).json({ error: 'Name and date are required' });
    }
    await scheduleModel.updateSchoolHoliday(Number(req.params.id), {
      name,
      date,
      isRecurringYearly: recurring === true || recurring === 'true' || recurring === 1
    });
    res.json({ success: true });
  } catch (e) {
    console.error('Error updating holiday:', e);
    res.status(500).json({ error: 'Failed to update holiday' });
  }
});

// ========== WEEK-SPECIFIC SCHEDULE CHECK ==========

// Check teacher availability for a specific week (accounts for time-off and holidays)
router.get('/schedule-builder/week-availability', async (req, res) => {
  try {
    const { teacherId, weekStart } = req.query;
    if (!teacherId || !weekStart) {
      return res.status(400).json({ error: 'Teacher ID and week start date are required' });
    }
    
    // Get the week dates (Monday to Sunday)
    const startDate = new Date(weekStart);
    const weekDates = [];
    for (let i = 0; i < 7; i++) {
      const date = new Date(startDate);
      date.setDate(startDate.getDate() + i);
      weekDates.push(date.toISOString().split('T')[0]);
    }
    
    // Check each day
    const availability = {};
    for (const date of weekDates) {
      const isAvailable = await scheduleModel.isTeacherAvailableOnDate(Number(teacherId), date);
      const dayName = new Date(date).toLocaleDateString('en-US', { weekday: 'long' }).toLowerCase();
      availability[date] = {
        dayName,
        available: isAvailable
      };
    }
    
    // Get time-off entries for this week
    const timeOff = await scheduleModel.getTeacherTimeOff(Number(teacherId));
    const weekTimeOff = timeOff.filter(t => {
      return weekDates.some(d => d >= t.startDate && d <= t.endDate);
    });
    
    // Get holidays for this week
    const holidays = await scheduleModel.getHolidaysInRange(weekDates[0], weekDates[6]);
    
    res.json({
      weekStart: weekDates[0],
      weekEnd: weekDates[6],
      availability,
      timeOff: weekTimeOff,
      holidays
    });
  } catch (e) {
    console.error('Error checking week availability:', e);
    res.status(500).json({ error: 'Failed to check week availability' });
  }
});

module.exports = router;// Admins list + actions
router.get('/admins', async (req, res) => {
  const admins = await userModel.getByRole('admin');
  res.render('admin_admins', { user: req.session.user, admins, error: null, saved: req.query.saved });
});

router.post('/admins/:id/reset', async (req, res) => {
  const id = Number(req.params.id);
  const admin = await userModel.findById(id);
  if (!admin || admin.role !== 'admin') return res.redirect('/admin/admins?saved=0');
  const crypto = require('crypto');
  const password = crypto.randomBytes(6).toString('base64');
  await userModel.updatePassword(admin.username, password);
  try {
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: admin.email,
      subject: 'Your MDTS Admin Password Was Reset',
      text: `Hello ${admin.name || admin.username},

Your admin password has been reset.

Username: ${admin.username}
New Password: ${password}

Login: https://mdts-apps.com/admin`,
      html: `<p>Hello ${admin.name || admin.username},</p><p>Your admin password has been reset.</p><p><strong>Username:</strong> ${admin.username}<br><strong>New Password:</strong> ${password}</p><p><a href="https://lms.mdts-apps.com/admin">Login</a></p>`
    });
  } catch (e) { console.error('Admin reset email failed', e); }
  res.redirect('/admin/admins?saved=1');
});

router.post('/admins/:id/delete', async (req, res) => {
  const id = Number(req.params.id);
  // Prevent deleting yourself
  if (req.session.user && req.session.user.id === id) return res.redirect('/admin/admins?saved=0');
  
  const admin = await userModel.findById(id);
  if (!admin || admin.role !== 'admin') return res.redirect('/admin/admins?saved=0');
  await userModel.deleteById(id);
  res.redirect('/admin/admins?saved=1');
});
const notificationEvents = [
  {
    key: 'preRegister',
    title: 'Student Pre-registers',
    description: 'Triggered when a new prospective student submits the pre-registration form.'
  },
  {
    key: 'register',
    title: 'Student Registers',
    description: 'Triggered once a student completes the full registration process.'
  },
  {
    key: 'eventSignup',
    title: 'Event Signup',
    description: 'Triggered when someone RSVPs or signs up for a hosted event.'
  },
  {
    key: 'studentApproved',
    title: 'Student Approved',
    description: 'Triggered when an admin approves a student application.'
  },
  {
    key: 'classEnrollment',
    title: 'Student Added to Class',
    description: 'Triggered when a student is enrolled into a class section.'
  },
  {
    key: 'careerAdvisorInterview',
    title: 'Career Advisor Interview Scheduled',
    description: 'Triggered when a student schedules a career advisor interview.'
  }
];
