/**
 * Document Signing Routes Module
 * Plug-and-play routes for document signing functionality
 * Similar to DocuSign
 */

const express = require('express');
const router = express.Router();
const multer = require('multer');
const path = require('path');
const fs = require('fs').promises;
const crypto = require('crypto');
const signatureDocumentModel = require('../models/signatureDocumentModel');
const documentSigningUtils = require('../utils/documentSigning');
const pdfToImage = require('../utils/pdfToImage');

// Configure multer for PDF uploads
const storage = multer.diskStorage({
  destination: async (req, file, cb) => {
    const uploadDir = path.join(__dirname, '../uploads/signature-documents');
    try {
      await fs.mkdir(uploadDir, { recursive: true });
      cb(null, uploadDir);
    } catch (error) {
      cb(error);
    }
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + crypto.randomBytes(6).toString('hex');
    cb(null, 'doc-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({
  storage: storage,
  fileFilter: (req, file, cb) => {
    if (file.mimetype === 'application/pdf') {
      cb(null, true);
    } else {
      cb(new Error('Only PDF files are allowed'));
    }
  },
  limits: {
    fileSize: 10 * 1024 * 1024 // 10MB limit
  }
});

// Middleware to check if user is admin
function requireAdmin(req, res, next) {
  if (!req.session || req.session.role !== 'admin') {
    return res.status(403).json({ error: 'Admin access required' });
  }
  next();
}

// Middleware to check if user is authenticated
function requireAuth(req, res, next) {
  if (!req.session || !req.session.user || !req.session.user.id) {
    // Store the original URL to redirect back after login
    req.session.returnTo = req.originalUrl;
    return res.redirect('/login');
  }
  next();
}

/**
 * GET /document-signing/student/:studentId
 * View all signature documents for a student (Admin only)
 */
router.get('/student/:studentId', requireAdmin, async (req, res) => {
  try {
    const documents = await signatureDocumentModel.getDocumentsByStudent(req.params.studentId);
    res.render('admin_signature_documents', {
      user: req.session.user,
      role: req.session.role,
      student_id: req.params.studentId,
      documents: documents
    });
  } catch (error) {
    console.error('Error loading signature documents:', error);
    res.status(500).send('Error loading signature documents');
  }
});

/**
 * GET /document-signing/create/:studentId
 * Show form to create new signature document (Admin only)
 */
router.get('/create/:studentId', requireAdmin, async (req, res) => {
  try {
    const db = require('../models/db');
    const [student] = await db.query('SELECT * FROM mdtslms_users WHERE id = ?', [req.params.studentId]);
    
    if (student.length === 0) {
      return res.status(404).send('Student not found');
    }

    // Get all available templates
    const templates = await signatureDocumentModel.getAllTemplates();

    res.render('admin_signature_document_create', {
      user: req.session.user,
      role: req.session.role,
      student: student[0],
      templates: templates
    });
  } catch (error) {
    console.error('Error loading create form:', error);
    res.status(500).send('Error loading form');
  }
});

/**
 * POST /document-signing/upload
 * Upload a PDF document or create from template (Admin only)
 */
router.post('/upload', requireAdmin, upload.array('documents', 20), async (req, res) => {
  try {
    const { student_id, title, description, signing_order, template_id } = req.body;

    if (!student_id || !title) {
      return res.status(400).json({ error: 'Student ID and title are required' });
    }

    let documentData;
    let documentId;

    // Check if using a template
    if (template_id && template_id !== '') {
      // Create document from template
      const template = await signatureDocumentModel.getTemplateById(parseInt(template_id));
      
      if (!template) {
        return res.status(404).json({ error: 'Template not found' });
      }

      // Use template's PDF file
      documentData = {
        student_id: parseInt(student_id),
        created_by: req.session.user.id,
        original_filename: template.original_filename,
        stored_filename: template.stored_filename,
        file_path: template.file_path,
        title: title,
        description: description || template.description,
        signing_order: signing_order || 'any_order'
      };

      documentId = await signatureDocumentModel.createDocument(documentData);

      // Copy fields from template to document
      if (template.fields && template.fields.length > 0) {
        const fieldsToAdd = template.fields.map(field => ({
          field_type: field.field_type,
          signer_type: field.signer_type,
          field_label: field.field_label,
          page_number: field.page_number,
          x_position: field.x_position,
          y_position: field.y_position,
          width: field.width,
          height: field.height,
          is_required: field.is_required,
          placeholder_text: field.placeholder_text
        }));
        
        await signatureDocumentModel.addFields(documentId, fieldsToAdd);
      }

      // Log audit
      await signatureDocumentModel.logAudit(
        documentId,
        req.session.user.id,
        'document_created_from_template',
        `Document "${title}" created from template "${template.template_name}"`,
        req.ip
      );

    } else {
      // Regular file upload
      if (!req.files || req.files.length === 0) {
        return res.status(400).json({ error: 'No files uploaded and no template selected' });
      }

      // If multiple files, merge them
      let filePath, originalFilename, storedFilename;

      if (req.files.length === 1) {
        // Single file - use as is
        filePath = req.files[0].path;
        originalFilename = req.files[0].originalname;
        storedFilename = req.files[0].filename;
      } else {
        // Multiple files - merge them using pdf-lib
        const PDFDocument = require('pdf-lib').PDFDocument;
        const fs = require('fs').promises;
        const path = require('path');

        // Create merged PDF
        const mergedPdf = await PDFDocument.create();

        // Load and copy pages from each file in order
        for (const file of req.files) {
          const pdfBytes = await fs.readFile(file.path);
          const pdf = await PDFDocument.load(pdfBytes);
          const copiedPages = await mergedPdf.copyPages(pdf, pdf.getPageIndices());
          copiedPages.forEach(page => mergedPdf.addPage(page));
        }

        // Save merged PDF
        const mergedPdfBytes = await mergedPdf.save();
        const mergedFilename = `merged-${Date.now()}.pdf`;
        const mergedPath = path.join('uploads', mergedFilename);
        
        await fs.writeFile(mergedPath, mergedPdfBytes);

        // Clean up individual files
        for (const file of req.files) {
          await fs.unlink(file.path).catch(err => console.error('Error deleting temp file:', err));
        }

        filePath = mergedPath;
        originalFilename = `${req.files.map(f => f.originalname).join(', ')}`;
        storedFilename = mergedFilename;
      }

      documentData = {
        student_id: parseInt(student_id),
        created_by: req.session.user.id,
        original_filename: originalFilename,
        stored_filename: storedFilename,
        file_path: filePath,
        title: title,
        description: description,
        signing_order: signing_order || 'any_order'
      };

      documentId = await signatureDocumentModel.createDocument(documentData);

      // Log audit
      await signatureDocumentModel.logAudit(
        documentId,
        req.session.user.id,
        'document_uploaded',
        req.files.length > 1 
          ? `Document "${title}" uploaded (${req.files.length} files merged)`
          : `Document "${title}" uploaded`,
        req.ip
      );
    }

    res.json({
      success: true,
      documentId: documentId,
      message: template_id ? 'Document created from template successfully' : 'Document uploaded successfully'
    });
  } catch (error) {
    console.error('Error uploading document:', error);
    res.status(500).json({ error: 'Error uploading document' });
  }
});

/**
 * GET /document-signing/pdf/:documentId
 * Serve the PDF file for browser-side rendering (original, for editing and signing)
 */
router.get('/pdf/:documentId', requireAuth, async (req, res) => {
  try {
    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    
    if (!document) {
      return res.status(404).send('Document not found');
    }

    // Check access - admin, party to document, or student
    const isAdmin = req.session.role === 'admin';
    const isParty = document.parties.some(p => p.user_id === req.session.user.id);
    const isStudent = document.student_id === req.session.user.id;

    if (!isAdmin && !isParty && !isStudent) {
      return res.status(403).send('Access denied');
    }

    res.setHeader('Content-Type', 'application/pdf');
    res.setHeader('Cache-Control', 'public, max-age=86400'); // Cache for 1 day
    res.sendFile(document.file_path);
  } catch (error) {
    console.error('Error serving PDF:', error);
    res.status(500).send('Error loading PDF');
  }
});

/**
 * GET /document-signing/pdf-signed/:documentId
 * Serve the PDF with signatures overlaid
 */
router.get('/pdf-signed/:documentId', requireAuth, async (req, res) => {
  try {
    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    
    if (!document) {
      return res.status(404).send('Document not found');
    }

    // Check access
    const isAdmin = req.session.role === 'admin';
    const isParty = document.parties.some(p => p.user_id === req.session.user.id);
    const isStudent = document.student_id === req.session.user.id;

    if (!isAdmin && !isParty && !isStudent) {
      return res.status(403).send('Access denied');
    }

    // Generate PDF with signatures
    const pdfBuffer = await documentSigningUtils.generateSignedPDF(req.params.documentId);

    res.setHeader('Content-Type', 'application/pdf');
    res.setHeader('Content-Disposition', 'inline');
    res.send(pdfBuffer);
  } catch (error) {
    console.error('Error serving signed PDF:', error);
    res.status(500).send('Error loading PDF');
  }
});

/**
 * GET /document-signing/edit/:documentId
 * Edit document and place signature fields (Admin only)
 */
router.get('/edit/:documentId', requireAdmin, async (req, res) => {
  try {
    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    
    if (!document) {
      return res.status(404).send('Document not found');
    }

    const db = require('../models/db');
    const [student] = await db.query('SELECT * FROM mdtslms_users WHERE id = ?', [document.student_id]);
    const [staff] = await db.query(`
      SELECT id,
        JSON_UNQUOTE(JSON_EXTRACT(profile, '$.firstName')) as firstName,
        JSON_UNQUOTE(JSON_EXTRACT(profile, '$.lastName')) as lastName,
        email, role 
      FROM mdtslms_users 
      WHERE role = "admin"
      ORDER BY JSON_UNQUOTE(JSON_EXTRACT(profile, '$.firstName')), JSON_UNQUOTE(JSON_EXTRACT(profile, '$.lastName'))
    `);

    // Get number of pages in PDF
    const pageCount = await pdfToImage.getPdfPageCount(document.file_path);

    res.render('admin_signature_document_edit', {
      user: req.session.user,
      role: req.session.role,
      document: document,
      student: student[0],
      staffList: staff,
      pageCount: pageCount
    });
  } catch (error) {
    console.error('Error loading document editor:', error);
    res.status(500).send('Error loading document');
  }
});

/**
 * POST /document-signing/fields/:documentId
 * Save signature fields for a document (Admin only)
 */
router.post('/fields/:documentId', requireAdmin, async (req, res) => {
  try {
    const { fields } = req.body;

    if (!fields || !Array.isArray(fields)) {
      return res.status(400).json({ error: 'Fields array is required' });
    }

    // Delete existing fields
    const db = require('../models/db');
    await db.query('DELETE FROM signature_fields WHERE document_id = ?', [req.params.documentId]);

    // Add new fields
    if (fields.length > 0) {
      await signatureDocumentModel.addFields(req.params.documentId, fields);
    }

    // Log audit
    await signatureDocumentModel.logAudit(
      req.params.documentId,
      req.session.user.id,
      'fields_updated',
      `${fields.length} fields configured`,
      req.ip
    );

    res.json({ success: true, message: 'Fields saved successfully' });
  } catch (error) {
    console.error('Error saving fields:', error);
    res.status(500).json({ error: 'Error saving fields' });
  }
});

/**
 * POST /document-signing/parties/:documentId
 * Set signing parties and send for signature (Admin only)
 */
router.post('/parties/:documentId', requireAdmin, async (req, res) => {
  try {
    const { parties } = req.body;

    if (!parties || !Array.isArray(parties) || parties.length === 0) {
      return res.status(400).json({ error: 'At least one signing party is required' });
    }

    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    if (!document) {
      return res.status(404).json({ error: 'Document not found' });
    }

    // Delete existing parties
    const db = require('../models/db');
    await db.query('DELETE FROM signature_parties WHERE document_id = ?', [req.params.documentId]);

    // Add new parties
    await signatureDocumentModel.addParties(req.params.documentId, parties);

    // Update document status
    await signatureDocumentModel.updateDocumentStatus(req.params.documentId, 'awaiting_signatures');

    // Send emails to signing parties
    await documentSigningUtils.sendSignatureRequestEmails(req.params.documentId);

    // Log audit
    await signatureDocumentModel.logAudit(
      req.params.documentId,
      req.session.user.id,
      'sent_for_signature',
      `Document sent to ${parties.length} parties`,
      req.ip
    );

    res.json({ success: true, message: 'Document sent for signature' });
  } catch (error) {
    console.error('Error sending document for signature:', error);
    res.status(500).json({ error: 'Error sending document' });
  }
});

/**
 * GET /document-signing/sign/:documentId/:partyId
 * View document for signing
 */
router.get('/sign/:documentId/:partyId', requireAuth, async (req, res) => {
  try {
    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    
    if (!document) {
      return res.status(404).send('Document not found');
    }

    // Find the party record
    const party = document.parties.find(p => p.id == req.params.partyId);
    
    if (!party) {
      return res.status(404).send('Signing party not found');
    }

    // Verify the user has access to this party record
    // Allow admins to sign on behalf of staff members
    const isAdmin = req.session.role === 'admin';
    const isPartyUser = party.user_id === req.session.user.id;
    const canSignAsStaff = isAdmin && party.party_type === 'staff';
    
    if (!isPartyUser && !canSignAsStaff) {
      return res.status(403).send('Access denied');
    }

    // Check signing order
    const canSign = await documentSigningUtils.canPartySign(document, party);
    
    // Get fields for this signer
    const fields = await signatureDocumentModel.getFieldsForSigner(
      req.params.documentId,
      party.party_type
    );

    // Get existing responses for this party
    const myResponses = await signatureDocumentModel.getResponsesForParty(req.params.partyId);
    
    // Also get all responses from other parties who have signed (to display existing signatures)
    const allResponses = [];
    for (const p of document.parties) {
      if (p.status === 'signed' || p.id == req.params.partyId) {
        const partyResponses = await signatureDocumentModel.getResponsesForParty(p.id);
        allResponses.push(...partyResponses);
      }
    }
    
    const responses = allResponses;

    // Mark as viewed
    if (party.status === 'pending') {
      await signatureDocumentModel.updatePartyStatus(req.params.partyId, 'viewed', {
        ip_address: req.ip,
        user_agent: req.get('user-agent')
      });
      await signatureDocumentModel.logAudit(
        req.params.documentId,
        req.session.user.id,
        'document_viewed',
        `Document viewed by ${party.party_type}`,
        req.ip
      );
    }

    res.render('signature_document_sign', {
      user: req.session.user,
      role: req.session.role,
      document: document,
      party: party,
      fields: fields,
      responses: responses,
      canSign: canSign
    });
  } catch (error) {
    console.error('Error loading signing page:', error);
    res.status(500).send('Error loading document');
  }
});

/**
 * POST /document-signing/sign/:documentId/:partyId
 * Submit signature and field responses
 */
router.post('/sign/:documentId/:partyId', requireAuth, async (req, res) => {
  try {
    const { fieldResponses, signatureData } = req.body;

    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    const party = document.parties.find(p => p.id == req.params.partyId);

    // Allow admins to sign on behalf of staff members
    const isAdmin = req.session.role === 'admin';
    const isPartyUser = party && party.user_id === req.session.user.id;
    const canSignAsStaff = isAdmin && party && party.party_type === 'staff';

    if (!party || (!isPartyUser && !canSignAsStaff)) {
      return res.status(403).json({ error: 'Access denied' });
    }

    // Check signing order
    const canSign = await documentSigningUtils.canPartySign(document, party);
    if (!canSign) {
      return res.status(400).json({ error: 'Not your turn to sign yet' });
    }

    // Save field responses
    if (fieldResponses && Array.isArray(fieldResponses)) {
      for (const response of fieldResponses) {
        await signatureDocumentModel.saveFieldResponse(
          response.field_id,
          req.params.partyId,
          response.value,
          response.type
        );
      }
    }

    // Update party status to signed
    await signatureDocumentModel.updatePartyStatus(req.params.partyId, 'signed', {
      signature_data: signatureData,
      ip_address: req.ip,
      user_agent: req.get('user-agent')
    });

    // Log audit
    await signatureDocumentModel.logAudit(
      req.params.documentId,
      req.session.user.id,
      'document_signed',
      `Document signed by ${party.party_type}`,
      req.ip
    );

    // Send progress notification to all parties
    const signerName = `${party.firstName} ${party.lastName}`;
    await documentSigningUtils.sendProgressNotificationEmail(
      req.params.documentId,
      signerName,
      party.party_type
    ).catch(err => {
      console.error('Error sending progress notification:', err);
      // Don't fail the signature process if email fails
    });

    // Check if all parties have signed
    const isComplete = await signatureDocumentModel.checkIfComplete(req.params.documentId);
    console.log(`Document ${req.params.documentId} completion check: ${isComplete ? 'COMPLETE' : 'INCOMPLETE'}`);

    if (isComplete) {
      await signatureDocumentModel.updateDocumentStatus(req.params.documentId, 'completed');
      console.log(`Document ${req.params.documentId} status updated to COMPLETED`);
      
      // Send completion emails with final PDF to all parties
      try {
        await documentSigningUtils.sendCompletionEmails(req.params.documentId);
        console.log(`✅ Completion emails sent for document ${req.params.documentId}`);
      } catch (emailError) {
        console.error('❌ Error sending completion emails:', emailError);
        // Continue even if email fails - document is still completed
      }
      
      await signatureDocumentModel.logAudit(
        req.params.documentId,
        req.session.user.id,
        'document_completed',
        'All parties have signed - completion emails sent',
        req.ip
      );
    } else {
      await signatureDocumentModel.updateDocumentStatus(req.params.documentId, 'partially_signed');
      
      // Notify next signer if applicable
      await documentSigningUtils.notifyNextSigner(req.params.documentId, party);
    }

    res.json({
      success: true,
      message: 'Document signed successfully',
      isComplete: isComplete
    });
  } catch (error) {
    console.error('Error signing document:', error);
    res.status(500).json({ error: 'Error signing document' });
  }
});

/**
 * GET /document-signing/pending
 * View documents pending signature for current user
 */
router.get('/pending', requireAuth, async (req, res) => {
  try {
    const documents = await signatureDocumentModel.getPendingDocumentsForUser(req.session.user.id);
    res.render('signature_documents_pending', {
      user: req.session.user,
      role: req.session.role,
      documents: documents
    });
  } catch (error) {
    console.error('Error loading pending documents:', error);
    res.status(500).send('Error loading documents');
  }
});

/**
 * GET /document-signing/view/:documentId
 * View completed document (read-only)
 */
router.get('/view/:documentId', requireAuth, async (req, res) => {
  try {
    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    
    if (!document) {
      return res.status(404).send('Document not found');
    }

    // Check if user has access (admin, or party to the document)
    const isAdmin = req.session.role === 'admin';
    const isParty = document.parties.some(p => p.user_id === req.session.user.id);
    const isStudent = document.student_id === req.session.user.id;

    if (!isAdmin && !isParty && !isStudent) {
      return res.status(403).send('Access denied');
    }

    // Get all responses
    const allResponses = [];
    for (const party of document.parties) {
      const responses = await signatureDocumentModel.getResponsesForParty(party.id);
      allResponses.push(...responses.map(r => ({ ...r, party_type: party.party_type })));
    }

    const auditLog = await signatureDocumentModel.getAuditLog(req.params.documentId);

    res.render('signature_document_view', {
      user: req.session.user,
      role: req.session.role,
      document: document,
      responses: allResponses,
      auditLog: auditLog
    });
  } catch (error) {
    console.error('Error viewing document:', error);
    res.status(500).send('Error loading document');
  }
});

/**
 * GET /document-signing/download/:documentId
 * Download the PDF with signatures
 */
router.get('/download/:documentId', requireAuth, async (req, res) => {
  try {
    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    
    if (!document) {
      return res.status(404).send('Document not found');
    }

    // Check access
    const isAdmin = req.session.role === 'admin';
    const isParty = document.parties.some(p => p.user_id === req.session.user.id);
    const isStudent = document.student_id === req.session.user.id;

    if (!isAdmin && !isParty && !isStudent) {
      return res.status(403).send('Access denied');
    }

    // Generate PDF with signatures
    const pdfBuffer = await documentSigningUtils.generateSignedPDF(req.params.documentId);

    res.setHeader('Content-Type', 'application/pdf');
    res.setHeader('Content-Disposition', `attachment; filename="${document.title.replace(/[^a-z0-9]/gi, '_')}_signed.pdf"`);
    res.send(pdfBuffer);
  } catch (error) {
    console.error('Error downloading document:', error);
    res.status(500).send('Error downloading document');
  }
});

/**
 * POST /document-signing/mark-complete/:documentId
 * Admin can manually mark a document as complete
 */
router.post('/mark-complete/:documentId', requireAdmin, async (req, res) => {
  try {
    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    
    if (!document) {
      return res.status(404).json({ error: 'Document not found' });
    }

    if (document.status === 'completed') {
      return res.status(400).json({ error: 'Document is already completed' });
    }

    // Update document status to completed
    await signatureDocumentModel.updateDocumentStatus(req.params.documentId, 'completed');
    console.log(`Admin manually marked document ${req.params.documentId} as COMPLETED`);
    
    // Send completion emails with final PDF to all parties
    try {
      await documentSigningUtils.sendCompletionEmails(req.params.documentId);
      console.log(`✅ Completion emails sent for document ${req.params.documentId} (admin override)`);
    } catch (emailError) {
      console.error('❌ Error sending completion emails:', emailError);
      // Continue even if email fails - document is still completed
    }
    
    // Log audit
    await signatureDocumentModel.logAudit(
      req.params.documentId,
      req.session.user.id,
      'document_completed',
      'Document manually marked as complete by admin - completion emails sent',
      req.ip
    );

    res.json({ 
      success: true, 
      message: 'Document marked as complete and emails sent to all parties' 
    });
  } catch (error) {
    console.error('Error marking document as complete:', error);
    res.status(500).json({ error: 'Error marking document as complete' });
  }
});

/**
 * DELETE /document-signing/:documentId
 * Delete a document (Admin only)
 */
router.delete('/:documentId', requireAdmin, async (req, res) => {
  try {
    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    
    if (!document) {
      return res.status(404).json({ error: 'Document not found' });
    }

    // Delete the file
    try {
      await fs.unlink(document.file_path);
    } catch (err) {
      console.error('Error deleting file:', err);
    }

    // Delete from database
    await signatureDocumentModel.deleteDocument(req.params.documentId);

    res.json({ success: true, message: 'Document deleted successfully' });
  } catch (error) {
    console.error('Error deleting document:', error);
    res.status(500).json({ error: 'Error deleting document' });
  }
});

/**
 * POST /document-signing/initialize-db
 * Initialize database tables (Admin only, run once)
 */
router.post('/initialize-db', requireAdmin, async (req, res) => {
  try {
    await signatureDocumentModel.initializeTables();
    res.json({ success: true, message: 'Database tables initialized successfully' });
  } catch (error) {
    console.error('Error initializing database:', error);
    res.status(500).json({ error: 'Error initializing database' });
  }
});

/**
 * POST /document-signing/save-as-template/:documentId
 * Save a document as a reusable template (Admin only)
 */
router.post('/save-as-template/:documentId', requireAdmin, async (req, res) => {
  try {
    const { templateName } = req.body;
    
    if (!templateName || templateName.trim() === '') {
      return res.status(400).send('Template name is required');
    }

    // Check if document exists
    const document = await signatureDocumentModel.getDocumentById(req.params.documentId);
    if (!document) {
      return res.status(404).send('Document not found');
    }

    // Check if any parties have signed
    const hasSignatures = document.parties.some(p => p.status === 'signed');
    if (hasSignatures) {
      return res.status(400).send('Cannot save a document with signatures as a template');
    }

    // Save as template
    const result = await signatureDocumentModel.saveAsTemplate(
      req.params.documentId,
      templateName.trim(),
      req.session.user.id
    );

    await signatureDocumentModel.logAudit(
      req.params.documentId,
      req.session.user.id,
      'template_created',
      `Saved as template: ${templateName}`,
      req.ip
    );

    res.json({ success: true, templateId: result.templateId });
  } catch (error) {
    console.error('Error saving template:', error);
    res.status(500).send('Error saving template');
  }
});

/**
 * GET /document-signing/templates
 * List all available templates (Admin only)
 */
router.get('/templates', requireAdmin, async (req, res) => {
  try {
    const templates = await signatureDocumentModel.getAllTemplates();
    res.render('signature_document_templates', {
      user: req.session.user,
      role: req.session.role,
      templates: templates
    });
  } catch (error) {
    console.error('Error loading templates:', error);
    res.status(500).send('Error loading templates');
  }
});

/**
 * GET /document-signing/template/:templateId
 * Get template details for creating a document from template
 */
router.get('/template/:templateId', requireAdmin, async (req, res) => {
  try {
    const template = await signatureDocumentModel.getTemplateById(req.params.templateId);
    if (!template) {
      return res.status(404).send('Template not found');
    }
    res.json(template);
  } catch (error) {
    console.error('Error loading template:', error);
    res.status(500).send('Error loading template');
  }
});

/**
 * GET /document-signing/template/:templateId/pdf
 * Serve template PDF file for preview
 */
router.get('/template/:templateId/pdf', requireAdmin, async (req, res) => {
  try {
    const template = await signatureDocumentModel.getTemplateById(req.params.templateId);
    
    if (!template) {
      return res.status(404).send('Template not found');
    }

    // Check if file exists
    try {
      await fs.access(template.file_path);
    } catch (error) {
      return res.status(404).send('Template PDF file not found');
    }

    // Send PDF file
    res.setHeader('Content-Type', 'application/pdf');
    res.setHeader('Content-Disposition', `inline; filename="${template.original_filename}"`);
    res.sendFile(template.file_path);
  } catch (error) {
    console.error('Error serving template PDF:', error);
    res.status(500).send('Error loading template PDF');
  }
});

/**
 * DELETE /document-signing/template/:templateId
 * Delete a template (Admin only)
 */
router.delete('/template/:templateId', requireAdmin, async (req, res) => {
  try {
    await signatureDocumentModel.deleteTemplate(req.params.templateId);
    res.json({ success: true, message: 'Template deleted successfully' });
  } catch (error) {
    console.error('Error deleting template:', error);
    res.status(500).json({ error: 'Error deleting template' });
  }
});

module.exports = router;
