const fs = require('fs');
const path = require('path');

const SETTINGS_PATH = path.join(__dirname, '..', 'data', 'notificationSettings.json');

const DEFAULTS = {
  preRegister: [],
  register: [],
  eventSignup: [],
  studentApproved: [],
  classEnrollment: []
};

function ensureShape(raw) {
  const out = { ...DEFAULTS };
  if (raw && typeof raw === 'object') {
    for (const key of Object.keys(DEFAULTS)) {
      const val = raw[key];
      if (Array.isArray(val)) {
        out[key] = [...new Set(val.map((v) => Number(v)).filter((n) => Number.isInteger(n) && n > 0))];
      }
    }
  }
  return out;
}

async function loadSettings() {
  try {
    const data = await fs.promises.readFile(SETTINGS_PATH, 'utf8');
    if (!data) return { ...DEFAULTS };
    const parsed = JSON.parse(data);
    return ensureShape(parsed);
  } catch (err) {
    if (err.code === 'ENOENT') {
      await saveSettings(DEFAULTS);
      return { ...DEFAULTS };
    }
    console.error('Notification settings load failed', err);
    return { ...DEFAULTS };
  }
}

async function saveSettings(settings) {
  const normalized = ensureShape(settings);
  const tmpPath = `${SETTINGS_PATH}.tmp`;
  await fs.promises.mkdir(path.dirname(SETTINGS_PATH), { recursive: true });
  await fs.promises.writeFile(tmpPath, JSON.stringify(normalized, null, 2), 'utf8');
  await fs.promises.rename(tmpPath, SETTINGS_PATH);
  return normalized;
}

module.exports = {
  loadSettings,
  saveSettings,
  DEFAULTS
};
