# Student Roster Export Feature

## Overview
The Student Roster Export feature provides administrators and teachers with a professional way to view and export comprehensive student information for their classes. The roster includes detailed student information and can be exported to PDF or Excel formats with professional letterhead and branding.

## Feature Location
- **Admin View**: Available on the class detail page at `/admin/classes/:id`
- **Teacher View**: Available on the class detail page at `/teacher/classes/:id`

## Roster Information Included

### Student Details
1. **Student Name** - Full name of the student
2. **Email** - Student's email address
3. **Username** - Student's login username
4. **Cohort** - The cohort/group the student belongs to
5. **Enrollment Date** - Date when the student enrolled in the class (tracked in `studentEnrollments` array)
6. **Teacher** - Assigned teacher/instructor name
7. **Actions** (view only, not exported) - Quick action buttons for contacting students or viewing profiles

### Export Formats

#### PDF Export
- **Professional Letterhead** featuring:
  - School logo (from branding configuration)
  - School name
  - "Official Student Roster" designation
- **Document Header** with:
  - Class name
  - Cohort information
  - Instructor name
  - Start and end dates
- **Student Table** with all roster columns (excluding Actions)
- **Footer** with:
  - Generation timestamp
  - Confidentiality notice
- **Print-ready format** optimized for saving as PDF

#### Excel Export (CSV Format)
- **Header rows** with class metadata:
  - Class name
  - Cohort and teacher information
  - Generation date
- **Column headers** matching the roster table
- **All student data** in CSV format
- **Automatic filename** with class name and date (e.g., `roster_Class_Name_2024-01-15.csv`)

## User Interface

### Roster Section Header
```html
<section class="cardish">
  <div class="section-title">Student Roster</div>
  <button id="exportRosterPDF">Export PDF</button>
  <button id="exportRosterExcel">Export Excel</button>
</section>
```

### Table Structure
- **Responsive design** with horizontal scrolling on small screens
- **Striped rows** for better readability
- **Hover effects** on rows
- **Total student count** displayed at bottom
- **Help button** with tooltip explaining the roster feature

### Action Buttons (View Only)
- **Contact button** - Opens email composer (admin/teacher view)
- **View Profile button** - Links to student detail page
- **Grade Report button** - Shows student's grade summary (admin view only)

## Technical Implementation

### Files Modified

#### 1. `/views/view_class.ejs` (Admin View)
**Lines ~766-855**: Student Roster section added
- Replaced basic "Students" section with enhanced roster
- Added export buttons with IDs `exportRosterPDF` and `exportRosterExcel`
- Enhanced table structure with all required columns
- Enrollment date pulled from `klass.studentEnrollments` array

**Lines ~1200-1405**: Export functionality JavaScript added
- PDF generation with letterhead and professional styling
- Excel/CSV export with proper escaping and formatting
- Uses branding information for school name and logo

#### 2. `/views/teacher_view_class.ejs` (Teacher View)
**Lines ~965-1033**: Student Roster section added
- Same enhanced table structure as admin view
- Simplified action buttons (contact and view profile only)
- Help tooltip integrated

**Lines ~1438-1643**: Export functionality JavaScript added
- Identical PDF and Excel export functionality
- Teacher-specific context and branding

### Data Sources

#### Enrollment Tracking
The enrollment date is retrieved from the `studentEnrollments` array in the class object:
```javascript
const enrollment = (klass.studentEnrollments || []).find(e => e.studentId === s.id);
const enrolledDate = enrollment ? new Date(enrollment.enrolledAt).toLocaleDateString() : 'N/A';
```

This data is automatically tracked when students are added to a class via the `classModel.addStudent()` method.

#### Branding Configuration
The export uses branding information from the `branding` object:
- `branding.logo` - Path to school logo image
- `branding.schoolName` - Name of the institution
- Defaults to "MD Technical School" and "/images/logo.png" if not configured

### Export Functionality Details

#### PDF Generation Process
1. Opens new browser window
2. Injects professional HTML template with:
   - Embedded CSS for print-ready styling
   - Letterhead with logo and school information
   - Class metadata in organized grid
   - Student roster table (excluding Actions column)
   - Confidentiality footer with generation timestamp
3. Provides "Print/Save as PDF" button
4. User saves via browser's print dialog (Cmd+P / Ctrl+P → Save as PDF)

#### Excel/CSV Generation Process
1. Builds CSV string with:
   - Header rows with class information
   - Column headers from table
   - Student data rows with proper CSV escaping
2. Creates Blob with CSV content
3. Generates temporary download link
4. Triggers automatic download with formatted filename
5. Cleans up temporary DOM elements

### CSS Classes Used
- `.no-export` - Applied to Actions column to exclude from exports
- `.table-wrap` - Provides responsive scrolling for table
- `.empty` - Styled empty state when no students enrolled
- `.section-title` - Consistent section header styling

## Usage Instructions

### For Admins and Teachers

#### Viewing the Roster
1. Navigate to a class detail page
2. Scroll to the "Student Roster" section
3. View comprehensive student information in table format
4. Use action buttons to interact with individual students

#### Exporting to PDF
1. Click the "Export PDF" button
2. A new window opens with the professional roster document
3. Review the formatted roster with letterhead
4. Click "Print/Save as PDF" button
5. Choose "Save as PDF" in the print dialog
6. Select destination and save

#### Exporting to Excel
1. Click the "Export Excel" button
2. CSV file automatically downloads to your default download location
3. File is named with class name and current date
4. Open in Excel, Google Sheets, or any spreadsheet application

### Use Cases
- **Record Keeping** - Maintain official class rosters for documentation
- **Parent Communication** - Share class composition with stakeholders
- **Administrative Reporting** - Generate enrollment reports for management
- **Attendance Tracking** - Print rosters for manual attendance marking
- **Contact Lists** - Export student email lists for communication
- **Compliance** - Provide official enrollment records for regulatory requirements

## Best Practices

### For Staff
1. **Export regularly** to maintain up-to-date records
2. **Protect exported files** as they contain sensitive student information
3. **Verify data accuracy** before sharing with external parties
4. **Use PDF format** for official documentation
5. **Use Excel format** for data analysis or mail merges

### Data Privacy
- Exported rosters contain personally identifiable information (PII)
- Files should be stored securely
- Follow institutional policies for data handling
- PDF footers include confidentiality notices
- Only authorized staff can access export functionality

## Integration with Other Features

### Drip Schedule Feature
- Enrollment dates in roster are used for drip schedule calculations
- The `studentEnrollments` array tracks when each student joined
- This enables day-based content unlocking relative to enrollment

### Gradebook Feature
- Roster action buttons link to gradebook functionality
- Admin view includes "Grade Report" button for each student
- Student IDs in roster match IDs in grades data structure

### Messaging System
- "Contact" button integrates with messaging/email system
- Pre-fills recipient information from roster data
- Available in both admin and teacher views

## Troubleshooting

### Issue: Logo not appearing in PDF
**Solution**: 
- Check that `branding.logo` path is correct
- Verify logo file exists at specified path
- Ensure logo is in a web-accessible format (PNG, JPG, SVG)
- Check browser console for image loading errors

### Issue: "N/A" showing for enrollment date
**Solution**:
- Student may have been added before enrollment tracking was implemented
- Manually update `studentEnrollments` array in database
- Use admin interface to remove and re-add student to class

### Issue: Export buttons not working
**Solution**:
- Check browser console for JavaScript errors
- Verify that table has `id="rosterTable"`
- Ensure export button IDs are correct (`exportRosterPDF`, `exportRosterExcel`)
- Check that student data exists in the class

### Issue: CSV file opens incorrectly in Excel
**Solution**:
- Try "Open With" → Excel instead of double-clicking
- Use Excel's "Import Data" feature for better formatting
- Check regional settings (comma vs semicolon delimiters)

## Future Enhancements (Potential)

### Possible Additions
1. **Filtering Options**
   - Filter by cohort
   - Filter by enrollment date range
   - Filter by grade/performance level

2. **Additional Export Formats**
   - Server-side PDF generation with more customization
   - Native XLSX format (not just CSV)
   - HTML email-ready format

3. **Roster Customization**
   - Choose which columns to include
   - Add custom fields
   - Sort options in UI

4. **Batch Operations**
   - Email all students from roster
   - Export multiple class rosters at once
   - Print labels from roster data

5. **Enhanced Branding**
   - Multiple letterhead templates
   - Custom school colors and fonts
   - Department-specific branding

## Technical Notes

### Browser Compatibility
- **PDF Export**: Uses `window.open()` and `window.print()` - works in all modern browsers
- **Excel Export**: Uses Blob API and download attribute - IE11+ and all modern browsers
- **Fallback**: For older browsers, CSV data can be copied manually from preview window

### Performance Considerations
- Client-side export suitable for classes up to ~1000 students
- For larger datasets, consider server-side generation
- PDF generation opens in new tab to avoid blocking main UI
- CSV generation is near-instant regardless of class size

### Security Considerations
- Export functionality only available to authenticated admins and teachers
- Route-level authorization checks required (should be implemented in routes)
- No API endpoints created - all export happens client-side
- Sensitive data only rendered for users with proper role access

## Related Documentation
- See `DRIP_SCHEDULE_FEATURE.md` for enrollment date tracking details
- See `STUDENT_ENROLLMENT_MANAGEMENT.md` for student management features
- See branding configuration guide for logo and school name setup
