# Template PDF Preview with Field Overlays

## Overview
Enhanced the template preview modal to display the actual PDF document with visual field overlays showing exactly where each field is positioned.

## New Features

### Visual PDF Preview
- **Canvas-based rendering** using PDF.js
- **Field overlays** showing exact positions and sizes
- **Color-coded fields**:
  - Blue borders/background for student fields
  - Green borders/background for staff fields
- **Multi-page support** with navigation controls
- **Page counter** showing current page
- **Field labels** displayed on each overlay

### Interactive Preview
- **Previous/Next buttons** for multi-page documents
- **Field tooltips** showing field type on hover
- **Responsive canvas** scales to modal width
- **High-quality rendering** at 1.5x scale

## User Interface

### Preview Modal Layout

```
┌────────────────────────────────────────────────────────────┐
│ 👁 Template Preview: Enrollment Agreement           [X]   │
├────────────────────────────────────────────────────────────┤
│                                                            │
│ ┌─────────────────────────┐  ┌────────────────────────┐  │
│ │ 📄 Document Preview     │  │ ℹ Template Details     │  │
│ │ Page 1 of 3             │  │                        │  │
│ │                         │  │ Description:           │  │
│ │ ┌─────────────────────┐ │  │ Standard enrollment... │  │
│ │ │     PDF Canvas      │ │  │                        │  │
│ │ │  ┌───────────────┐  │ │  │ PDF File:              │  │
│ │ │  │Student Sig    │  │ │  │ enrollment.pdf         │  │
│ │ │  └───────────────┘  │ │  │                        │  │
│ │ │     ┌─────┐         │ │  │ Number of Fields: 12   │  │
│ │ │     │Date │         │ │  │                        │  │
│ │ │     └─────┘         │ │  │ Fields:                │  │
│ │ └─────────────────────┘ │  │ ┌──────────────────┐   │  │
│ │                         │  │ │Label   Type  Sign│   │  │
│ │ [← Previous] [Next →]   │  │ │Student sig   stud│   │  │
│ └─────────────────────────┘  │ │Date    date  stud│   │  │
│                              │ │Staff   sig   staf│   │  │
│                              │ └──────────────────┘   │  │
│                              │                        │  │
│                              │ ℹ This template will   │  │
│                              │   use the existing PDF │  │
│                              └────────────────────────┘  │
│                                                            │
│              [Close]  [✓ Use This Template]                │
└────────────────────────────────────────────────────────────┘
```

## Implementation Details

### Frontend Components

#### Modal HTML
```html
<div class="modal-dialog modal-xl">
  <div class="modal-content">
    <div class="modal-body">
      <div class="row">
        <!-- Left: PDF Preview -->
        <div class="col-md-7">
          <div id="previewPdfPage" style="position: relative;">
            <canvas id="previewPdfCanvas"></canvas>
            <!-- Field overlays appended here -->
          </div>
          <div class="d-flex justify-content-center gap-2">
            <button id="previewPrevBtn">Previous</button>
            <button id="previewNextBtn">Next</button>
          </div>
        </div>
        
        <!-- Right: Details -->
        <div class="col-md-5">
          <!-- Template info and field list -->
        </div>
      </div>
    </div>
  </div>
</div>
```

#### Field Overlay Styling
```css
.preview-field-overlay {
  position: absolute;
  border: 2px solid;
  background: rgba(255,255,255,0.3);
  border-radius: 4px;
  pointer-events: none;
  display: flex;
  align-items: center;
  justify-content: center;
  font-size: 0.75rem;
  font-weight: 600;
  padding: 2px 4px;
  box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.preview-field-overlay.student {
  border-color: #3b82f6;
  color: #1e40af;
  background: rgba(59, 130, 246, 0.15);
}

.preview-field-overlay.staff {
  border-color: #10b981;
  color: #065f46;
  background: rgba(16, 185, 129, 0.15);
}
```

### JavaScript Functions

#### loadTemplatePreviewPDF()
Loads PDF.js library if needed and renders first page.

```javascript
async function loadTemplatePreviewPDF(template) {
  // Store fields for page navigation
  window.currentPreviewFields = template.fields;
  
  const pdfUrl = `/document-signing/template/${template.id}/pdf`;
  
  // Load PDF.js if not present
  if (!window.pdfjsLib) {
    const script = document.createElement('script');
    script.src = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js';
    document.head.appendChild(script);
    await new Promise((resolve) => {
      script.onload = () => {
        pdfjsLib.GlobalWorkerOptions.workerSrc = '...pdf.worker.min.js';
        resolve();
      };
    });
  }
  
  previewPdfDoc = await pdfjsLib.getDocument(pdfUrl).promise;
  await renderPreviewPage(1, template.fields);
}
```

#### renderPreviewPage()
Renders specific page to canvas and adds field overlays.

```javascript
async function renderPreviewPage(pageNum, fields) {
  const page = await previewPdfDoc.getPage(pageNum);
  const canvas = document.getElementById('previewPdfCanvas');
  const context = canvas.getContext('2d');
  
  // Render at 1.5x scale for quality
  const viewport = page.getViewport({ scale: 1.5 });
  canvas.width = viewport.width;
  canvas.height = viewport.height;
  
  await page.render({
    canvasContext: context,
    viewport: viewport
  }).promise;
  
  // Add field overlays
  renderPreviewFieldOverlays(fields, pageNum);
  updatePreviewPageNavigation();
}
```

#### renderPreviewFieldOverlays()
Creates and positions field overlay divs.

```javascript
function renderPreviewFieldOverlays(fields, pageNum) {
  const pdfPage = document.getElementById('previewPdfPage');
  
  // Remove existing overlays
  pdfPage.querySelectorAll('.preview-field-overlay').forEach(el => el.remove());
  
  // Add overlays for current page fields
  fields.filter(f => (f.page_number || 1) === pageNum).forEach(field => {
    const overlay = document.createElement('div');
    overlay.className = `preview-field-overlay ${field.signer_type}`;
    overlay.style.left = `${field.x_position}%`;
    overlay.style.top = `${field.y_position}%`;
    overlay.style.width = `${field.width}%`;
    overlay.style.height = `${field.height}%`;
    overlay.textContent = field.field_label;
    overlay.title = `${field.field_label} (${field.field_type})`;
    
    pdfPage.appendChild(overlay);
  });
}
```

### Backend Route

#### GET /document-signing/template/:templateId/pdf
Serves template PDF file for preview.

```javascript
router.get('/template/:templateId/pdf', requireAdmin, async (req, res) => {
  try {
    const template = await signatureDocumentModel.getTemplateById(req.params.templateId);
    
    if (!template) {
      return res.status(404).send('Template not found');
    }

    // Check if file exists
    await fs.access(template.file_path);

    // Send PDF file
    res.setHeader('Content-Type', 'application/pdf');
    res.setHeader('Content-Disposition', `inline; filename="${template.original_filename}"`);
    res.sendFile(template.file_path);
  } catch (error) {
    console.error('Error serving template PDF:', error);
    res.status(500).send('Error loading template PDF');
  }
});
```

## Features

### PDF Rendering
- **PDF.js integration** for browser-based rendering
- **Dynamic script loading** - PDF.js loaded on demand
- **High-quality rendering** - 1.5x scale factor
- **Responsive canvas** - scales to container width
- **Loading indicator** - spinner while PDF loads
- **Error handling** - fallback message if PDF unavailable

### Field Visualization
- **Absolute positioning** using percentage coordinates
- **Color coding** by signer type
- **Field labels** displayed inside overlays
- **Tooltips** showing field type
- **Non-interactive** overlays (pointer-events: none)
- **Visible contrast** with semi-transparent backgrounds

### Page Navigation
- **Previous/Next buttons** for multi-page PDFs
- **Page counter** showing current/total
- **Button states** - disabled at boundaries
- **Field persistence** across page changes
- **Smooth transitions** between pages

### Modal Enhancements
- **Larger modal** - changed from `modal-lg` to `modal-xl`
- **Two-column layout** - PDF on left, details on right
- **Scrollable field list** - max height with overflow
- **Sticky table header** - header stays visible while scrolling
- **Responsive design** - stacks on mobile

## User Experience

### Preview Workflow
1. User clicks "Preview" on template card
2. Modal opens with loading spinner
3. PDF.js library loads (if first time)
4. PDF renders to canvas
5. Field overlays appear on PDF
6. User can navigate pages with buttons
7. Field list updates to show page fields
8. User can select template or close

### Visual Feedback
- **Loading state** - spinner and text
- **Field highlights** - clear borders and colors
- **Page context** - page number always visible
- **Navigation state** - disabled buttons at boundaries
- **Hover effects** - tooltips on field overlays

## Technical Details

### Dependencies
- **PDF.js 3.11.174** - CDN loaded dynamically
- **Bootstrap 5.3.3** - Modal and layout
- **Canvas API** - PDF rendering

### Performance
- **Lazy loading** - PDF.js loaded only when needed
- **Page-by-page rendering** - only current page rendered
- **Overlay reuse** - overlays recreated on page change
- **Memory management** - old overlays removed before new ones added

### Browser Compatibility
- **Modern browsers** - Chrome, Firefox, Safari, Edge
- **Canvas support** required
- **ES6+ JavaScript** - async/await used
- **PDF.js compatibility** - follows PDF.js requirements

## Error Handling

### Missing PDF File
```javascript
try {
  await fs.access(template.file_path);
} catch (error) {
  return res.status(404).send('Template PDF file not found');
}
```

### PDF Loading Failure
```javascript
catch (error) {
  console.error('Error loading template preview PDF:', error);
  document.getElementById('previewLoadingPdf').innerHTML = 
    '<div class="alert alert-warning small">Unable to load PDF preview...</div>';
}
```

### Script Loading Failure
- Graceful degradation if PDF.js doesn't load
- Console error logging for debugging
- User-friendly error messages

## Benefits

### For Admins
- **Visual confirmation** before selecting template
- **See exact field placement** without creating document
- **Multi-page preview** for complex documents
- **Quick comparison** between templates
- **Confidence in selection** knowing what will be created

### For System
- **Reduced errors** - admins see what they get
- **Better template adoption** - easier to understand
- **Less trial and error** - preview before committing
- **Improved UX** - visual vs text-only preview

## Files Modified

1. **views/admin_signature_document_create.ejs**
   - Changed modal size to `modal-xl`
   - Added two-column layout
   - Added canvas and loading indicator
   - Added page navigation buttons
   - Added field overlay styles
   - Added PDF rendering JavaScript
   - Added page navigation handlers

2. **routes/documentSigning.js**
   - Added GET `/template/:templateId/pdf` route
   - Serves template PDF file for preview
   - Added file existence check
   - Set proper content headers

## Testing Scenarios

### Test PDF Preview
1. ✅ Open template preview modal
2. ✅ See PDF loading spinner
3. ✅ PDF renders on canvas
4. ✅ Field overlays appear in correct positions
5. ✅ Fields have correct colors (blue/green)
6. ✅ Field labels are readable

### Test Page Navigation
1. ✅ Previous button disabled on page 1
2. ✅ Click Next - goes to page 2
3. ✅ Field overlays update for new page
4. ✅ Page counter updates correctly
5. ✅ Next button disabled on last page
6. ✅ Previous button works to go back

### Test Multi-Page Documents
1. ✅ Load template with 3+ pages
2. ✅ Navigate through all pages
3. ✅ Fields appear only on their respective pages
4. ✅ Page counter shows correct total

### Test Edge Cases
1. ✅ Template with no fields - shows empty PDF
2. ✅ Template with missing PDF - shows error message
3. ✅ Single-page PDF - both nav buttons disabled
4. ✅ Many fields on one page - all overlays visible

## Future Enhancements

1. **Zoom controls** - magnify PDF for detail
2. **Field highlighting** - click field in list to highlight on PDF
3. **Full-screen mode** - expand preview to full screen
4. **Thumbnail navigation** - page thumbnails for quick jump
5. **Download template PDF** - download option in preview
6. **Field editing** - modify fields directly in preview
7. **Side-by-side comparison** - compare two templates
8. **Annotation layer** - add notes to preview

## Related Documentation

- [TEMPLATE_SELECTION_AND_PREVIEW.md](./TEMPLATE_SELECTION_AND_PREVIEW.md) - Template selection feature
- [DOCUMENT_TEMPLATES_FEATURE.md](./DOCUMENT_TEMPLATES_FEATURE.md) - Template management
- [RESIZABLE_FIELDS_AND_PROGRESS_EMAILS.md](./RESIZABLE_FIELDS_AND_PROGRESS_EMAILS.md) - Field editing
