const express = require('express');
const router = express.Router();
const classModel = require('../models/classModel');
const userModel = require('../models/userModel');
const db = require('../models/db');
const { generateQuestions } = require('../utils/questionGenerator');
const announcementModel = require('../models/announcementModel');
const discussionModel = require('../models/discussionModel');
const messageModel = require('../models/messageModel');
const emailTemplates = require('../utils/emailTemplates');
const testModel = require('../models/testModel');
const resourceLibrary = require('../utils/resourceLibrary');

const fs = require('fs');
const path = require('path');
let marked;
try {
  marked = require('marked');
} catch (err) {
  console.warn('[teacher manual] marked package not found, falling back to plain text manual rendering');
  marked = null;
}
const nodemailer = require('nodemailer');
const crypto = require('crypto');
const PDFDocument = require('pdfkit');

const CONFIG_PATH = path.join(__dirname, '..', 'data', 'signature-docs.json');
let signatureDocsConfig = {};

function loadSignatureDocsConfig() {
  try {
    signatureDocsConfig = JSON.parse(fs.readFileSync(CONFIG_PATH, 'utf8'));
  } catch (err) {
    console.error('Failed to load signature-docs.json:', err.message);
    signatureDocsConfig = {};
  }
}
loadSignatureDocsConfig();


const transporter = nodemailer.createTransport({
  host: 'mdts-apps.com',
  port: 465,
  secure: true,
  auth: {
    user: 'noreply@mdts-apps.com',
    pass: 'c@r,5ysPI@&s'
  }
});
const multer = require('multer');
const upload = multer();

router.get('/manual', (req, res) => {
  if (!req.session || !['teacher', 'admin'].includes(req.session.role)) {
    return res.status(403).send('Forbidden');
  }
  const manualPath = path.join(__dirname, '..', 'docs', 'teacher-manual.md');
  let manualHtml = '';
  let updatedAt = null;
  try {
    const md = fs.readFileSync(manualPath, 'utf8');
    const render = (src, opts) => {
      if (!marked) {
        const escape = (str) => str
          .replace(/&/g, '&amp;')
          .replace(/</g, '&lt;')
          .replace(/>/g, '&gt;');
        return `<pre style="white-space:pre-wrap;">${escape(src)}</pre>`;
      }
      if (typeof marked.parse === 'function') return marked.parse(src, opts);
      return marked(src, opts);
    };
    manualHtml = render(md, { mangle: false, headerIds: true });
    const stats = fs.statSync(manualPath);
    updatedAt = stats.mtime;
  } catch (err) {
    console.error('Failed to load teacher manual', err);
    manualHtml = '';
  }
  res.render('teacher_manual', { user: req.session.user, manualHtml, manualUpdatedAt: updatedAt });
});

// Storage for test media uploads
const mediaStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, path.join(__dirname, '../uploads'));
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + '-' + file.originalname);
  }
});
const mediaUpload = multer({ storage: mediaStorage });

// Storage for syllabus uploads
const syllabusStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, path.join(__dirname, '../uploads/syllabi'));
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, 'syllabus-' + uniqueSuffix + path.extname(file.originalname));
  }
});
const syllabusUpload = multer({ 
  storage: syllabusStorage,
  limits: { fileSize: 10 * 1024 * 1024 }, // 10MB
  fileFilter: (req, file, cb) => {
    const allowedTypes = /pdf|doc|docx/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);
    if (extname && mimetype) {
      return cb(null, true);
    }
    cb(new Error('Only PDF, DOC, and DOCX files are allowed'));
  }
});

router.use((req, res, next) => {
  if (!req.session || !['teacher', 'admin'].includes(req.session.role)) return res.status(403).send('Forbidden');
  next();
});

// Parse JSON bodies for AJAX contact
router.use(express.json());

// Download CSV template for test uploads
router.get('/tests/template.csv', (_req, res) => {
  const header = [
    'Question','Answer','Explanation','Picture',
    'OptionA','OptionB','OptionC','OptionD','OptionE','OptionF','OptionG',
    'Test','Content Type','Title','Item Type','Path'
  ].join(',') + '\n';
  res.setHeader('Content-Type', 'text/csv; charset=utf-8');
  res.setHeader('Content-Disposition', 'attachment; filename="test_template.csv"');
  res.send(header);
});

router.get('/profile', async (req, res) => {
  const teacher = await userModel.findById(req.session.user.id);
  res.render('teacher_profile', { teacher, role: 'teacher', saved: req.query.saved });
});

router.post('/profile', mediaUpload.single('photo'), async (req, res) => {
  const id = req.session.user.id;
  if (req.file) {
    await userModel.updateProfile(id, {
      photo: {
        url: `/uploads/${req.file.filename}`,
        originalName: req.file.originalname
      }
    });
  }
    // Optional drawn signature
  if (req.body && req.body.signatureDataUrl && req.body.signatureDataUrl.startsWith('data:image/')) {
    await userModel.updateProfile(id, { signature: { url: req.body.signatureDataUrl } });
  }
  res.redirect('/teacher/profile?saved=1');
});

router.post('/profile/links', mediaUpload.single('image'), async (req, res) => {
  const { url } = req.body;
  if (url) {
    const link = { url };
    if (req.file) {
      link.image = { url: `/uploads/${req.file.filename}`, originalName: req.file.originalname };
    }
    try { await userModel.addLinks(req.session.user.id, [link]); }
    catch (e) { console.error('add link', e); }
  }
    // Optional drawn signature
  if (req.body && req.body.signatureDataUrl && req.body.signatureDataUrl.startsWith('data:image/')) {
    await userModel.updateProfile(id, { signature: { url: req.body.signatureDataUrl } });
  }
  res.redirect('/teacher/profile?saved=1');
});


// Dashboard with weekly schedule
router.get('/', async (req, res) => {
  const classes = await classModel.byTeacher(req.session.user.id);
  const users = await userModel.getAll();
  const students = users.filter(u => u.role === 'student');
  const pendingStudents = students.filter(u => u.status === 'pending' || !u.status);
  const approvedStudents = students.filter(u => u.status === 'approved');
  const classSizes = classes.map(c => ({ name: c.name, size: (c.studentIds || []).length }));
  const weekly = { Monday:[], Tuesday:[], Wednesday:[], Thursday:[], Friday:[], Saturday:[], Sunday:[] };
  classes.forEach(k => (k.schedule||[]).forEach(s => {
    if (weekly[s.day]) weekly[s.day].push({ className: k.name, start: s.start, end: s.end });
  }));
  res.render('teacher_dashboard', {
    teacher: req.session.user,
    classes,
    weekly,
    pendingCount: pendingStudents.length,
    approvedCount: approvedStudents.length,
    classSizes
  });
});

router.get('/announcements', async (req, res) => {
  const classes = await classModel.byTeacher(req.session.user.id);
  const announcements = await announcementModel.forTeacher(req.session.user.id);
  res.render('teacher_announcements', { teacher: req.session.user, classes, announcements });
});

router.get('/simplify', (req, res) => {
  res.render('topic_helper', { user: { role: 'teacher' }, result: null, topic: '' });
});

router.post('/simplify', async (req, res) => {
  const { topic } = req.body;
  const apiKey = process.env.OPENAI_API_KEY;
  let result = '';
  if (!apiKey) result = 'Missing OpenAI API key';
  if (!topic) result = 'Topic is required';
  if (!result) {
    try {
      const response = await fetch('https://api.openai.com/v1/chat/completions', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${apiKey}`
        },
        body: JSON.stringify({
          model: 'gpt-3.5-turbo',
          messages: [{ role: 'user', content: `Explain the following in simple terms for a 5th grade student: ${topic}` }]
        })
      });
      const data = await response.json();
      result = data.choices?.[0]?.message?.content || 'No summary generated';
    } catch (e) {
      console.error('OpenAI error', e);
      result = 'Error generating summary';
    }
  }
  res.render('topic_helper', { user: { role: 'teacher' }, result, topic });
});


router.get('/mailbox', async (req, res) => {
  const messages = await messageModel.getMailbox(req.session.user.id);
  const allUsers = await userModel.getAll();
  const userMap = new Map(allUsers.map(u => [u.id, u]));
  const formatted = messages.map(m => ({
    ...m,
    fromName: userMap.get(m.senderId)?.name || `User ${m.senderId}`,
    toName: userMap.get(m.recipientId)?.name || `User ${m.recipientId}`
  }));
  const students = allUsers.filter(u => u.role === 'student');
  res.render('mailbox', { messages: formatted, users: students, user: req.session.user });
});

router.post('/mailbox', async (req, res) => {
  const recipientId = Number(req.body.to);
  const { subject, body } = req.body;
  if (recipientId && body) {
    await messageModel.sendMessage(req.session.user.id, recipientId, subject || '', body);
  }
  res.redirect('/teacher/mailbox');
});

// Contact a single student (AJAX)
router.post('/students/:id/contact', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    if (!student || !student.email) return res.status(404).json({ error: 'Student not found' });
    const { subject, message } = req.body || {};
    if (!subject || !message) return res.status(400).json({ error: 'Missing subject or message' });
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      subject,
      html: message,
      text: String(message).replace(/<[^>]*>/g, '')
    });
    const now = new Date();
    try { await userModel.setLastContacted(id, now); } catch(_) {}
    return res.json({ ok: true, lastContacted: now.toISOString() });
  } catch (e) {
    console.error('Teacher student contact error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

function dataUrlToBuffer(url) {
  const match = /^data:.+;base64,(.+)$/.exec(url || '');
  return match ? Buffer.from(match[1], 'base64') : null;
}

function buildCompletionPdf({ studentName, courseName, dateStr, instructorSig, directorSig }) {
  return new Promise(resolve => {
    const doc = new PDFDocument({ size: 'LETTER', layout: 'landscape', margin: 50 });
    const buffers = [];
    doc.on('data', buffers.push.bind(buffers));
    doc.on('end', () => resolve(Buffer.concat(buffers)));

    const { width, height } = doc.page;

    // simple border background so text overlays an image-like backdrop
    doc.rect(0, 0, width, height).fill('#fff');
    doc.lineWidth(4).strokeColor('#0d274d').rect(20, 20, width - 40, height - 40).stroke();

    doc.fontSize(32).fillColor('#0d274d').text('CERTIFICATE OF COMPLETION', 0, 80, { align: 'center' });
    doc.moveDown(0.5);
    doc.fontSize(14).fillColor('black').text('This is to certify that', { align: 'center' });

    // name on a line
    const nameY = doc.y + 40;
    doc.moveTo(100, nameY).lineTo(width - 100, nameY).stroke('#0d274d');
    doc.fontSize(24).fillColor('black').text(studentName, 0, nameY - 22, { align: 'center' });

    doc.moveDown();
    doc.fontSize(14).text('has successfully completed', { align: 'center' });
    doc.fontSize(18).fillColor('#0d274d').text(courseName, { align: 'center' });
    doc.moveDown(0.5);
    doc.fontSize(14).fillColor('black').text(`on ${dateStr}`, { align: 'center' });

    const y = height - 150;
    if (instructorSig) {
      const buf = dataUrlToBuffer(instructorSig);
      if (buf) {
        try { doc.image(buf, 100, y, { width: 150 }); } catch (_) {}
      }
      doc.text('Instructor', 100, y + 70, { width: 150, align: 'center' });
    }
    if (directorSig) {
      const buf = dataUrlToBuffer(directorSig);
      if (buf) {
        try { doc.image(buf, width - 250, y, { width: 150 }); } catch (_) {}
      }
      doc.text('Director', width - 250, y + 70, { width: 150, align: 'center' });
    }

    doc.end();
  });
}

// Send course completion certificate to student
router.post('/classes/:classId/students/:studentId/certificate', async (req, res) => {
  try {
    const classId = Number(req.params.classId);
    const studentId = Number(req.params.studentId);
    const student = await userModel.findById(studentId);
    if (!student || !student.email) return res.status(404).json({ error: 'Student not found' });

    const klass = await classModel.findClassById(classId);
    const teacher = klass ? await userModel.findById(klass.teacherId) : null;
    const dateStr = new Date().toLocaleDateString();
    const directorSig = req.app.locals.branding?.directorSignature;
    const instructorSig = teacher?.profile?.signature?.url || directorSig; // Use director signature as fallback
    const pdfBuffer = await buildCompletionPdf({
      studentName: student.name || `${student.firstName || ''} ${student.lastName || ''}`.trim(),
      courseName: klass ? klass.name : 'Course',
      dateStr,
      instructorSig,
      directorSig
    });

    const admins = await userModel.getByRole('admin');
    const cc = admins.map(a => a.email).filter(Boolean);
    const link = `${req.protocol}://${req.get('host')}/student/certificates/${classId}/completion`;

    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      cc,
      subject: 'Certificate of Completion',
      html: `Congratulations ${student.name || ''}! Your certificate of completion is attached or available <a href="${link}">here</a>.`,
      text: `Congratulations ${student.name || ''}! Your certificate of completion is attached or available at ${link}`,
      attachments: [{ filename: 'certificate.pdf', content: pdfBuffer }]
    });

    try {
      await userModel.addCertificate(studentId, {
        title: 'Course Completion',
        classId,
        testId: 0,
        url: `/student/certificates/${classId}/completion`
      });
    } catch (_) {}

    return res.json({ ok: true });
  } catch (e) {
    console.error('Send completion certificate error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

router.post('/announcements', async (req, res) => {
  const { message, classId } = req.body;
  if (message && classId) {
    await announcementModel.create({ authorId: req.session.user.id, audience: 'class', classId: Number(classId), message });
  }
  res.redirect('/teacher/announcements');
});

router.post('/announcements/:id/delete', async (req, res) => {
  const id = Number(req.params.id);
  if (!Number.isNaN(id)) {
    await announcementModel.remove(id);
  }
  res.redirect('/teacher/announcements');
});

// Class view + roster + tests
router.get('/classes/:id', async (req, res) => {
  const klass = await classModel.findClassById(Number(req.params.id));
  if (!klass) return res.status(404).send('Not found');
  const users = await userModel.getAll();
  const students = users
    .filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id))
    .sort((a, b) => {
      // Sort alphabetically by name
      const nameA = (a.name || a.profile?.firstName || '').toLowerCase();
      const nameB = (b.name || b.profile?.firstName || '').toLowerCase();
      return nameA.localeCompare(nameB);
    });
 const today = new Date().toISOString().slice(0,10);
  const attendanceToday = (klass.attendance || []).find(a => a.date === today) || { present: [] };
  const discussions = await discussionModel.getByClass(klass.id);
  // Load available simulations from sims folder
  const fs = require('fs');
  const path = require('path');
  const simsDir = path.join(__dirname, '..', 'sims');
  let availableSims = [];
  try {
    const entries = fs.readdirSync(simsDir, { withFileTypes: true });
    availableSims = entries
      .filter(d => d.isDirectory())
      .map(d => ({ name: d.name, indexPath: path.join(simsDir, d.name, 'index.html') }))
      .filter(e => fs.existsSync(e.indexPath))
      .map(e => ({ title: e.name, url: `/sims/${e.name}/index.html` }));
  } catch (_) {}
  const lib = resourceLibrary.loadLibrary();
  res.render('teacher_view_class', { klass, students, today, attendanceToday, discussions, availableSims, availableLectures: lib.lectures, availableAssignments: lib.assignments, availableTests: lib.tests });
});

router.post('/classes/:id/rename', async (req, res) => {
  const id = Number(req.params.id);
  const { name } = req.body;
  if (name && name.trim()) {
    await classModel.renameClass(id, name.trim());
  }
  res.redirect(`/teacher/classes/${id}`);
});

// add discussion message
router.post('/classes/:id/discussion', async (req, res) => {
  const classId = Number(req.params.id);
  const { message } = req.body;
  if (message && message.trim()) {
    await discussionModel.addMessage(classId, req.session.user.id, message.trim());
    const klass = await classModel.findClassById(classId);
    const teacher = await userModel.findById(klass.teacherId);
    if (teacher && teacher.email) {
      try {
        const { subject, html, text } = emailTemplates.render('discussionNotification', {
          klassName: klass.name,
          message: message.trim()
        });
        await transporter.sendMail({
          to: teacher.email,
          from: process.env.SMTP_USER,
          subject,
          html,
          text
        });
      } catch (e) {
        console.error('Email send failed', e);
      }
    }
  }
  res.redirect(`/teacher/classes/${classId}#discussion`);
});
router.post('/classes/:id/checklist', async (req, res) => {
  const classId = Number(req.params.id);
  const items = Array.isArray(req.body.item) ? req.body.item : (req.body.item ? [req.body.item] : []);
  const done = Array.isArray(req.body.done) ? req.body.done.map(Number) : (req.body.done ? [Number(req.body.done)] : []);
  const checklist = items.map((text, idx) => ({ text, done: done.includes(idx) }));
  await classModel.updateChecklist(classId, checklist);
  res.redirect(`/teacher/classes/${classId}`);
});

// Upload syllabus for a class
router.post('/classes/:id/syllabus', syllabusUpload.single('syllabus'), async (req, res) => {
  try {
    const classId = Number(req.params.id);
    
    if (!req.file) {
      return res.status(400).send('No file uploaded');
    }
    
    // Ensure uploads/syllabi directory exists
    const fs = require('fs');
    const syllabusDir = path.join(__dirname, '../uploads/syllabi');
    if (!fs.existsSync(syllabusDir)) {
      fs.mkdirSync(syllabusDir, { recursive: true });
    }
    
    const syllabusPath = '/uploads/syllabi/' + req.file.filename;
    const syllabusFileName = req.file.originalname;
    
    await classModel.updateSyllabus(classId, syllabusPath, syllabusFileName);
    
    res.redirect(`/teacher/classes/${classId}`);
  } catch (error) {
    console.error('Syllabus upload error:', error);
    res.status(500).send('Failed to upload syllabus');
  }
});

// Delete syllabus from a class
router.delete('/classes/:id/syllabus', async (req, res) => {
  try {
    const classId = Number(req.params.id);
    
    // Get the class to find the file path
    const klass = await classModel.findClassById(classId);
    
    if (klass.syllabusPath) {
      // Delete the physical file
      const fs = require('fs');
      const filePath = path.join(__dirname, '..', klass.syllabusPath);
      
      if (fs.existsSync(filePath)) {
        fs.unlinkSync(filePath);
      }
    }
    
    // Remove from database
    await classModel.removeSyllabus(classId);
    
    res.sendStatus(200);
  } catch (error) {
    console.error('Syllabus delete error:', error);
    res.status(500).send('Failed to delete syllabus');
  }
});


// new test creation form
router.get('/classes/:id/tests/new', async (req, res) => {
  const klass = await classModel.findClassById(Number(req.params.id));
  if (!klass) return res.status(404).send('Not found');
  res.render('create_test', { klass });
});

// generate and save a test
router.post('/classes/:id/tests', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, lecture, questionCount, optionCount, timeLimit, official } = req.body;
  const questions = generateQuestions(
    lecture || '',
    Number(questionCount) || 0,
    Number(optionCount) || 4,
    title || ''
  );
  await testModel.replaceTestQuestions(title, questions);
  await classModel.addTest(classId, { 
    title, 
    timeLimit: Number(timeLimit) || 90,
    official: official === 'on' || official === true || official === 'true' // checkbox value
  });
  try { resourceLibrary.addTest({ title, timeLimit: Number(timeLimit) || 90 }); } catch (_) {}
  const lines = questions.map(q => {
    const optionCells = [];
    for (let i = 0; i < 7; i++) optionCells.push(q.options[i] || '');
    return [
      q.question,
      q.options[q.answer] || '',
      q.explanation || '',
      q.picture || '',
      ...optionCells,
      title,
      q.contentType,
      q.title,
      q.itemType,
      q.path
    ].join('\t');
  }).join('\n');
  const outDir = path.join(__dirname, '..', 'data', 'tests');
  if (!fs.existsSync(outDir)) fs.mkdirSync(outDir, { recursive: true });
  fs.writeFileSync(path.join(outDir, `${title.replace(/\s+/g, '_')}.tsv`), lines, 'utf8');
  res.redirect(`/teacher/classes/${classId}`);
});

router.post('/classes/:id/lectures', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, date, dripDay } = req.body;
  const finalUrl = url && url.trim();
  if (title && finalUrl) {
    await classModel.addLecture(classId, {
      title: title.trim(),
      url: finalUrl,
      date,
      dripDay: dripDay ? Number(dripDay) : null
    });
    try { resourceLibrary.addLecture({ title: title.trim(), url: finalUrl }); } catch (_) {}
  }
  res.redirect(`/teacher/classes/${classId}#lectures`);
});

router.post('/classes/:id/lectures/:lectureId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const lectureId = Number(req.params.lectureId);
  await classModel.removeLecture(classId, lectureId);
  res.redirect(`/teacher/classes/${classId}#lectures`);
});

router.post('/classes/:id/lectures/:lectureId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const lectureId = Number(req.params.lectureId);
  const { title, date, dripDay } = req.body;
  await classModel.updateLecture(classId, lectureId, { 
    title: title && title.trim(), 
    date,
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/teacher/classes/${classId}#lectures`);
});

router.post('/classes/:id/assignments', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, date, dripDay } = req.body;
  if (title && url) {
    await classModel.addAssignment(classId, { 
      title: title.trim(), 
      url: url.trim(), 
      date,
      dripDay: dripDay ? Number(dripDay) : null
    });
    try { resourceLibrary.addAssignment({ title: title.trim(), url: url.trim() }); } catch (_) {}
  }
  res.redirect(`/teacher/classes/${classId}#assignments`);
});

router.post('/classes/:id/assignments/:assignmentId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const assignmentId = Number(req.params.assignmentId);
  await classModel.removeAssignment(classId, assignmentId);
  res.redirect(`/teacher/classes/${classId}#assignments`);
});

router.post('/classes/:id/assignments/:assignmentId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const assignmentId = Number(req.params.assignmentId);
  const { title, date, dripDay } = req.body;
  await classModel.updateAssignment(classId, assignmentId, { 
    title: title && title.trim(), 
    date,
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/teacher/classes/${classId}#assignments`);
});


router.post('/classes/:id/simulations', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, date, dripDay } = req.body;
  if (title && url) {
    await classModel.addSimulation(classId, { 
      title: title.trim(), 
      url: url.trim(), 
      date,
      dripDay: dripDay ? Number(dripDay) : null
    });
  }
  res.redirect(`/teacher/classes/${classId}#simulations`);
});

router.post('/classes/:id/simulations/:simId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const simId = Number(req.params.simId);
  await classModel.removeSimulation(classId, simId);
  res.redirect(`/teacher/classes/${classId}#simulations`);
});

router.post('/classes/:id/simulations/:simId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const simId = Number(req.params.simId);
  const { title, date, dripDay } = req.body;
  await classModel.updateSimulation(classId, simId, { 
    title: title && title.trim(), 
    date,
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/teacher/classes/${classId}#simulations`);
});

router.post('/classes/:id/tests/library', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, timeLimit, dueDate, official } = req.body;
  if (title) {
    await classModel.addTest(classId, { 
      title: title.trim(), 
      timeLimit: Number(timeLimit) || 90, 
      dueDate,
      official: official === 'on' || official === true || official === 'true' // Default to true for library tests
    });
  }
  res.redirect(`/teacher/classes/${classId}#tests`);
});

  router.post('/classes/:id/tests/upload', upload.single('csv'), async (req, res) => {
    const classId = Number(req.params.id);
    const title = (req.body.title || 'Uploaded Test').trim();
    const timeLimit = Number(req.body.timeLimit) || 90;
    const official = req.body.official === 'on' || req.body.official === true || req.body.official === 'true';

    const csv = req.file && req.file.buffer.toString('utf-8');
    if (csv) {
      const lines = csv.split(/\r?\n/).filter(l => l.trim());
      const [, ...rows] = lines; // skip header
      const questions = rows.map(line => {
        const cols = line.split(',');
        return {
          question: cols[0],
          answer: cols[1],
          explanation: cols[2],
          picture: cols[3],
          options: cols.slice(4, 11).filter(Boolean),
          test: cols[11],
          contentType: cols[12],
          title: cols[13],
          itemType: cols[14],
          path: cols[15]
        };
      });
      await testModel.replaceTestQuestions(title, questions);
      await classModel.addTest(classId, { title, timeLimit, official });
      try { resourceLibrary.addTest({ title, timeLimit }); } catch (_) {}
    }
    res.redirect(`/teacher/classes/${classId}#tests`);
  });

// Upload media assets for tests
router.post('/classes/:id/tests/media', mediaUpload.single('media'), (req, res) => {
  const classId = Number(req.params.id);
  if (!req.file) return res.status(400).send('No file uploaded');
  res.redirect(`/teacher/classes/${classId}#tests`);
});

router.post('/classes/:id/tests/:testId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  await classModel.removeTest(classId, testId);
  res.redirect(`/teacher/classes/${classId}#tests`);
});

router.post('/classes/:id/tests/:testId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  const { title, dueDate, official, dripDay } = req.body;
  await classModel.updateTest(classId, testId, { 
    title: title && title.trim(), 
    dueDate,
    official: official === 'on' || official === true || official === 'true',
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/teacher/classes/${classId}#tests`);
});

// Links management routes
router.post('/classes/:id/links', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, description } = req.body;
  if (title && url) {
    await classModel.addLink(classId, {
      title: title.trim(),
      url: url.trim(),
      description: description ? description.trim() : ''
    });
  }
  res.redirect(`/teacher/classes/${classId}#links`);
});

router.post('/classes/:id/links/:linkId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const linkId = Number(req.params.linkId);
  await classModel.removeLink(classId, linkId);
  res.redirect(`/teacher/classes/${classId}#links`);
});

router.post('/classes/:id/links/:linkId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const linkId = Number(req.params.linkId);
  const { title, url, description } = req.body;
  await classModel.updateLink(classId, linkId, { 
    title: title && title.trim(), 
    url: url && url.trim(),
    description: description ? description.trim() : ''
  });
  res.redirect(`/teacher/classes/${classId}#links`);
});


router.post('/classes/:id/tests/generate', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, prompt, timeLimit } = req.body;
  const questionCount = Math.max(1, Math.min(200, Number(req.body.questionCount) || 10));
  const apiKey = process.env.OPENAI_API_KEY;
  if (!apiKey) return res.status(500).send('Missing OpenAI API key');

  try {
    // 1) Call OpenAI – ask for JSON **only**
    const response = await fetch('https://api.openai.com/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        Authorization: `Bearer ${apiKey}`
      },
      body: JSON.stringify({
        model: 'gpt-3.5-turbo', // swap to a current model in your env if needed
        messages: [
          {
            role: 'system',
            content:
              'You are a data generator. Respond with ONLY a JSON array, no prose, no code fences.'
          },
          {
            role: 'user',
            content:
              `Create a JSON array of exactly ${questionCount} questions for: ${prompt}.
Each array item must be an object with the EXACT keys:
"Question","Answer","Explanation","Picture",
"OptionA","OptionB","OptionC","OptionD","OptionE","OptionF","OptionG",
"Test","Content Type","Title","Item Type","Path".
Do not include any extra keys. Do not include backticks or code fences.`
          }
        ],
        temperature: 0.2
      })
    });

    const data = await response.json();
    if (!response.ok) {
      console.error('OpenAI error payload:', data);
      return res.status(502).send('AI generation failed');
    }

    let raw = data?.choices?.[0]?.message?.content || '';
    // 2) Strip code fences if the model still included them
    raw = raw.trim();
    raw = raw.replace(/^```(?:json)?\s*/i, '').replace(/```$/i, '').trim();

    // 3) Parse and validate
    let items;
    try {
      items = JSON.parse(raw);
    } catch (e) {
      console.error('JSON parse error. Raw content:', raw);
      items = [];
    }
    if (!Array.isArray(items)) {
      console.error('Model did not return an array. Content:', raw);
      items = [];
    }
    // Enforce max question count
    items = items.slice(0, questionCount);

    const testTitle = title || 'AI Generated Test';

    // 4) Ensure answer is present among options, then insert
    const sql = `
      INSERT INTO mdtsapps_myclass.LMSTest5
      (Question, Answer, Explanation, Picture,
       OptionA, OptionB, OptionC, OptionD, OptionE, OptionF, OptionG,
       Test, \`Content Type\`, Title, \`Item Type\`, Path)
      VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
    `;

    // If you're using mysql2/promise:
    // const [stmt] = await db.prepare(sql);  // if you use prepare()
    // We'll just execute per-row here:
    let inserted = 0;
    for (const obj of items) {
      // 5) Defensive access (normalize keys in case of casing drift)
      const get = (k) => obj?.[k] ?? '';
      // Ensure options contain the answer
      const ans = String(get('Answer') || '').trim();
      const keys = ['OptionA','OptionB','OptionC','OptionD','OptionE','OptionF','OptionG'];
      let opts = keys.map(k => String(obj?.[k] || '').trim());
      const hasAns = ans && opts.some(o => o.toLowerCase() === ans.toLowerCase());
      if (ans && !hasAns) {
        const emptyIdx = opts.findIndex(o => !o);
        if (emptyIdx >= 0) opts[emptyIdx] = ans; else opts[opts.length - 1] = ans;
        // Write back so get() below reflects corrected options
        keys.forEach((k,i) => { obj[k] = opts[i]; });
      }

      const values = [
        get('Question'),
        get('Answer'),
        get('Explanation'),
        get('Picture'),
        get('OptionA'),
        get('OptionB'),
        get('OptionC'),
        get('OptionD'),
        get('OptionE'),
        get('OptionF'),
        get('OptionG'),
        get('Test') || testTitle,
        get('Content Type'),
        get('Title'),
        get('Item Type'),
        get('Path')
      ];

      try {
        // mysql2/promise:
        // await db.execute(sql, values);
        await db.query(sql, values); // works if your helper wires to execute internally
        inserted++;
      } catch (dbErr) {
        console.error('DB insert error for values:', values, dbErr);
      }
    }

    // 6) Record the test on the class regardless of per-row failures
    try {
      await classModel.addTest(classId, {
        title: testTitle,
        timeLimit: Number(timeLimit) || 90
      });
      try { resourceLibrary.addTest({ title: testTitle, timeLimit: Number(timeLimit) || 90 }); } catch (_) {}
    } catch (e) {
      console.error('classModel.addTest failed:', e);
    }

    console.log(`Inserted ${inserted} question rows into LMSTest5.`);
    res.redirect(`/teacher/classes/${classId}#tests`);
  } catch (e) {
    console.error('Route error', e);
    res.status(500).send('Unexpected error');
  }
});

// Add external test link
router.post('/classes/:id/tests/external', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, dueDate, dripDay, official } = req.body;
  
  if (title && url) {
    await classModel.addTest(classId, {
      title: title.trim(),
      externalUrl: url.trim(),
      dueDate: dueDate || null,
      dripDay: dripDay ? Number(dripDay) : null,
      official: official === 'on' || official === true || official === 'true',
      isExternal: true
    });
  }
  res.redirect(`/teacher/classes/${classId}#tests`);
});

// Grade submission (upsert)
router.post('/classes/:id/grades', async (req, res) => {
  const classId = Number(req.params.id);
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Not found');
  // Test grades (if any were submitted)
  for (const key of Object.keys(req.body)) {
    const m = key.match(/^grade_(\d+)_(\d+)$/);
    if (!m) continue;
    const studentId = Number(m[1]);
    const testId = Number(m[2]);
    const val = req.body[key];
    if (val === '') continue;
    let score = Number(val);
    if (Number.isNaN(score)) continue;
    if (score < 0) score = 0;
    if (score > 100) score = 100;
    await classModel.upsertGrade(classId, testId, studentId, score);
  }

  // Assignment grades
  for (const studentId of klass.studentIds || []) {
    for (const a of klass.assignments || []) {
      const key = `asg_${studentId}_${a.id}`;
      const val = req.body[key];
      if (val === undefined || val === '') continue;
      let score = Number(val);
      if (Number.isNaN(score)) continue;
      if (score < 0) score = 0;
      if (score > 100) score = 100;
      await classModel.upsertAssignmentGrade(classId, a.id, Number(studentId), score);
    }
    for (const l of klass.simulations || []) {
      const key = `lab_${studentId}_${l.id}`;
      const passed = !!req.body[key];
      await classModel.upsertLabStatus(classId, l.id, Number(studentId), passed);
    }
  }

  res.redirect(`/teacher/classes/${classId}#grades`);
});

// View student profile
router.get('/students/:id', async (req, res) => {
  const student = await userModel.findById(Number(req.params.id));
  if (!student) return res.status(404).send('Not found');
  let lead = null, leadContacts = [];
  try {
    if (student.email) {
      const leadModel = require('../models/leadModel');
      lead = await leadModel.getByEmail(student.email);
      if (lead) leadContacts = await leadModel.getContacts(lead.id);
    }
  } catch (e) { console.error('Lead fetch failed', e); }
  
  // Get all classes where this student is enrolled
  const allClasses = await classModel.getAllClasses();
  const studentClasses = allClasses.filter(c => (c.studentIds || []).includes(student.id));
  
  res.render('student_profile', { 
    student, 
    role: 'teacher', 
    reset: req.query.reset, 
    signatureDocsConfig, 
    lead, 
    leadContacts,
    studentClasses
  });
});

router.post('/students/:id/reset-password', async (req, res) => {
  const id = Number(req.params.id);
  const user = await userModel.findById(id);
  if (!user) return res.status(404).send('Not found');
  const newPassword = crypto.randomBytes(4).toString('hex');
  await userModel.updatePassword(user.username, newPassword);
  if (user.email) {
    try {
      const { subject, html, text } = emailTemplates.render('passwordReset', {
        name: user.name || 'User',
        newPassword
      });
      await transporter.sendMail({
        from: 'no-reply@mdts-apps.com',
        to: user.email,
        subject,
        html,
        text
      });
    } catch (e) {
      console.error('Error sending reset email', e);
    }
  }
  res.redirect(`/teacher/students/${id}?reset=1`);});

router.get('/', async (req, res) => {
  const classes = await classModel.byTeacher(req.session.user.id);
  const weekly = { Monday:[], Tuesday:[], Wednesday:[], Thursday:[], Friday:[], Saturday:[], Sunday:[] };

  const events = [];
  const now = new Date();
  const threeWeeksFromNow = new Date();
  threeWeeksFromNow.setDate(now.getDate() + 21);

classes.forEach(klass => {
  (klass.schedule || []).forEach(s => {
    const dayIndex = ['Sunday','Monday','Tuesday','Wednesday','Thursday','Friday','Saturday'].indexOf(s.day);
    let date = new Date(now);
    while (date <= threeWeeksFromNow) {
      if (date.getDay() === dayIndex) {
        const [sh, sm] = String(s.start || '0:0').split(':').map(Number);
        events.push({
          title: klass.name + ' (Class)',
          start: new Date(date.getFullYear(), date.getMonth(), date.getDate(), sh, sm || 0),
          url: `/teacher/classes/${klass.id}` // NEW: link to teacher's class view
        });
      }
      date.setDate(date.getDate() + 1);
    }
  });

  (klass.tests || []).forEach(test => {
    if (test.dueDate) {
      const due = new Date(test.dueDate);
      if (due >= now && due <= threeWeeksFromNow) {
        events.push({
          title: klass.name + ' - ' + test.title + ' (Due)',
          start: due,
          url: `/teacher/classes/${klass.id}` // Could link to grade view or test edit
        });
      }
    }
  });
});


  res.render('teacher_dashboard', { teacher: req.session.user, classes, weekly, events });
});
// Attendance submission
router.post('/classes/:id/attendance', async (req, res) => {
  const classId = Number(req.params.id);
  const date = req.body.date || new Date().toISOString().slice(0,10);
  const presentIds = Object.keys(req.body)
    .filter(k => k.startsWith('present_'))
    .map(k => Number(k.replace('present_', '')));
  
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Class not found');
  
  // Record attendance in class model
  await classModel.recordAttendance(classId, date, presentIds);
  
  // Update each student's profile with attendance record
  for (const studentId of presentIds) {
    const student = await userModel.findById(studentId);
    if (student) {
      const profile = student.profile || {};
      profile.attendance = profile.attendance || [];
      
      // Check if attendance for this date and class already exists
      const existing = profile.attendance.find(a => a.date === date && a.classId === classId);
      if (!existing) {
        profile.attendance.push({
          date: date,
          classId: classId,
          className: klass.name,
          studentName: student.name
        });
        await userModel.updateProfile(studentId, { attendance: profile.attendance });
      }
    }
  }
  
  res.redirect(`/teacher/classes/${classId}`);
});

// Attendance report for a class
router.get('/classes/:id/attendance', async (req, res) => {
  const klass = await classModel.findClassById(Number(req.params.id));
  if (!klass) return res.status(404).send('Not found');
  const users = await userModel.getAll();
  const students = users
    .filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id))
    .sort((a, b) => {
      // Sort alphabetically by name
      const nameA = (a.name || a.profile?.firstName || '').toLowerCase();
      const nameB = (b.name || b.profile?.firstName || '').toLowerCase();
      return nameA.localeCompare(nameB);
    });
  res.render('attendance_report', { klass, students, user: req.session.user });
});

// Individual student grade report
router.get('/classes/:classId/students/:studentId/grade-report', async (req, res) => {
  const classId = Number(req.params.classId);
  const studentId = Number(req.params.studentId);
  
  const klass = await classModel.findClassById(classId);
  if (!klass || klass.teacherId !== req.session.user.id) {
    return res.status(404).send('Class not found or access denied');
  }
  
  const users = await userModel.getAll();
  const student = users.find(u => u.id === studentId);
  
  if (!student || student.role !== 'student' || !(klass.studentIds || []).includes(studentId)) {
    return res.status(404).send('Student not found or not enrolled in this class');
  }
  
  res.render('grade_report', { klass, student });
});

// Print gradebook for entire class
router.get('/classes/:id/gradebook/print', async (req, res) => {
  const classId = Number(req.params.id);
  const klass = await classModel.findClassById(classId);
  
  if (!klass || klass.teacherId !== req.session.user.id) {
    return res.status(404).send('Class not found or access denied');
  }
  
  const users = await userModel.getAll();
  const students = users.filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id));
  
  res.render('gradebook_print', { klass, students });
});

router.get('/reports/class/:id', async (req, res) => {
  const id = Number(req.params.id);
  const klass = await classModel.findClassById(id);
  if (!klass || klass.teacherId !== req.session.user.id) return res.status(404).send('Not found');
  const users = await userModel.getAll();
  const students = users.filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id));
  res.render('class_report', { klass, students, scope: 'teacher' });
});

// Reports
router.get('/reports', async (req, res) => {
  const classes = await classModel.byTeacher(req.session.user.id);
  const users = await userModel.getAll();
  const studentMap = Object.fromEntries(users.filter(u => u.role === 'student').map(u => [u.id, u]));
  const report = classes.map(k => {
    const studentsArr = (k.studentIds || [])
      .map(id => studentMap[id])
      .filter(Boolean)
      .map(u => ({ id: u.id, name: u.name }));
    return {
      classId: k.id,
      className: k.name,
      studentsList: studentsArr,
      studentCount: studentsArr.length,
      tests: (k.tests || []).length,
      grades: (k.grades || []).length
    };
  });
  res.render('reports', { report, scope: 'teacher' });
});

// Preview test routes
router.get('/classes/:id/tests/:testId/preview', async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  console.log('Teacher preview test', { classId, testId, userId: req.session.user.id });
  const klass = await classModel.findClassById(classId);
  if (!klass) {
    console.log('Class not found', classId);
    return res.status(404).send('Not found');
  }
  const test = (klass.tests || []).find(t => t.id === testId);
  if (!test) {
    console.log('Test not found', { classId, testId });
    return res.status(404).send('Test not found');
  }
  test.questions = await testModel.getQuestionsByTest(test.title);
  res.render('take_test', {
    klass,
    test,
    attempts: 0,
    user: req.session.user,
    action: `/teacher/classes/${classId}/tests/${testId}/preview`
  });
});

router.post('/classes/:id/tests/:testId/preview', async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  console.log('Teacher submit preview', { classId, testId, userId: req.session.user.id });
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Not found');
  const test = (klass.tests || []).find(t => t.id === testId);
  if (!test) return res.status(404).send('Test not found');
  test.questions = await testModel.getQuestionsByTest(test.title);
  let score = 0;
  test.questions.forEach((q, i) => {
    const chosen = Number(req.body[`q_${i}`]);
    const correct = Number(q.answer);
    if (!Number.isNaN(chosen) && chosen === correct) score++;
  });
  const pct = Math.round((score / test.questions.length) * 100);
  console.log('Preview score', { classId, testId, userId: req.session.user.id, score: pct });
  res.render('test_result', { klass, test, score: pct, student: req.session.user, user: req.session.user });
});


// Upload media for a question (teacher preview)
router.post('/classes/:id/tests/:testId/questions/:qIndex/media', mediaUpload.single('media'), async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  const qIndex = Number(req.params.qIndex);
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Not found');
  const test = (klass.tests || []).find(t => t.id === testId);
  if (!test) return res.status(404).send('Test not found');
  if (!req.file) return res.redirect(`/teacher/classes/${classId}/tests/${testId}/preview`);
  const url = `/uploads/${req.file.filename}`;
  const isVideo = (req.file.mimetype || '').startsWith('video/');
  try {
    // Resolve question text by index to avoid attribute escaping issues
    let questionText = '';
    try {
      const questions = await testModel.getQuestionsByTest(test.title);
      if (Array.isArray(questions) && questions[qIndex]) questionText = questions[qIndex].question || '';
    } catch (_) {}
    await testModel.updateQuestionMediaByText(test.title, questionText, url, isVideo);
  } catch (e) {
    console.error('update media failed', e);
  }
  res.redirect(`/teacher/classes/${classId}/tests/${testId}/preview`);
});
module.exports = router;
